/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog

import android.app.Application
import android.content.Context
import android.content.SharedPreferences
import androidx.preference.PreferenceManager
import com.zell_mbc.medilog.preferences.SettingsActivity
import com.zell_mbc.medilog.support.BiometricHelper
import org.acra.config.dialog
import org.acra.config.mailSender
import org.acra.data.StringFormat
import org.acra.dialog.BuildConfig
import org.acra.ktx.initAcra
import androidx.core.content.edit
import com.zell_mbc.medilog.support.AppStart
import com.zell_mbc.medilog.support.checkAppStart

class MediLog : Application() {
    var authenticated = false
    private var hasBiometric = true
    private lateinit var preferences: SharedPreferences

    lateinit var appStart: AppStart

    companion object {
        @Volatile
        private var instance: MediLog? = null

        fun getInstance(): MediLog = instance ?: throw IllegalStateException("App not initialized yet!")

        val context: Context?
            get() = getInstance().applicationContext
    }

    // Called when the application is starting, before any other application objects have been created.
    // Overriding this method is totally optional!
    override fun onCreate() {
        super.onCreate()

        // ********************* Debug code **************************
        // Force crash for debug purposes
        //throw RuntimeException("This is a forced crash for debug purposes");
        // ******** Reset the app for testing purposes! Use with caution!!!!!!!!
        // Delete preferences file to get around Googles auto backup
        //val prefsName = "${context?.packageName}_preferences"
        //deleteSharedPreferences(prefsName)
        // Remove db
        //preferences.edit { remove(LAST_APP_VERSION) }
        // ********************* /Debug code **************************

        preferences = PreferenceManager.getDefaultSharedPreferences(this)
        appStart = checkAppStart(this) // Needs to be here to ensure it runs only once
        //appStart = AppStart.FIRST_TIME_APP

        val crashedLastTime = preferences.getInt("crash_count", 0)
        if (crashedLastTime > 0) {
            // Handle crash
            // Emergency backup, etc.
        }
        preferences.edit(commit = true) { putInt("crash_count", crashedLastTime + 1) }

        // If Authenticated is false lets check if we want to and if this device can authenticate
      if (!authenticated) {
          if (!preferences.getBoolean(SettingsActivity.KEY_PREF_BIOMETRIC, false)) {
              authenticated = true         // If user set biometric login off, we are always authenticated
          } else { // Biometric setting is on -> Let's check if authentication is possible
              val biometricHelper = BiometricHelper(this)
              // Check if biometric device  exists, if not remove biometric setting in settings activity and set authenticated to always true
              if (biometricHelper.hasHardware(false)) {
                  authenticated = false    // Make sure we launch the Biometric logon process in onStart
              } else {
                  authenticated = true
                  hasBiometric = false
              }
          }
      }
    }

    override fun attachBaseContext(base: Context) {
        super.attachBaseContext(base)

        preferences = PreferenceManager.getDefaultSharedPreferences(this)
        if (!preferences.getBoolean(SettingsActivity.KEY_PREF_ENABLECRASHLOGS, true)) return

        initAcra {
            //core configuration:
            buildConfigClass = BuildConfig::class.java
            reportFormat = StringFormat.KEY_VALUE_LIST

            mailSender {
                //required
                mailTo = "medilog@zell-mbc.com"
                reportAsFile = true
                reportFileName = "crashLog.txt"
                subject = getString(R.string.crashTitle)
                body = getString(R.string.crashMessageEmail)
            }
            dialog {
                text = getString(R.string.crashMessageDialog)
                title = getString(R.string.crashTitle)
                positiveButtonText = getString(R.string.ok)
                negativeButtonText = getString(R.string.cancel)
                commentPrompt = getString(R.string.crashContext)
                resIcon = android.R.drawable.ic_dialog_alert
                emailPrompt = ""
            }
        }
    }
}

