package com.zell_mbc.medilog.profiles

import android.widget.Toast
import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.font.FontStyle
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import androidx.lifecycle.viewmodel.compose.viewModel
import com.zell_mbc.medilog.ActiveProfile
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.data.Profiles
import com.zell_mbc.medilog.weight.WeightViewModel

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun ProfilesScreen(
    onProfileActivated: (Int) -> Unit,
    onBack: () -> Unit
) {
    val context = LocalContext.current

    val profilesViewModel: ProfilesViewModel = viewModel()
    val dataViewModel: WeightViewModel = viewModel()
    val profiles by profilesViewModel.getAllRecords().collectAsState(initial = emptyList())
    val selectedIds = rememberSaveable { mutableStateListOf<Int>() }

    val showAdd = remember { mutableStateOf(false) }
    val showEdit = remember { mutableStateOf(false) }
    val showDelete = remember { mutableStateOf(false) }

    Box( modifier = Modifier
        .fillMaxSize()
        .background(MaterialTheme.colorScheme.background)) {
    Scaffold(
        topBar = {
            TopAppBar(title = { Text(text = context.getString(R.string.editProfilesDialog)) },
                navigationIcon = { IconButton(onClick = onBack) {
                    Icon(Icons.AutoMirrored.Filled.ArrowBack, contentDescription = "Back")
                    }
                },
                //modifier = Modifier.fillMaxSize(), // scaffold fills full screen
            )
        },
        content = { innerPadding ->
            Box(modifier = Modifier.fillMaxSize().padding(innerPadding)) {
                Column(modifier = Modifier.fillMaxSize().padding(horizontal = 16.dp, vertical = 12.dp), // your fixed content padding
                    verticalArrangement = Arrangement.spacedBy(16.dp)) {
                    Text(
                        text = "${context.getString(R.string.activeProfileName)}: ${ActiveProfile.name}",
                        style = MaterialTheme.typography.titleMedium, color = MaterialTheme.colorScheme.primary
                    )
                    Surface(modifier = Modifier.fillMaxWidth().weight(1f, fill = true)) {
                        LazyColumn(modifier = Modifier.fillMaxWidth()) {
                            //item { HeaderRow() }
                            items(profiles, key = { it._id }) { p ->
                                ProfileRow(item = p, checked = selectedIds.contains(p._id),
                                    onCheckChanged = { checked ->
                                        if (checked) selectedIds.add(p._id) else selectedIds.remove(p._id)
                                    },
                                    onActivate = { onProfileActivated(p._id) }
                                )
                            }
                        }
                    }

                    Row(horizontalArrangement = Arrangement.End, modifier = Modifier.fillMaxWidth().padding(8.dp)) {
                        OutlinedButton(onClick = { showAdd.value = true }, modifier = Modifier.padding(end = 8.dp)) {
                            Text(context.getString(R.string.add))
                        }
                        OutlinedButton(onClick = {
                            when (selectedIds.size) {
                                0 -> Toast.makeText(context, context.getString(R.string.noProfileSelected), Toast.LENGTH_LONG).show()
                                1 -> showEdit.value = true
                                else -> Toast.makeText(context, context.getString(R.string.moreThanOneProfileSelected), Toast.LENGTH_LONG).show()
                            }
                        }, enabled = selectedIds.isNotEmpty(), modifier = Modifier.padding(end = 8.dp)) {
                            Text(context.getString(R.string.edit))
                        }

                        OutlinedButton(onClick = {
                            val deletableIds = selectedIds.filter { it != ActiveProfile.id }
                            if (deletableIds.size < selectedIds.size) {
                                Toast.makeText(context, context.getString(R.string.unableToDeleteActiveProfile), Toast.LENGTH_LONG).show()
                            }
                            else showDelete.value = true }, enabled = selectedIds.isNotEmpty()) {
                            Text(context.getString(R.string.delete))
                        }
                    }
                }
            }

            // Todo: Show tab selection dialog -> Think about how a new profile here can set it's values?
            if (showAdd.value) AddProfileDialog(onDismiss = { showAdd.value = false }) { name, desc ->
                if (name.isBlank()) Toast.makeText(context, context.getString(R.string.noProfileName), Toast.LENGTH_LONG).show()
                else {
                    profilesViewModel.createNewProfile(name = name, description = desc)
                    showAdd.value = false
                }
            }

            if (showEdit.value) { // && editTargetId.value != null) {
                val id = selectedIds.first() //editTargetId.value!!
                val selectedProfile = profilesViewModel.get(id)
                if (selectedProfile != null) EditProfileDialog(item = selectedProfile, onDismiss = { showEdit.value = false }) { newName, newDesc ->
                    if (newName.isBlank()) Toast.makeText(context, context.getString(R.string.noProfileName), Toast.LENGTH_LONG).show()
                    else {
                        selectedProfile.name = newName
                        selectedProfile.description = newDesc // Update name in db
                        profilesViewModel.upsert(selectedProfile)
                        if (selectedProfile._id == ActiveProfile.id)
                            ActiveProfile.name = selectedProfile.name // Update name in dialog

                        selectedIds.clear()
                        showEdit.value = false
                    }
                } else {
                    Toast.makeText(context, "Entry not found", Toast.LENGTH_LONG).show()
                    selectedIds.clear()
                    showEdit.value = false
                }
            }

            if (showDelete.value) {
                val ids = selectedIds.toList()
                val recordsForSelected = dataViewModel.getCountForProfileIds(ids)

                DeleteProfilesDialog(countProfiles = ids.size, associatedRecords = recordsForSelected, onCancel = {
                    showDelete.value = false
                }, onConfirm = {
                    val deletableIds = selectedIds.filter { it != ActiveProfile.id }
                    if (deletableIds.size < selectedIds.size) {
                        Toast.makeText(context, context.getString(R.string.unableToDeleteActiveProfile), Toast.LENGTH_LONG).show()
                    }
                    else {
                        for (id in ids)
                            profilesViewModel.delete(id) // Delete actual profile
                        selectedIds.clear()
                        showDelete.value = false
                        }
                })
            }
        }
    )
} }

@Composable
private fun HeaderRow() {
    Row(modifier = Modifier.fillMaxWidth().background(MaterialTheme.colorScheme.surface).padding(8.dp), verticalAlignment = Alignment.CenterVertically) {
        Spacer(modifier = Modifier.width(8.dp))
        Text(text = stringResource(R.string.profileName), fontWeight = FontWeight.Bold, modifier = Modifier.width(160.dp))
        Divider(modifier = Modifier.width(1.dp).height(24.dp))
        //Spacer(modifier = Modifier.width(8.dp))
        //Text(text = stringResource(R.string.description), fontWeight = FontWeight.Bold)
    }
}

@Composable
private fun ProfileRow(item: Profiles, checked: Boolean, onCheckChanged: (Boolean) -> Unit, onActivate: () -> Unit) {
    Row(modifier = Modifier.fillMaxWidth().clickable(onClick = onActivate).padding(8.dp), verticalAlignment = Alignment.CenterVertically) {
        Checkbox(checked = checked, onCheckedChange = onCheckChanged)
        Spacer(modifier = Modifier.width(8.dp))
        Column(modifier = Modifier.weight(1f)) {
            Text(text = item.name, maxLines = 1)
            Text(text = item.description, overflow = TextOverflow.Ellipsis, style = TextStyle(fontStyle = FontStyle.Italic))
        }
        //Spacer(modifier = Modifier.width(8.dp))
        //Button(onClick = onActivate) { Text(stringResource(R.string.activate)) }
    }
}

@Composable
private fun AddProfileDialog(onDismiss: () -> Unit, onSave: (String, String) -> Unit) {
    val name = rememberSaveable { mutableStateOf("") }
    val desc = rememberSaveable { mutableStateOf("") }

    AlertDialog(
        onDismissRequest = onDismiss,
        title = { Text(stringResource(R.string.addProfileDialog)) },
        text = {
            Column {
                OutlinedTextField(value = name.value, onValueChange = { name.value = it }, label = { Text(stringResource(R.string.profileName)) })
                OutlinedTextField(value = desc.value, onValueChange = { desc.value = it }, label = { Text(stringResource(R.string.description)) })
            }
        },
        confirmButton = { TextButton(onClick = { onSave(name.value.trim(), desc.value.trim()) }) { Text(stringResource(R.string.save)) } },
        dismissButton = { TextButton(onClick = onDismiss) { Text(stringResource(R.string.cancel)) } }
    )
}

@Composable
private fun EditProfileDialog(item: Profiles, onDismiss: () -> Unit, onSave: (String, String) -> Unit) {
    val name = rememberSaveable { mutableStateOf(item.name) }
    val desc = rememberSaveable { mutableStateOf(item.description ?: "") }

    AlertDialog(
        onDismissRequest = onDismiss,
        title = { Text(stringResource(R.string.editProfileDialog)) },
        text = {
            Column {
                OutlinedTextField(value = name.value, onValueChange = { name.value = it }, label = { Text(stringResource(R.string.profileName)) })
                OutlinedTextField(value = desc.value, onValueChange = { desc.value = it }, label = { Text(stringResource(R.string.description)) })
            }
        },
        confirmButton = { TextButton(onClick = { onSave(name.value.trim(), desc.value.trim()) }) { Text(stringResource(R.string.save)) } },
        dismissButton = { TextButton(onClick = onDismiss) { Text(stringResource(R.string.cancel)) } }
    )
}

@Composable
private fun DeleteProfilesDialog(countProfiles: Int, associatedRecords: Int, onCancel: () -> Unit, onConfirm: () -> Unit) {
    AlertDialog(
        onDismissRequest = onCancel,
        title = { Text(stringResource(R.string.warning)) },
        text = {
            Column {
                Text(stringResource(R.string.deleteProfiles, countProfiles, associatedRecords))
                Spacer(modifier = Modifier.height(8.dp))
                Text(stringResource(R.string.doYouReallyWantToContinue))
            }
        },
        confirmButton = { TextButton(onClick = onConfirm) { Text(stringResource(R.string.yes)) } },
        dismissButton = { TextButton(onClick = onCancel) { Text(stringResource(R.string.cancel)) } }
    )
}
