/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

/*
 */

package com.zell_mbc.medilog.oximetry

import android.content.Context
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.IntrinsicSize
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.text.KeyboardActions
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.outlined.Abc
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.material3.TextField
import androidx.compose.material3.VerticalDivider
import androidx.compose.runtime.Composable
import androidx.compose.runtime.SideEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusDirection
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.text.PlatformTextStyle
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.font.FontStyle
import androidx.compose.ui.text.input.KeyboardCapitalization
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.TextUnit
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.zell_mbc.medilog.ActiveProfile
import com.zell_mbc.medilog.Filter.NO_TAGS
import com.zell_mbc.medilog.MainActivity
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.R.string
import com.zell_mbc.medilog.Thresholds
import com.zell_mbc.medilog.base.BaseTab
import com.zell_mbc.medilog.data.Data
import com.zell_mbc.medilog.preferences.SettingsActivity.Companion.KEY_PREF_QUICKENTRY
import com.zell_mbc.medilog.scaffold.Screens
import com.zell_mbc.medilog.support.SnackbarDelegate
import com.zell_mbc.medilog.support.checkMinMax
import com.zell_mbc.medilog.support.matchesTemplate
import com.zell_mbc.medilog.support.shouldMoveFocus
import com.zell_mbc.medilog.support.valueToString
import com.zell_mbc.medilog.tags.TagsViewModel
import java.util.Date

class OximetryTab(override val label: String?, override val viewModel: OximetryViewModel, override val tagsViewModel: TagsViewModel, context: Context, snackbarDelegate: SnackbarDelegate): BaseTab(viewModel, tagsViewModel, context, snackbarDelegate) {
    override var editActivityClass: Class<*> = OximetryEditActivity::class.java
    override var infoActivityClass: Class<*> = OximetryInfoActivity::class.java
    override var chartActivityClass: Class<*> = OximetryChartActivity::class.java

    override val route = Screens.Oximetry.route
    override val fabEnabled = true
    override val iconActive = R.drawable.ic_spo2_filled
    override val iconInactive = R.drawable.ic_spo2_outlined

    @Composable
    override fun ShowContent(padding: PaddingValues) {
        super.ShowContent(padding)
        // Global variables

        var showComment by remember { mutableStateOf(viewModel.comment.value.isNotEmpty()) }
        val value1Width = measureTextWidth(viewModel.value1.template, MainActivity.fontSize)

        LazyColumn(state = listState, horizontalAlignment = Alignment.Start, modifier = Modifier.fillMaxWidth().padding(start = 8.dp, end = 8.dp)) {
            item {
                if (viewModel.quickEntry) {
                    // Oximetry
                    val (textField1, textField2, textField3) = remember { FocusRequester.createRefs() }
                    Row(Modifier.fillMaxWidth()) {
                        // Oximetry
                        TextField(
                            value = viewModel.value1.value,
                            colors = textFieldColors,
                            onValueChange = {
                                // 1. Check template
                                if (matchesTemplate(it, viewModel.value1.template))
                                // 2. Max/min reached?
                                    if (checkMinMax(field = it, min = 0, max = Thresholds.MAX_OXYGEN)) {
                                        viewModel.value1.value = it // Single source of truth!
                                        // 3. Jump if needed
                                        if (shouldMoveFocus(it, viewModel.value1.template, Thresholds.MAX_OXYGEN))
                                            focusManager?.moveFocus(FocusDirection.Next)
                                    }
                            },
                            singleLine = true,
                            keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Decimal),
                            textStyle = TextStyle(),
                            label = { Text(text = "${viewModel.value1.label}*", maxLines = 1, overflow = TextOverflow.Ellipsis) },
                            placeholder = { Text("${viewModel.value1.template} ${viewModel.value1.unit}") },
                            modifier = Modifier.weight(1f).focusRequester(textField1).padding(end = 10.dp),
                            keyboardActions = KeyboardActions(onDone = {
                                keyboardController?.hide()
                                addItem()
                            })
                        )
                        // Pulse
                        TextField(
                            value = viewModel.value2.value,
                            colors = textFieldColors,
                            onValueChange = {
                                showComment = true
                                // 1. Check template
                                if (matchesTemplate(it, viewModel.value2.template))
                                // 2. Max/min reached?
                                    if (checkMinMax(field = it, min = 0, max = Thresholds.MAX_PULSE)) {
                                        viewModel.value2.value = it // Single source of truth!
                                        // 3. Jump if needed
                                        if (shouldMoveFocus(it, viewModel.value2.template, Thresholds.MAX_PULSE))
                                            focusManager?.moveFocus(FocusDirection.Next)
                                    }
                            },
                            singleLine = true,
                            keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Decimal),
                            textStyle = TextStyle(),
                            label = { Text(text = viewModel.value2.label, maxLines = 1, overflow = TextOverflow.Ellipsis) },
                            placeholder = { Text("${viewModel.value2.template} ${viewModel.value2.unit}") },
                            modifier = Modifier.weight(1f).focusRequester(textField2).padding(end = cellPadding.dp),
                            keyboardActions = KeyboardActions( onDone = { addItem() })
                        )
                    } // Row

                    if (showComment)
                        TextField(
                            value = viewModel.comment.value,
                            colors = textFieldColors,
                            onValueChange = { viewModel.comment.value = it },
                            singleLine = true,
                            textStyle = TextStyle(),
                            label = { Text(text = viewModel.comment.label) },
                            keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Text, capitalization = KeyboardCapitalization.Sentences),
                            modifier = Modifier.fillMaxWidth().focusRequester(textField3),
                            trailingIcon = {
                                IconButton(onClick = {
                                    openTextTemplatesDialog.value = true
                                })
                                { Icon(Icons.Outlined.Abc, contentDescription = null) }
                            },
                            keyboardActions = KeyboardActions(onDone = {
                                keyboardController?.hide()
                                addItem()
                            })
                        )
                    SideEffect {
                        // Set cursor to first field after addItem completed
                        if (activateFirstField) {
                            textField1.requestFocus()
                            activateFirstField = false }
                    }

                    // Dialog section
                    if (openTextTemplatesDialog.value) {
                        textTemplateDialog.ShowDialog(viewModel.dataType, setShowDialog = { openTextTemplatesDialog.value = it }) {
                            viewModel.comment.value += it
                            textField3.requestFocus()
                        }
                    }
                } // End quick entry
            } // item
            items(dataList.value, key = { it._id }) { item ->
                // Lines starting from the top
                HorizontalDivider(color = MaterialTheme.colorScheme.secondaryContainer, thickness = 1.dp)
                Row(modifier = Modifier.fillMaxWidth().height(IntrinsicSize.Min).clickable { selection.value = item }, verticalAlignment = Alignment.CenterVertically) {
                    ShowRow(item, value1Width)
                }
                if (selection.value != null) {
                    ItemClicked(selection.value!!._id)
                    selection.value = null
                }
            }
        }
    }

    @Composable
    fun ShowRow(item: Data, value1Width: Dp) {
        // Timestamp
        ResizableDateColumn(formatDateString(item.timestamp))

        var fontStyle = FontStyle.Normal
        var textColor = MaterialTheme.colorScheme.primary

        // A diary item blended in
        if (item.type != viewModel.dataType) {
            fontStyle = FontStyle.Italic
            val s = item.comment + if (item.attachment.isNotEmpty()) " $PAPERCLIP" else ""
            Text(s, Modifier.padding(start = cellPadding.dp), color = MaterialTheme.colorScheme.primary, fontSize = MainActivity.fontSize.sp, fontStyle = fontStyle)
            return
        }

        val value1String = valueToString( value = item.value1, viewModel.value1.template)
        val stringToValue = try { item.value1.replace(",",".").toFloat() } catch (_: NumberFormatException) { 0f }
        val highlightRow = (viewModel.highlightValues && ((stringToValue < viewModel.lowerThreshold) || (stringToValue > viewModel.upperThreshold)))
        if (highlightRow) textColor = MaterialTheme.colorScheme.error // highlightRow is set in convertToString

        // Oximetry
        Text(value1String, Modifier.padding(start = cellPadding.dp, end = cellPadding.dp).width(value1Width),textAlign = TextAlign.Center,
            color = textColor, fontSize = MainActivity.fontSize.sp, fontStyle = fontStyle)
        VerticalDivider(color = MaterialTheme.colorScheme.secondaryContainer, modifier = Modifier.width(1.dp).fillMaxHeight()) // Vertical separator

        // Pulse

        textColor = MaterialTheme.colorScheme.primary
        Text(item.value2, Modifier.padding(start = cellPadding.dp, end = cellPadding.dp).width(value1Width),
            textAlign = TextAlign.Center, color = textColor, fontSize = MainActivity.fontSize.sp, fontStyle = fontStyle)
        VerticalDivider(color = MaterialTheme.colorScheme.secondaryContainer, modifier = Modifier.width(1.dp).fillMaxHeight()) // Vertical separator

        // Comment
        Spacer(modifier = Modifier.width(cellPadding.dp))
        val s = item.comment + if (item.attachment.isNotEmpty()) " $PAPERCLIP" else ""
        Text(s, color = textColor, fontSize = MainActivity.fontSize.sp, fontStyle = fontStyle,
            style = TextStyle(lineHeight = TextUnit.Unspecified, platformStyle = PlatformTextStyle(includeFontPadding = false)) // style is for removing the padding between multiline text
        )
    }


    override fun addItem() {
        if (!preferences.getBoolean(KEY_PREF_QUICKENTRY, true)) {
            startEditing(-1) // Indicate that this is about a new item
            return
        }

        // Close keyboard after entry is done
        hideKeyboard()

        var validInput = matchesTemplate(viewModel.value1)
        if (!validInput) {
            val message =  context.getString(string.invalidFormat) + "\n" + context.getString(string.configuredFormat, viewModel.value1.template)
            snackbarDelegate.showSnackbar(message)
            return
        }

        // Valid pulse?
        if (viewModel.value2.value.isNotBlank()) {
            validInput = matchesTemplate(viewModel.value2)
            if (!validInput) {
                val message =  context.getString(string.invalidFormat) + "\n" + context.getString(string.configuredFormat, viewModel.value1.template)
                snackbarDelegate.showSnackbar(message)
                return
            }
        }

        val item = Data(_id = 0, timestamp = Date().time, comment = viewModel.comment.value, type = viewModel.dataType,
            value1 = viewModel.value1.value.replace(",","."),
            value2 = viewModel.value2.value.replace(",","."),
            value3 = "", value4 = "", attachment = "", tags = NO_TAGS, category_id = -1, profile_id = ActiveProfile.id)
        viewModel.upsert(item)

        if ((viewModel.filterEnd > 0L) && (viewModel.filterEnd < item.timestamp))
            snackbarDelegate.showSnackbar(context.getString(string.filteredOut))

        cleanUpAfterAddItem()
    }
}