/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

/*
 */

package com.zell_mbc.medilog.glucose

import android.app.Application
import android.content.SharedPreferences
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableFloatStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.core.content.edit
import com.zell_mbc.medilog.Tabs.GLUCOSE
import com.zell_mbc.medilog.R.string
import com.zell_mbc.medilog.data.DataViewModel
import com.zell_mbc.medilog.glucoseUnitUndefined
import com.zell_mbc.medilog.preferences.SettingsActivity
import com.zell_mbc.medilog.support.checkThresholds

class GlucoseViewModel(application: Application): DataViewModel(application, GLUCOSE) {
    override val filterStartPref = "GLUCOSEFILTERSTART"
    override val filterEndPref = "GLUCOSEFILTEREND"
    override val timeFilterModePref =  "GLUCOSE_FILTER_MODE"
    override val rollingFilterValuePref = "GLUCOSE_ROLLING_FILTER_VALUE"
    override val rollingFilterTimeframePref = "GLUCOSE_ROLLING_FILTER_TIMEFRAME"
    override val tagFilterPref = "GLUCOSE_TAG_FILTER"
    override val dateTimeColumnWidthPref = "GLUCOSE_DATE_COLUMN_WIDTH"

    //All preferences that affect UI or app behavior displayed in UI should be here
    // General settings
    override var blendInItems    by mutableStateOf(preferences.getBoolean(SettingsActivity.KEY_PREF_GLUCOSE_BLENDINITEMS, app.getString(string.BLENDINITEMS_DEFAULT).toBoolean()))
    override var highlightValues by mutableStateOf(preferences.getBoolean(SettingsActivity.KEY_PREF_GLUCOSE_HIGHLIGHT_VALUES, app.getString(string.GLUCOSE_HIGHLIGHT_VALUES_DEFAULT).toBoolean()))
    override var showTime        by mutableStateOf(preferences.getBoolean(SettingsActivity.KEY_PREF_GLUCOSE_SHOWTIME, app.getString(string.SHOWTIME_DEFAULT).toBoolean()))
    override var upperThreshold  by mutableFloatStateOf(0f)
    override var lowerThreshold  by mutableFloatStateOf(0f)

    //Tab specific
    var mg_dl = false
    var logKetone by mutableStateOf(preferences.getBoolean(SettingsActivity.KEY_PREF_LOG_KETONE, false))

    init {
        setupEntryFields()
        loadThresholds()

        // Register preference change listener
        // Handle updates
        prefListener = SharedPreferences.OnSharedPreferenceChangeListener { _, key ->
            when (key) {
                SettingsActivity.KEY_PREF_QUICKENTRY -> quickEntry = preferences.getBoolean(SettingsActivity.KEY_PREF_QUICKENTRY, true)
                SettingsActivity.KEY_PREF_GLUCOSE_CUSTOM_NAME,
                SettingsActivity.KEY_PREF_GLUCOSE_CUSTOM_UNIT,
                SettingsActivity.KEY_PREF_GLUCOSE_CUSTOM_TEMPLATE,
                SettingsActivity.KEY_PREF_GLUCOSE_UNIT, -> setupEntryFields() // refresh the fields on any related change
                SettingsActivity.KEY_PREF_GLUCOSE_THRESHOLDS -> loadThresholds()
                SettingsActivity.KEY_PREF_LOG_KETONE -> logKetone = preferences.getBoolean(SettingsActivity.KEY_PREF_LOG_KETONE, app.getString(string.GLUCOSE_HIGHLIGHT_VALUES_DEFAULT).toBoolean())
                SettingsActivity.KEY_PREF_GLUCOSE_HIGHLIGHT_VALUES -> highlightValues = preferences.getBoolean(SettingsActivity.KEY_PREF_GLUCOSE_HIGHLIGHT_VALUES, false)
                SettingsActivity.KEY_PREF_GLUCOSE_SHOWTIME -> {
                    showTime = preferences.getBoolean(SettingsActivity.KEY_PREF_GLUCOSE_SHOWTIME, app.getString(string.SHOWTIME_DEFAULT).toBoolean())
                    preferences.edit { remove(dateTimeColumnWidthPref) } // Make sure we measure again
                }
                SettingsActivity.KEY_PREF_GLUCOSE_BLENDINITEMS -> blendInItems = preferences.getBoolean(SettingsActivity.KEY_PREF_GLUCOSE_BLENDINITEMS, app.getString(string.BLENDINITEMS_DEFAULT).toBoolean())
            }
        }
        preferences.registerOnSharedPreferenceChangeListener(prefListener)
    }

    // Specific parameters, applicable to just this field
    override fun setupEntryFields() {
        value1.label = preferences.getString(SettingsActivity.KEY_PREF_GLUCOSE_CUSTOM_NAME, "").toString()
        value1.unit = preferences.getString(SettingsActivity.KEY_PREF_GLUCOSE_CUSTOM_UNIT, "").toString()
        value1.template = preferences.getString(SettingsActivity.KEY_PREF_GLUCOSE_CUSTOM_TEMPLATE, "").toString()

        // After custom was set once the above won't provide the default any longer
        if (value1.label.isEmpty()) value1.label = app.getString(string.glucose)
        if (value1.unit.isEmpty()) value1.unit = preferences.getString(SettingsActivity.KEY_PREF_GLUCOSE_UNIT, glucoseUnitUndefined).toString()
        if (value1.template.isEmpty()) value1.template = if (value1.unit == app.getString(string.GLUCOSE_UNIT_MG_DL)) app.getString(string.glucoseHintMg) else app.getString(string.glucoseHintMm)

        // mg_dl = integers, mmol_L = decimals
        mg_dl = (value1.unit == app.getString(string.GLUCOSE_UNIT_MG_DL))

        value2.label = app.getString(string.ketone)
        value2.unit = preferences.getString(SettingsActivity.KEY_PREF_GLUCOSE_UNIT, app.getString(string.GLUCOSE_UNIT_MMOL_L)).toString()
        value2.template = app.getString(string.ketoneEntryHint).replace('.', decimalSeparator)

        value3.label = app.getString(string.glucoseKetoneIndex)
        value3.unit = preferences.getString(SettingsActivity.KEY_PREF_GLUCOSE_UNIT, app.getString(string.GLUCOSE_UNIT_MMOL_L)).toString()
        value3.template = value2.template
    }

    fun loadThresholds() {
                /*
        Condition	                mmol/L	            mg/dL
        Fasting (no food for 8h)	3.9 – 5.5 mmol/L	70 – 99 mg/dL
        2h after a meal	under       7.8 mmol/L	        under 140 mg/dL
        Pre-diabetes (fasting)	    5.6 – 6.9 mmol/L	100 – 125 mg/dL
        Diabetes (diagnostic)	    ≥ 7.0 mmol/L	    ≥ 126 mg/dL
        */

        logKetone = preferences.getBoolean(SettingsActivity.KEY_PREF_LOG_KETONE, app.getString(string.LOG_KETONE_DEFAULT).toBoolean())

        val defaultThresholds = if(mg_dl) app.getString(string.GLUCOSE_THRESHOLDS_DEFAULT_MG_DL) else app.getString(string.GLUCOSE_THRESHOLDS_DEFAULT_MMOL_L)
        val s = "" + preferences.getString(SettingsActivity.KEY_PREF_GLUCOSE_THRESHOLDS, defaultThresholds)

        /* v3.2.0 Temporary fix for a bug where the backup stored the unit instead of the threshold
        // ToDo: Remove after a few releases down the line
        if (s == "mg/dL" || s == "mmol/L") {
            s = ""
            preferences.edit { putString(SettingsActivity.KEY_PREF_GLUCOSE_THRESHOLDS, s) }
        }
        */
        val th = checkThresholds(app, s, defaultThresholds, value1.label)
        lowerThreshold = try { th[0].toFloat() } catch  (_: NumberFormatException) { 0f }
        upperThreshold = try { th[1].toFloat() } catch  (_: NumberFormatException) { 0f }
    }
}
