/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.dialogs

import android.content.Context
import androidx.core.content.FileProvider
import android.content.Intent
import android.net.Uri
import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.heightIn
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Share
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.unit.dp
import java.io.File
import java.text.SimpleDateFormat
import java.util.Date
import java.util.Locale
import android.text.format.Formatter
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.material3.Button
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.material3.OutlinedButton
import androidx.compose.runtime.LaunchedEffect
import com.zell_mbc.medilog.data.restoreDatabaseFromUri

@Composable
fun DatabaseShareDialog(
    dbDir: File,
    onDismiss: () -> Unit
) {
    val context = LocalContext.current
    var message by remember { mutableStateOf<String?>(null) }

    // Remember a launcher that uses Activity Result API to launch the chooser
    val shareLauncher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.StartActivityForResult(),
        onResult = {
            // Optionally handle share result (usually nothing needed)
            message = null
        }
    )

    val files = remember(dbDir) {
        dbDir.listFiles()
            ?.filter { it.isFile && it.name.endsWith(".backup") }
            ?.sortedByDescending { it.lastModified() }
            ?: emptyList()
    }
    val filesAll = remember(dbDir) {
        dbDir.listFiles()?.sortedByDescending { it.lastModified() } ?: emptyList()
    }
    AlertDialog(
        onDismissRequest = onDismiss,
        title = { Text("Database Files") },
        text = {
            Column(verticalArrangement = Arrangement.spacedBy(8.dp)) {
                if (files.isEmpty())
                    Text("No backup files found in ${dbDir.path}")
                else
                    LazyColumn(
                        modifier = Modifier.heightIn(max = 300.dp)
                    ) {
                        items(files) { file ->
                            Row(
                                modifier = Modifier
                                    .fillMaxWidth()
                                    .clickable {
                                        message = "Sharing ${file.name}..."
                                        val backupFile = File(context.filesDir, file.name)
                                        val uri = FileProvider.getUriForFile(context, context.packageName + ".provider", backupFile
                                        )
                                        val shareIntent = Intent(Intent.ACTION_SEND).apply {
                                            type = "application/octet-stream"
                                            putExtra(Intent.EXTRA_STREAM, uri)
                                            // Optional: add subject/text if emailing
                                            putExtra(Intent.EXTRA_SUBJECT, "Database Backup: ${file.name}")
                                            putExtra(Intent.EXTRA_TEXT, "Sharing backup file.")
                                            addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION)
                                        }
                                        val chooser = Intent.createChooser(shareIntent, "Share database file")
                                        shareLauncher.launch(chooser)
                                    }
                                    .padding(8.dp),
                                verticalAlignment = Alignment.CenterVertically
                            ) {
                                Column(modifier = Modifier.weight(1f)) {
                                    Text(file.name, style = MaterialTheme.typography.bodyLarge)
                                    Text(
                                        "${formatSize(context, file.length())} — ${formatDate(file.lastModified())}",
                                        style = MaterialTheme.typography.bodySmall
                                    )
                                }
                                Icon(Icons.Default.Share, contentDescription = "Share")
                            }
                        }
                    }
                if (message != null) Text(message!!, color = MaterialTheme.colorScheme.primary)
            }
        },
        confirmButton = {
            Button(onClick = onDismiss) { Text("Close") }
        }
    )
}

@Composable
fun DatabaseImportDialog(
    databaseName: String,
    onDismissRequest: () -> Unit,
    onRefreshActivity: () -> Unit  // Callback to refresh the activity
) {
    val context = LocalContext.current

    var selectedUri by remember { mutableStateOf<Uri?>(null) }
    var isImporting by remember { mutableStateOf(false) }
    var resultMessage by remember { mutableStateOf<String?>(null) }
    var importSuccessful by remember { mutableStateOf(false) }

    val launcher = rememberLauncherForActivityResult(ActivityResultContracts.OpenDocument()) { uri: Uri? ->
        if (uri != null) {
            selectedUri = uri
            resultMessage = null
        }
    }

    // Effect to run restore when URI selected
    LaunchedEffect(selectedUri) {
        selectedUri?.let { uri ->
            isImporting = true
            restoreDatabaseFromUri(context, databaseName, uri) { success ->
                isImporting = false
                importSuccessful = success
                resultMessage = if (success) {
                    "Database import successful!"
                } else {
                    "Failed to import database."
                }
            }
            selectedUri = null  // Reset the URI after processing
        }
    }

    AlertDialog(
        onDismissRequest = {
            if (!isImporting) onDismissRequest()
        },
        title = { Text("Import Database Backup") },
        text = {
            Column {
                if (!importSuccessful) Text("Select a backup file to restore your database.") else Text("Press 'ok' to restart!")
                if (isImporting) {
                    Spacer(Modifier.height(8.dp))
                    Row(verticalAlignment = Alignment.CenterVertically) {
                        CircularProgressIndicator(modifier = Modifier.size(20.dp))
                        Spacer(Modifier.width(8.dp))
                        Text("Importing...")
                    }
                }
                resultMessage?.let { msg ->
                    Spacer(Modifier.height(8.dp))
                    Text(msg)
                }
            }
        },
        confirmButton = {
            if (importSuccessful) {
                Button(
                    onClick = {
                        onDismissRequest()
                        onRefreshActivity()  // Trigger activity refresh
                    }
                ) {
                    Text("OK")
                }
            } else {
                Button(
                    enabled = !isImporting,
                    onClick = {
                        launcher.launch(arrayOf("*/*"))
                    }
                ) {
                    Text("Select Backup File")
                }
            }
        },
        dismissButton = {
            if (!importSuccessful) {
                OutlinedButton(
                    enabled = !isImporting,
                    onClick = onDismissRequest
                ) {
                    Text("Cancel")
                }
            }
            // If import is successful, no dismiss button to force user to acknowledge
        }
    )
}


/** Format file size in a user-friendly way using Android's Formatter */
fun formatSize(context: Context, bytes: Long): String =
    Formatter.formatShortFileSize(context, bytes)

/** Format last modified date as a string */
fun formatDate(milliseconds: Long): String {
    val sdf = SimpleDateFormat("yyyy-MM-dd HH:mm", Locale.getDefault())
    return sdf.format(Date(milliseconds))
}
