/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.base

import android.annotation.SuppressLint
import android.app.Activity
import android.content.ContentResolver
import android.content.ContentValues
import android.content.Intent
import android.content.SharedPreferences
import android.database.Cursor
import android.graphics.BitmapFactory
import android.net.Uri
import android.os.Bundle
import android.provider.MediaStore
import android.util.Log
import android.view.MenuItem
import android.view.WindowManager
import android.view.inputmethod.InputMethodManager
import android.widget.Toast
import androidx.activity.addCallback
import androidx.activity.enableEdgeToEdge
import androidx.appcompat.app.AppCompatActivity
import androidx.compose.foundation.Image
import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.gestures.scrollBy
import androidx.compose.foundation.isSystemInDarkTheme
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.WindowInsets
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.ime
import androidx.compose.foundation.layout.imePadding
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.requiredSize
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.shape.CircleShape
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Attachment
import androidx.compose.material.icons.filled.Clear
import androidx.compose.material.icons.filled.DeleteForever
import androidx.compose.material.icons.filled.Save
import androidx.compose.material3.BottomAppBarDefaults
import androidx.compose.material3.Button
import androidx.compose.material3.DatePicker
import androidx.compose.material3.DatePickerDialog
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.FloatingActionButton
import androidx.compose.material3.FloatingActionButtonDefaults
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.Scaffold
import androidx.compose.material3.SnackbarHost
import androidx.compose.material3.SnackbarHostState
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.material3.rememberDatePickerState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableIntStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.scale
import androidx.compose.ui.focus.FocusManager
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.ColorFilter
import androidx.compose.ui.graphics.asImageBitmap
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.platform.LocalFocusManager
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.unit.dp
import androidx.core.content.FileProvider
import androidx.preference.PreferenceManager
import com.google.android.material.dialog.MaterialAlertDialogBuilder
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.R.string
import com.zell_mbc.medilog.data.Data
import com.zell_mbc.medilog.data.DataViewModel
import com.zell_mbc.medilog.dialogs.MediLogTimePickerDialog
import com.zell_mbc.medilog.texttemplates.TextTemplateDialog
import com.zell_mbc.medilog.preferences.SettingsActivity
import com.zell_mbc.medilog.support.MedilogTheme
import kotlinx.coroutines.launch
import java.io.File
import java.io.InputStream
import java.text.DateFormat
import java.text.SimpleDateFormat
import java.util.Date
import java.util.Locale
import com.zell_mbc.medilog.support.SnackbarDelegate
import com.zell_mbc.medilog.support.deleteApplicationCache
import com.zell_mbc.medilog.support.getAttachmentFolder
import com.zell_mbc.medilog.support.getCorrectedDateFormat
import com.zell_mbc.medilog.tags.TagsDialog
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import java.io.FileNotFoundException
import java.text.DecimalFormatSymbols
import java.util.Calendar
import java.util.TimeZone
import kotlin.text.contains
import androidx.core.net.toUri
import com.zell_mbc.medilog.ActiveProfile
import com.zell_mbc.medilog.Filter.NO_TAGS
import com.zell_mbc.medilog.support.valueToString

abstract class EditActivity: AppCompatActivity() {
    abstract val dataType: Int
    protected abstract val viewModel: DataViewModel

    lateinit var preferences: SharedPreferences
    lateinit var editItem: Data
    lateinit var dateFormat: DateFormat
    private lateinit var timeFormat: DateFormat
    lateinit var snackbarDelegate: SnackbarDelegate
    lateinit var activity: Activity

    var focusManager: FocusManager? = null

    //private val PERMISSION_CODE = 1000
    private val IMAGE_CAPTURE_CODE = 1001
    private val FILE_CAPTURE_CODE = 1002

    var fileName = ""
    var attachmentFolder = ""
    var attachementsEnabled = false
    var unsavedAttachment = false
    var unsavedAttachmentUri: Uri? = null // Unlink images, Local "files" may not be available as files, hence use this to save the Uri

    var highlightValues = false

    var isDirtyTimestamp = 0L
    var isDirtyValue1 = ""
    var isDirtyValue2 = ""
    var isDirtyValue3 = ""
    var isDirtyValue4 = ""
    var isDirtyComment = ""
    var isDirtyTagIds = ""
    var isDirtyAttachment = ""

    // Dialog control variables
    var showTextTemplatesDialog by mutableStateOf(false)
    var showDatePickerDialog by mutableStateOf(false)
    var showTimePickerDialog by mutableStateOf(false)
    var showTagsDialog by mutableStateOf(false)

    var attachmentDrawable by mutableIntStateOf(0)

    // Date/Time picker button labels
    var itemTimeString by mutableStateOf("")
    var itemDateString by mutableStateOf("")

    lateinit var textTemplateDialog: TextTemplateDialog
    lateinit var tagsDialog: TagsDialog

    // Retrieve decimal separator for current language
    val decimalSeparator = DecimalFormatSymbols.getInstance().decimalSeparator
    val modifyDecimalSeparator = (decimalSeparator.compareTo('.') != 0)

    override fun onSaveInstanceState(outState: Bundle) {
        super.onSaveInstanceState(outState)
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        //   Log.d(DEBUGTAG, "onCreate:" + savedInstanceState )
        activity = this
        viewModel.editItemIndex = savedInstanceState?.getInt("editItemIndex", 0) // Initial launch of activity
            ?: intent.getIntExtra("editItemIndex", -1)

        editItem = if (viewModel.editItemIndex < 1) { // We are dealing with a new item, create it first
            Data(0, timestamp = Date().time, type = dataType, profile_id = ActiveProfile.id, tags = NO_TAGS, category_id = -1, attachment = "")
        } else {
            val tmpItem = viewModel.getItem(viewModel.editItemIndex)
            if (tmpItem == null) {
                //(getSystemService(Context.INPUT_METHOD_SERVICE) as InputMethodManager).hideSoftInputFromWindow(requireView().windowToken, 0) // Close keyboard after entry is done
                Toast.makeText(this, getString(string.waitForUIUpdate), Toast.LENGTH_LONG).show()
                finish()
                return
            }
            tmpItem
        }

        // Prepare viewModel
        viewModel.timestamp = editItem.timestamp
        viewModel.value1.value = valueToString(value = editItem.value1, viewModel.value1.template)
        viewModel.value2.value = valueToString(value = editItem.value2, viewModel.value2.template)
        viewModel.value3.value = valueToString(value = editItem.value3, viewModel.value3.template)
        viewModel.value4.value = valueToString(value = editItem.value4, viewModel.value4.template)
        viewModel.comment.value = editItem.comment
        viewModel.tagIds = editItem.tags
        viewModel.attachment = editItem.attachment

        isDirtyTimestamp = viewModel.timestamp
        isDirtyValue1 = viewModel.value1.value
        isDirtyValue2 = viewModel.value2.value
        isDirtyValue3 = viewModel.value3.value
        isDirtyValue4 = viewModel.value4.value
        isDirtyComment = viewModel.comment.value
        isDirtyAttachment = viewModel.attachment
        isDirtyTagIds = viewModel.tagIds

        dateFormat = getCorrectedDateFormat(this) //Handle EN_DE bug?
        timeFormat = DateFormat.getTimeInstance(DateFormat.SHORT, Locale.getDefault())

        // Collect values from editItem and convert to strings for Compose fields
        itemTimeString = timeFormat.format(viewModel.timestamp)
        itemDateString = dateFormat.format(viewModel.timestamp)

        viewModel.attachment = editItem.attachment
        if (viewModel.attachment.isNotEmpty()) attachmentDrawable = R.drawable.ic_baseline_attach_file_24

        textTemplateDialog = TextTemplateDialog(this)
        tagsDialog = TagsDialog(this)

        preferences = PreferenceManager.getDefaultSharedPreferences(this)

        if (preferences.getBoolean(SettingsActivity.KEY_PREF_BLOCK_SCREENSHOTS, false))
            window.setFlags(WindowManager.LayoutParams.FLAG_SECURE, WindowManager.LayoutParams.FLAG_SECURE)

        highlightValues = preferences.getBoolean(SettingsActivity.KEY_PREF_WEIGHT_HIGHLIGHT_VALUES, false)
        attachmentFolder = getAttachmentFolder(this)

        enableEdgeToEdge() //This will include/color the top Android info bar
        onBackPressedDispatcher.addCallback(this) { handleBackPress() } // Handle the 'hardware' back button
    }

    // This handles the toolbar back arrow
    override fun onOptionsItemSelected(item: MenuItem): Boolean {
        return if (item.itemId == android.R.id.home) {
            if (isDirty()) {
                MaterialAlertDialogBuilder(this)
                    .setTitle(resources.getString(string.attention))
                    .setMessage(getString(string.unsavedData))
                    .setNegativeButton(resources.getString(string.no)) { _, _ ->  // Respond to negative button press
                    }
                    .setPositiveButton(resources.getString(string.yes)) { _, _ ->  // Respond to positive button press
                        finish()
                    }
                    .show()
            } else finish()
            true
        } else {
            super.onOptionsItemSelected(item)
        }
    }

    private fun handleBackPress(): Boolean {
        val dirty = isDirty()
        // perform action
        if (dirty) {
            MaterialAlertDialogBuilder(activity)
                .setTitle(resources.getString(string.attention))
                .setMessage(getString(string.unsavedData))
                .setNegativeButton(resources.getString(string.no)) { _, _ ->  // Respond to negative button press
                    //return@setNegativeButton
                }
                .setPositiveButton(resources.getString(string.yes)) { _, _ ->  // Respond to positive button press
                    finish()
                }
                .show()
        }
        return !dirty
    }

    override fun onDestroy() {
        super.onDestroy()
        deleteApplicationCache(this) // delete orphaned images
    }

    // Check if any field has changed
    fun isDirty(): Boolean {
        return ((editItem.timestamp != viewModel.timestamp) ||
                (isDirtyValue1 != viewModel.value1.value) ||
                (isDirtyValue2 != viewModel.value2.value) ||
                (isDirtyValue3 != viewModel.value3.value) ||
                (isDirtyValue4 != viewModel.value4.value) ||
                (isDirtyComment != viewModel.comment.value) ||
                (isDirtyTagIds != viewModel.tagIds) ||
                (isDirtyAttachment != viewModel.attachment))
    }


    //Kick off compose framework
    @Composable
    abstract fun ShowContent()

    @SuppressLint("UnusedMaterial3ScaffoldPaddingParameter")
    @Composable
    fun StartCompose() {
        val snackbarHostState = remember { SnackbarHostState() }
        snackbarDelegate = SnackbarDelegate(snackbarHostState, rememberCoroutineScope())

        // Scroll UI when keyboard opens
        // https://medium.com/@mark.frelih_9464/how-to-handle-automatic-content-resizing-when-keyboard-is-visible-in-jetpack-compose-1c76e0e17c57
        val scrollState = rememberScrollState()
        val coroutineScope = rememberCoroutineScope()
        val keyboardHeight = WindowInsets.ime.getBottom(LocalDensity.current)

        focusManager = LocalFocusManager.current

        LaunchedEffect(key1 = keyboardHeight) {
            coroutineScope.launch {
                scrollState.scrollBy(keyboardHeight.toFloat())
            }
        }

        KeyboardAware {

            MedilogTheme {
                Scaffold( //contentWindowInsets = WindowInsets(0.dp),
                    snackbarHost = { SnackbarHost(snackbarHostState) },
                    /* topBar = {
                    TopAppBar(modifier = Modifier.height(40.dp),
                        title = { "" },
                        actions = {
                            IconButton(onClick = { saveItem() }) { Icon(imageVector = Icons.Default.Save, contentDescription = "Search") }
                            IconButton(onClick = { getAttachment() }) { Icon(imageVector = Icons.Default.Attachment, contentDescription = "Attachment") }
                        }
                    )
                },*/
                    floatingActionButton = {
                        FloatingActionButton(
                            onClick = { deleteItem() },
                            containerColor = BottomAppBarDefaults.bottomAppBarFabColor,
                            elevation = FloatingActionButtonDefaults.elevation()
                        ) {
                            Icon(Icons.Filled.DeleteForever, "Delete item")
                        }
                    },
                    content = { paddingValues ->
                        Column(modifier = Modifier.padding(paddingValues).verticalScroll(scrollState)) { // This is necessary so the topbar is recognised
                            LaunchedEffect(key1 = keyboardHeight) {
                                coroutineScope.launch {
                                    scrollState.scrollBy(keyboardHeight.toFloat())
                                }
                            }
                            ShowContent()
                        }
                    }
                )
            }
        }
    }

    @Composable
    fun KeyboardAware(
        content: @Composable () -> Unit
    ) {
        Box(modifier = Modifier.imePadding()) {
            content()
        }
    }

    @Composable
    fun DateTimeBlock() {
        Surface(
            color = MaterialTheme.colorScheme.secondaryContainer,
            contentColor = MaterialTheme.colorScheme.onSecondaryContainer,
            shape = RoundedCornerShape(12.dp), tonalElevation = 8.dp, modifier = Modifier.fillMaxWidth()) {
            Column(modifier = Modifier.fillMaxWidth().padding(16.dp), verticalArrangement = Arrangement.spacedBy(12.dp)) {

                // ---------- ROW 1: DATE + SAVE ----------
                Row(modifier = Modifier.fillMaxWidth(), verticalAlignment = Alignment.CenterVertically) {

                    // Left side = Date label + button
                    Row(verticalAlignment = Alignment.CenterVertically) {
                        Text(text = getString(string.date), modifier = Modifier.width(80.dp))
                        OutlinedButton(onClick = { showDatePickerDialog = true }) {
                            Text(itemDateString)
                        }
                    }

                    Spacer(Modifier.weight(1f))

                    // Right side = SAVE
                    IconButton(onClick = { saveItem() }, modifier = Modifier.scale(1.3f)) {
                        Icon(Icons.Default.Save, contentDescription = "Save")
                    }
                }

                // ---------- ROW 2: TIME + ATTACHMENT ----------
                Row(modifier = Modifier.fillMaxWidth(), verticalAlignment = Alignment.CenterVertically) {

                    // Left side = Time label + button
                    Row(verticalAlignment = Alignment.CenterVertically) {
                        Text(text = getString(string.time), modifier = Modifier.width(80.dp))
                        OutlinedButton(onClick = { showTimePickerDialog = true }) {
                            Text(itemTimeString)
                        }
                    }

                    Spacer(Modifier.weight(1f))

                    // Right side = ATTACHMENT (only if enabled)
                    if (attachementsEnabled) {
                        IconButton(onClick = { getAttachment() }) {
                            Icon(Icons.Default.Attachment, contentDescription = "Attachment")
                        }
                    }
                }
            }
        }

        Spacer(Modifier.height(8.dp))
    }


    @Composable
    fun AttachmentBlock() {
        //if (!attchementsEnabled) return
        //var showAttachmentString = ""

        // If there's a temporary string show this one instead of what's stored
        //showAttachmentString = if (temporaryAttachmentUri != null) temporaryAttachmentUri.toString() else attachmentString

        if (viewModel.attachment.isEmpty()) return

        var showDeleteAttachmentDialog by remember { mutableStateOf(false) }

        Box(modifier = Modifier.fillMaxWidth()) {
            val colorFilter = if (isSystemInDarkTheme()) ColorFilter.tint(Color.White) else null
            with(viewModel.attachment) {
                when {
                    contains(".jpg") ||
                            contains(".png") -> {
                        val u = getAttachmentUri(viewModel.attachment)
                        if (u == null) return

                        val input = contentResolver.openInputStream(u)

                        if (input == null) return
                        val bitmap by remember { mutableStateOf(BitmapFactory.decodeStream(input, null, null)!!.asImageBitmap()) }

                        //bitmap.value = BitmapFactory.decodeStream(input, null,null)
                        input.close()
                        // if (bitmap != null) {
                        //   val bm = bitmap!!.asImageBitmap()
                        Image(bitmap = bitmap, contentDescription = "Image", modifier = Modifier.fillMaxWidth().clickable { onAttachmentClick() })
                    }

                    contains(".pdf") -> Image(
                        modifier = Modifier.requiredSize(50.dp).clickable { onAttachmentClick() },
                        painter = painterResource(id = R.drawable.baseline_picture_as_pdf_24),
                        colorFilter = colorFilter,
                        contentDescription = "PDF Attachment"
                    )

                    else -> Image(
                        modifier = Modifier.requiredSize(50.dp).clickable { onAttachmentClick() },
                        painter = painterResource(id = R.drawable.ic_baseline_attach_file_24),
                        colorFilter = colorFilter,
                        contentDescription = "Attachment"
                    )
                }
            }
            IconButton(
                modifier = Modifier.padding(8.dp).size(32.dp) // smaller than default 48.dp
                    .align(Alignment.TopEnd).background(Color.Black.copy(alpha = 0.6f), shape = CircleShape), onClick = { showDeleteAttachmentDialog = true }) {
                Icon(imageVector = Icons.Default.Clear, modifier = Modifier.size(16.dp), contentDescription = "Delete")
            }
        }

        if (showDeleteAttachmentDialog) {
            MaterialAlertDialogBuilder(this)
                .setTitle(resources.getString(string.deleteAttachment))
                .setMessage(resources.getString(string.doYouReallyWantToContinue))
                .setNeutralButton(resources.getString(string.cancel)) { _, _ ->
                    showDeleteAttachmentDialog = false
                }
                .setPositiveButton(resources.getString(string.yes)) { _, _ ->
                    // Does this entry have an associated photo?
                    if (viewModel.attachment.isNotEmpty()) {
                        val file = File(attachmentFolder, viewModel.attachment)
                        if (file.exists()) file.delete()
                    }
                    viewModel.attachment = ""
                    editItem.attachment = ""
                    viewModel.upsert(editItem)

                    snackbarDelegate.showSnackbar(getString(string.attachment) + " " + getString(string.word_deleted))
                }
                .show()
        }

    }

    @Composable
    fun OpenDatePickerDialog() {
        MediLogDatePickerDialog(
            onAccept = {
                showDatePickerDialog = false // close dialog
                if (it != null) { // Set the date
                    viewModel.timestamp = it
                    itemDateString = dateFormat.format(it) // Update text button
                }
            },
            onCancel = { showDatePickerDialog = false }//close dialog
        )
    }

    @OptIn(ExperimentalMaterial3Api::class)
    @Composable
    fun MediLogDatePickerDialog(
        onAccept: (Long?) -> Unit,
        onCancel: () -> Unit,
    ) {
        val state = rememberDatePickerState(initialSelectedDateMillis = viewModel.timestamp + TimeZone.getDefault().rawOffset)

        // Preserve time value
        val oldDateTime = Calendar.getInstance()
        oldDateTime.timeInMillis = viewModel.timestamp

        DatePickerDialog(
            onDismissRequest = { },
            confirmButton = {
                Button(onClick = {
                    if (state.selectedDateMillis != null) {
                        // Convert from UTC and apply preserved time
                        val selectedLocal = Calendar.getInstance()
                        selectedLocal.timeInMillis = state.selectedDateMillis!! - TimeZone.getDefault().rawOffset
                        selectedLocal.set(Calendar.HOUR_OF_DAY, oldDateTime.get(Calendar.HOUR_OF_DAY))
                        selectedLocal.set(Calendar.MINUTE, oldDateTime.get(Calendar.MINUTE))
                        selectedLocal.set(Calendar.SECOND, 0)
                        onAccept(selectedLocal.timeInMillis)
                    }
                }) { Text(getString(string.ok)) }
            },
            dismissButton = { Button(onClick = onCancel) { Text(getString(string.cancel)) } }) {
            DatePicker(state = state)
        }
    }

    @Composable
    fun OpenTimePickerDialog() {
        MediLogTimePickerDialog(
            activity = activity,
            initValue = viewModel.timestamp,
            onCancel = { showTimePickerDialog = false },
            onConfirm = {
                val cal = Calendar.getInstance()
                cal.timeInMillis = viewModel.timestamp
                cal.set(Calendar.HOUR_OF_DAY, it.get(Calendar.HOUR_OF_DAY))
                cal.set(Calendar.MINUTE, it.get(Calendar.MINUTE))
                //cal.isLenient = false
                viewModel.timestamp = cal.timeInMillis // Update timestamp with new time settings
                itemTimeString = timeFormat.format(cal.timeInMillis)
                showTimePickerDialog = false
            },
        )
    }

    // Supporting functions
    fun deleteItem() {
        // Do your really?
        val title = getString(string.deleteItem)
        MaterialAlertDialogBuilder(this)
            .setTitle(title)
            .setMessage(resources.getString(string.doYouReallyWantToContinue))
            .setNeutralButton(resources.getString(string.cancel)) { _, _ ->
                // Respond to neutral button press
            }
            .setPositiveButton(resources.getString(string.yes)) { _, _ ->
                viewModel.delete(editItem._id)

                // Does this entry have an associated photo?
                if (viewModel.attachment.isNotEmpty()) {
                    val file = File(attachmentFolder, viewModel.attachment)
                    //var ret = false
                    if (file.exists()) file.delete()
                    //Log.d("File delete:", ret.toString())
                }
                snackbarDelegate.showSnackbar(getString(string.word_item).replaceFirstChar { it.titlecase(Locale.ROOT) } + " " + getString(string.word_deleted))
                finish()
            }
            .show()
    }

    @SuppressLint("SimpleDateFormat")
    open fun saveItem() {
        // Any temp attachments which need to be confirmed?
        if (unsavedAttachment) viewModel.attachment = createAttachment()

        editItem.value1 = viewModel.value1.value.replace(",", ".")
        editItem.value2 = viewModel.value2.value.replace(",", ".")
        editItem.value3 = viewModel.value3.value.replace(",", ".")
        editItem.value4 = viewModel.value4.value.replace(",", ".")
        editItem.tags = viewModel.tagIds
        editItem.comment = viewModel.comment.value
        editItem.timestamp = viewModel.timestamp
        editItem.attachment = viewModel.attachment

        viewModel.upsert(editItem)
    }

    // +++++++++++++++++++++++++++++++++
    // Attachment support functions
    // +++++++++++++++++++++++++++++++++

    fun attachmentPresent(): Boolean {
        return (viewModel.attachment.isNotEmpty())
    }


    // Move and rename selected file/photo to local encrypted MediLog storage
    // This function gets called upon saving this editItem
    // FileUri is the new file, attachment is either empty or points to an old attachment
    @SuppressLint("SimpleDateFormat")
    fun createAttachment(): String {
        // Any attachments to look after?
        if (viewModel.attachment.isNotEmpty() && !viewModel.attachment.contains("files")) { // If attachment string points to local files it has been pulled into the MediLog file store already
            val resolver = contentResolver
            val fileUri = getAttachmentUri(viewModel.attachment) ?: return ""
            val prefix = com.zell_mbc.medilog.support.ATTACHMENT_LABEL + viewModel.value1.label + "_" + SimpleDateFormat("yyMMddHHmmss").format(Date().time) + "_"
            val newFileName = prefix + getFileName(resolver, fileUri) // Retrieve the filename before saving so we can copy asynch

            viewModel.attachment = newFileName // Store file name only, the actual location is app dependend
            unsavedAttachment = false
            CoroutineScope(Dispatchers.IO).launch { copyFile(resolver, fileUri, newFileName) }
        }
        return viewModel.attachment
    }

    fun copyFile(resolver: ContentResolver, fileUri: Uri, newFileName: String) {
        val inputStream: InputStream? = openFile(resolver, fileUri)
        if (inputStream != null) {
            val newFile = File(attachmentFolder, newFileName)
            inputStream.use { input -> newFile.outputStream().use { output -> input.copyTo(output) } }

            // Do we hold a different file as attachment = Is this an update? Delete old file
            if (editItem.attachment != viewModel.attachment) {
                val oldFile = File(attachmentFolder, editItem.attachment)
                if (oldFile.exists()) oldFile.delete()
            }
        }
    }

    fun openFile(resolver: ContentResolver, fileUri: Uri): InputStream? {
        return try {
            val inputStream: InputStream? = resolver.openInputStream(fileUri)
            inputStream // Return the input stream if successful
        } catch (e: FileNotFoundException) {
            // Handle the case where the file is not found
            snackbarDelegate.showSnackbar("File not found: ${e.message}")
            Log.d("MediLog", "File not found: ${e.message}")
            null
        } catch (e: SecurityException) {
            // Handle the case where the application does not have permission to access this file.
            snackbarDelegate.showSnackbar("No permission to open: ${e.message}")
            null
        } catch (e: Exception) {
            // Catch other potential exceptions (e.g., IOException)
            Log.d("MediLog", "An unknown error occurred: ${e.message}")
            snackbarDelegate.showSnackbar("An unknown error occurred: ${e.message}")
            null
        }
    }

    // Turn Uri into String
    fun getFileName(resolver: ContentResolver, fileUri: Uri): String? {
        var uri = fileUri
        var result: String?

        //if uri is content
        if (uri.scheme != null && uri.scheme == "content") {
            val cursor: Cursor? = resolver.query(uri, null, null, null, null)
            try {
                if (cursor != null && cursor.moveToFirst()) {
                    //local filesystem
                    var index = cursor.getColumnIndex("_data")
                    if (index == -1) //google drive
                        index = cursor.getColumnIndex("_display_name")
                    result = cursor.getString(index)
                    uri = (result?.toUri() ?: return null)
                }
            } finally {
                cursor!!.close()
            }
        }
        result = uri.path

        //get filename + ext of path
        val cut = result!!.lastIndexOf('/')
        if (cut != -1) result = result.substring(cut + 1)
        return result
    }


    // Creates Uri from string
    fun getAttachmentUri(attachment: String): Uri? {
        var fileUri: Uri?

        // Are we dealing with an unsaved file and file != image?
        // If yes return the Uri we kept upon selecting
        if (unsavedAttachmentUri != null) return unsavedAttachmentUri

        //From here on in it can be only unsaved images or saved attachments

        // If a path is present this is an unsaved and unencrypted(!) image
        val tmpAttachment = attachment.contains("/")
        val file = if (tmpAttachment) File(attachment) else File(attachmentFolder, attachment)

        if (file.exists()) {
            fileUri = FileProvider.getUriForFile(this, applicationContext.packageName + ".provider", file)
        } else { // Should not happen…
            snackbarDelegate.showSnackbar(getString(string.eAttachmentError))
            return null
        }

        // Proper attachments are encrypted
        //if (!tmpAttachment) fileUri = decryptAttachment(this, fileUri)

        return fileUri
    }

    fun onAttachmentClick() {
        val uri = getAttachmentUri(viewModel.attachment)
        val intent = Intent(Intent.ACTION_VIEW)
        intent.data = uri
        intent.flags = Intent.FLAG_GRANT_READ_URI_PERMISSION
        try {
            startActivity(intent)
            //Toast.makeText(this, "Attachment clicked", Toast.LENGTH_LONG).show()
        } catch (e: Exception) {
            snackbarDelegate.showSnackbar(getString(string.eShareError) + ": " + e.localizedMessage)
        }
    }

    fun hideKeyboard() {
        val v = currentFocus
        if (v != null) {
            val imm = getSystemService(INPUT_METHOD_SERVICE) as InputMethodManager
            imm.hideSoftInputFromWindow(v.windowToken, 0)
        }
    }


    // Add Attachment functions
    private fun getAttachment() {
        MaterialAlertDialogBuilder(this)
            .setTitle(resources.getString(string.attachmentDialogTitle))
            .setMessage(resources.getString(string.attachmentDialogText))
            .setNegativeButton(resources.getString(string.photo)) { _, _ -> openCamera() }
            .setPositiveButton(resources.getString(string.file)) { _, _ -> getFile() }
            .show()
    }


    private fun getFile() {
        val intent = Intent(Intent.ACTION_OPEN_DOCUMENT).apply {
            addCategory(Intent.CATEGORY_OPENABLE)
            type = "application/*"
        }
        startActivityForResult(intent, FILE_CAPTURE_CODE)
    }

    private fun openCamera() {
        val values = ContentValues()
        values.put(MediaStore.Images.Media.TITLE, "MediLogPicture")
        values.put(MediaStore.Images.Media.DESCRIPTION, "Ad-hoc snapshot")

        //camera intent
        val cameraIntent = Intent(MediaStore.ACTION_IMAGE_CAPTURE)

        // set filename
        //val timeStamp = SimpleDateFormat(MainActivity.DATE_TIME_PATTERN).format(Date().time)
        fileName = "photo.jpg" //ATTACHMENT_LABEL +  viewModel.itemName + "_" + timeStamp + ".jpg"
        //Log.d("New file:", fileName)

        // set directory folder
        val file = File(cacheDir, fileName)
        val fileUri = FileProvider.getUriForFile(this, applicationContext.packageName + ".provider", file)

        cameraIntent.putExtra(MediaStore.EXTRA_OUTPUT, fileUri)
        startActivityForResult(cameraIntent, IMAGE_CAPTURE_CODE)
    }

    @Deprecated("Deprecated in Java")
    override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        super.onActivityResult(requestCode, resultCode, data)
        if (resultCode == RESULT_OK) {
            //Log.d("Request code:",requestCode.toString())
            when (requestCode) {
                FILE_CAPTURE_CODE -> {
                    try {
                        val fileUri = data?.data
                        //attachmentDrawable = R.drawable.ic_baseline_attach_file_24
                        unsavedAttachment = true
                        unsavedAttachmentUri = fileUri
                        viewModel.attachment = fileUri.toString()
                    } catch (_: Exception) {
                        //Toast userOutputService.showMessageAndWaitForLong(this.getString(R.string.eSelectDirectory) + " " + data)
                    }
                }
                // Camera image will be stored in cache folder by Android
                IMAGE_CAPTURE_CODE -> {
                    //File object of camera image
                    val file = File(cacheDir, fileName)
                    //val fileSize = file.length()
                    //Log.d("Photo size:", fileSize.toString())
                    // Todo: if (fileSize > MAX_IMAGE) resize image
                    viewModel.attachment = file.toString() //fileUri.toString()
                    unsavedAttachment = true
                }
            }
        }
    }
}