/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

/*
 */

package com.zell_mbc.medilog.weight

import android.app.Application
import android.content.SharedPreferences
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableFloatStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.core.content.edit
import com.zell_mbc.medilog.ACTIVE_TABS_KEY
import com.zell_mbc.medilog.Tabs
import com.zell_mbc.medilog.R.string
import com.zell_mbc.medilog.data.DataViewModel
import com.zell_mbc.medilog.preferences.SettingsActivity
import com.zell_mbc.medilog.support.checkThresholds
import com.zell_mbc.medilog.support.getDecimalSeparator

class WeightViewModel(application: Application): DataViewModel(application, Tabs.WEIGHT) {
    override val filterStartPref = "WEIGHTFILTERSTART"
    override val filterEndPref = "WEIGHTFILTEREND"
    override val timeFilterModePref =  "WEIGHT_FILTER_MODE"
    override val rollingFilterValuePref = "WEIGHT_ROLLING_FILTER_VALUE"
    override val rollingFilterTimeframePref = "WEIGHT_ROLLING_FILTER_TIMEFRAME"
    override val tagFilterPref = "WEIGHT_TAG_FILTER"
    override val dateTimeColumnWidthPref = "WEIGHT_DATE_COLUMN_WIDTH"

    //All preferences that affect UI or app behavior displayed in UI should be here
    // General settings
    override var blendInItems    by mutableStateOf(preferences.getBoolean(SettingsActivity.KEY_PREF_WEIGHT_BLENDINITEMS, app.getString(string.BLENDINITEMS_DEFAULT).toBoolean()))
    override var highlightValues by mutableStateOf(preferences.getBoolean(SettingsActivity.KEY_PREF_WEIGHT_HIGHLIGHT_VALUES, app.getString(string.WEIGHT_HIGHLIGHT_VALUES_DEFAULT).toBoolean()))
    override var showTime        by mutableStateOf(preferences.getBoolean(SettingsActivity.KEY_PREF_WEIGHT_SHOWTIME, app.getString(string.SHOWTIME_DEFAULT).toBoolean()))
    override var upperThreshold  by mutableFloatStateOf(0f)
    override var lowerThreshold  by mutableFloatStateOf(0f)

    //Tab specific
    var showBodyFat by mutableStateOf(preferences.getBoolean(SettingsActivity.KEY_PREF_LOG_FAT, app.getString(string.LOG_FAT_DEFAULT).toBoolean()))

    var fatUpperThreshold by mutableFloatStateOf(0f)
    var fatLowerThreshold by mutableFloatStateOf(0f)

    // Handle updates
    init {
        setupEntryFields()
        loadThresholds()
        // Handle updates
        prefListener = SharedPreferences.OnSharedPreferenceChangeListener { _, key ->
            when (key) {
                ACTIVE_TABS_KEY,
                SettingsActivity.KEY_PREF_WEIGHT_UNIT,
                SettingsActivity.KEY_PREF_WEIGHT_CUSTOM_NAME,
                SettingsActivity.KEY_PREF_WEIGHT_CUSTOM_UNIT,
                SettingsActivity.KEY_PREF_WEIGHT_CUSTOM_TEMPLATE -> setupEntryFields() // refresh the fields on any related change
                SettingsActivity.KEY_PREF_LOG_FAT -> { showBodyFat = preferences.getBoolean(SettingsActivity.KEY_PREF_LOG_FAT, app.getString(string.LOG_FAT_DEFAULT).toBoolean()) }
                SettingsActivity.KEY_PREF_WEIGHT_BLENDINITEMS -> { blendInItems = preferences.getBoolean(SettingsActivity.KEY_PREF_WEIGHT_BLENDINITEMS, app.getString(string.BLENDINITEMS_DEFAULT).toBoolean()) }
                SettingsActivity.KEY_PREF_WEIGHT_HIGHLIGHT_VALUES -> { highlightValues =preferences.getBoolean(SettingsActivity.KEY_PREF_WEIGHT_HIGHLIGHT_VALUES, app.getString(string.WEIGHT_HIGHLIGHT_VALUES_DEFAULT).toBoolean())}
                SettingsActivity.KEY_PREF_WEIGHT_SHOWTIME -> {
                    showTime = preferences.getBoolean(SettingsActivity.KEY_PREF_WEIGHT_SHOWTIME, app.getString(string.SHOWTIME_DEFAULT).toBoolean())
                    preferences.edit { remove(dateTimeColumnWidthPref) } // Make sure we measure again
                }
                SettingsActivity.KEY_PREF_WEIGHT_THRESHOLDS -> loadThresholds()
            }
            //Toast.makeText(application, "Weight - BlendIn: $blendInItems, Highlight: $highlightValues, ShowTime: $showTime", Toast.LENGTH_LONG).show()
        }
        preferences.registerOnSharedPreferenceChangeListener(prefListener)         // Register preference change listener
    }

    // Specific parameters, applicable to just this field
    override fun setupEntryFields() {
        value1.label = preferences.getString(SettingsActivity.KEY_PREF_WEIGHT_CUSTOM_NAME, app.getString(string.weight)) ?: ""
        value1.unit = preferences.getString(SettingsActivity.KEY_PREF_WEIGHT_CUSTOM_UNIT, "").toString() //preferences.getString(SettingsActivity.KEY_PREF_WEIGHT_UNIT, app.getString(R.string.WEIGHT_UNIT_DEFAULT))) ?: ""
        value1.template = preferences.getString(SettingsActivity.KEY_PREF_WEIGHT_CUSTOM_TEMPLATE, app.getString(string.weightHint)) ?: ""

        // After custom was set once the above won't provide the default any longer
        if (value1.label.isEmpty()) value1.label = app.getString(string.weight)
        if (value1.unit.isEmpty()) value1.unit = preferences.getString(SettingsActivity.KEY_PREF_WEIGHT_UNIT, app.getString(string.WEIGHT_UNIT_DEFAULT)).toString()
        if (value1.template.isEmpty()) value1.template   = app.getString(string.weightHint)
        value1.template = value1.template.replace(".",getDecimalSeparator()) // In case we need a , instead of a .

        // No custom values supported, yet
        value2.label    = app.getString(string.bodyFat)
        value2.unit     = "%"
        value2.template = app.getString(string.bodyFatEntryHint)

        comment.label = app.getString(string.comment)
    }

    private fun loadThresholds() {
        var s = preferences.getString(SettingsActivity.KEY_PREF_WEIGHT_THRESHOLDS, app.getString(string.WEIGHT_THRESHOLD_DEFAULT_KG)) ?: ""
        if (!s.contains("-")) s = "-$s" // Legacy support where only upper limit present
        val th = checkThresholds(app, s, app.getString(string.WEIGHT_THRESHOLD_DEFAULT_KG), value1.label)
        lowerThreshold = th[0].toFloatOrNull() ?: 0f
        upperThreshold = th[1].toFloatOrNull() ?: 0f

        if (showBodyFat && highlightValues) {
            val fatTh = preferences.getString(SettingsActivity.KEY_PREF_FAT_MIN_MAX, app.getString(string.FAT_MIN_MAX_DEFAULT)) ?: app.getString(string.FAT_MIN_MAX_DEFAULT)
            val fatThresh = checkThresholds(app, fatTh, app.getString(string.FAT_MIN_MAX_DEFAULT), app.getString(string.bodyFat))

            fatLowerThreshold = fatThresh[0].toFloatOrNull() ?: 0f //. try { fatThresh[0].toFloat() } catch (_: NumberFormatException) { 0f }
            fatUpperThreshold = fatThresh[1].toFloatOrNull() ?: 0f //} catch (_: NumberFormatException) { 0f }
        }
    }
}