/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

/*
 */

package com.zell_mbc.medilog.temperature

import android.app.Application
import android.content.SharedPreferences
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableFloatStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.core.content.edit
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.R.string
import com.zell_mbc.medilog.Tabs
import com.zell_mbc.medilog.data.DataViewModel
import com.zell_mbc.medilog.preferences.SettingsActivity
import com.zell_mbc.medilog.support.checkThresholds
import com.zell_mbc.medilog.support.getDecimalSeparator

class TemperatureViewModel(application: Application): DataViewModel(application, Tabs.TEMPERATURE) {
    override val filterStartPref = "TEMPERATUREFILTERSTART"
    override val filterEndPref = "TEMPERATUREFILTEREND"
    override val timeFilterModePref = "TEMPERATURE_FILTER_MODE"
    override val rollingFilterValuePref = "TEMPERATURE_ROLLING_FILTER_VALUE"
    override val rollingFilterTimeframePref = "TEMPERATURE_ROLLING_FILTER_TIMEFRAME"
    override val tagFilterPref = "TEMPERATURE_TAG_FILTER"
    override val dateTimeColumnWidthPref = "TEMPERATURE_DATE_COLUMN_WIDTH"

    //All preferences that affect UI or app behavior displayed in UI should be here
    // General settings
    override var blendInItems    by mutableStateOf(preferences.getBoolean(SettingsActivity.KEY_PREF_TEMPERATURE_BLENDINITEMS, app.getString(string.BLENDINITEMS_DEFAULT).toBoolean()))
    override var highlightValues by mutableStateOf(preferences.getBoolean(SettingsActivity.KEY_PREF_TEMPERATURE_HIGHLIGHT_VALUES, application.getString(R.string.TEMPERATURE_HIGHLIGHT_VALUES_DEFAULT).toBoolean()))
    override var showTime        by mutableStateOf(preferences.getBoolean(SettingsActivity.KEY_PREF_TEMPERATURE_SHOWTIME, app.getString(string.SHOWTIME_DEFAULT).toBoolean()))
    override var upperThreshold  by mutableFloatStateOf(0f)
    override var lowerThreshold  by mutableFloatStateOf(0f)

    //Tab specific
    init {
        setupEntryFields()
        loadThresholds()

        // Register preference change listener
        // Handle updates
        prefListener = SharedPreferences.OnSharedPreferenceChangeListener { _, key ->
            when (key) {
                SettingsActivity.KEY_PREF_TEMPERATURE_CUSTOM_NAME,
                SettingsActivity.KEY_PREF_TEMPERATURE_CUSTOM_UNIT,
                SettingsActivity.KEY_PREF_TEMPERATURE_CUSTOM_TEMPLATE -> setupEntryFields() // refresh the fields on any related change
                SettingsActivity.KEY_PREF_TEMPERATURE_THRESHOLDS -> loadThresholds()
                SettingsActivity.KEY_PREF_TEMPERATURE_BLENDINITEMS -> blendInItems = preferences.getBoolean(SettingsActivity.KEY_PREF_TEMPERATURE_BLENDINITEMS, app.getString(string.BLENDINITEMS_DEFAULT).toBoolean())
                SettingsActivity.KEY_PREF_TEMPERATURE_HIGHLIGHT_VALUES -> highlightValues = preferences.getBoolean(SettingsActivity.KEY_PREF_TEMPERATURE_HIGHLIGHT_VALUES, false)
                SettingsActivity.KEY_PREF_TEMPERATURE_SHOWTIME -> {
                    showTime = preferences.getBoolean(SettingsActivity.KEY_PREF_TEMPERATURE_SHOWTIME, app.getString(string.SHOWTIME_DEFAULT).toBoolean())
                    preferences.edit { remove(dateTimeColumnWidthPref) } // Make sure we measure again
                }
            }
        }
        preferences.registerOnSharedPreferenceChangeListener(prefListener)
    }

    // Specific parameters, applicable to just this field
    override fun setupEntryFields() {
        value1.label = preferences.getString(SettingsActivity.KEY_PREF_TEMPERATURE_CUSTOM_NAME, app.getString(R.string.temperature)) ?: app.getString(R.string.temperature)
        value1.unit = preferences.getString(SettingsActivity.KEY_PREF_TEMPERATURE_CUSTOM_UNIT, "").toString() //preferences.getString(SettingsActivity.KEY_PREF_TEMPERATURE_UNIT, app.getString(R.string.TEMPERATURE_UNIT_DEFAULT))) ?: ""
        value1.template = preferences.getString(SettingsActivity.KEY_PREF_TEMPERATURE_CUSTOM_TEMPLATE, app.getString(R.string.temperatureHint)) ?: ""

        // After custom was set once the above won't provide the default any longer
        if (value1.label.isEmpty()) value1.label = app.getString(string.temperature)
        if (value1.unit.isEmpty()) value1.unit = preferences.getString(SettingsActivity.KEY_PREF_TEMPERATURE_UNIT, app.getString(R.string.TEMPERATURE_UNIT_DEFAULT)).toString()
        if (value1.template.isEmpty()) value1.template = app.getString(R.string.temperatureHint)
        value1.template = value1.template.replace(".",getDecimalSeparator()) // In case we need a , instead of a .

        comment.label = app.getString(R.string.comment)
    }

    fun loadThresholds() {
        val defaultThresholds = if (value1.unit == app.getString(R.string.TEMPERATURE_CELSIUS)) app.getString(R.string.TEMPERATURE_THRESHOLDS_CELSIUS_DEFAULT) else app.getString(R.string.TEMPERATURE_THRESHOLDS_FAHRENHEIT_DEFAULT)
        val s = "" + preferences.getString(SettingsActivity.KEY_PREF_TEMPERATURE_THRESHOLDS, defaultThresholds)
        val th = checkThresholds(app, s, defaultThresholds, value1.label)
        lowerThreshold = try { th[0].toFloat() } catch (_: NumberFormatException) { 0f }
        upperThreshold = try { th[1].toFloat() } catch (_: NumberFormatException) { 0f }
    }
}

