/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

/*
 */

package com.zell_mbc.medilog.fluid

import android.app.Application
import android.content.SharedPreferences
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableFloatStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.core.content.edit
import com.zell_mbc.medilog.ActiveProfile
import com.zell_mbc.medilog.MainActivity.Companion.DAY_FORMAT
import com.zell_mbc.medilog.Tabs
import com.zell_mbc.medilog.R.string
import com.zell_mbc.medilog.data.Data
import com.zell_mbc.medilog.data.DataViewModel
import com.zell_mbc.medilog.preferences.SettingsActivity
import com.zell_mbc.medilog.support.checkThresholds
import java.text.SimpleDateFormat
import java.util.Date

class FluidViewModel(application: Application): DataViewModel(application, Tabs.FLUID) {
    override val filterStartPref = "FLUIDFILTERSTART"
    override val filterEndPref = "FLUIDFILTEREND"
    override val timeFilterModePref =  "FLUID_FILTER_MODE"
    override val rollingFilterValuePref = "FLUID_ROLLING_FILTER_VALUE"
    override val rollingFilterTimeframePref = "FLUID_ROLLING_FILTER_TIMEFRAME"
    override val tagFilterPref = "FLUID_TAG_FILTER"
    override val dateTimeColumnWidthPref = "FLUID_DATE_COLUMN_WIDTH"

    //All preferences that affect UI or app behavior displayed in UI should be here
    // General settings
    override var blendInItems    by mutableStateOf(preferences.getBoolean(SettingsActivity.KEY_PREF_FLUID_BLENDINITEMS, app.getString(string.BLENDINITEMS_DEFAULT).toBoolean()))
    override var highlightValues = false //by mutableStateOf(preferences.getBoolean(SettingsActivity.KEY_PREF_FLUID_HIGHLIGHT_VALUES, false))
    override var showTime        by mutableStateOf(preferences.getBoolean(SettingsActivity.KEY_PREF_FLUID_SHOWTIME, app.getString(string.SHOWTIME_DEFAULT).toBoolean()))
    override var upperThreshold  by mutableFloatStateOf(0f)
    override var lowerThreshold  by mutableFloatStateOf(0f)

    //Tab specific
    var tagsEnabled     by mutableStateOf(preferences.getBoolean(SettingsActivity.KEY_PREF_FLUID_ENABLE_TAGS, false))


    init {
        setupEntryFields()
        loadThresholds()
        // Register preference change listener
        // Handle updates
        prefListener = SharedPreferences.OnSharedPreferenceChangeListener { _, key ->
            when (key) {
                SettingsActivity.KEY_PREF_FLUID_CUSTOM_NAME,
                SettingsActivity.KEY_PREF_FLUID_CUSTOM_UNIT,
                SettingsActivity.KEY_PREF_FLUID_CUSTOM_TEMPLATE -> setupEntryFields() // refresh the fields on any related change
                SettingsActivity.KEY_PREF_FLUID_THRESHOLD -> loadThresholds()
                SettingsActivity.KEY_PREF_FLUID_BLENDINITEMS -> blendInItems = preferences.getBoolean(SettingsActivity.KEY_PREF_FLUID_BLENDINITEMS, app.getString(string.BLENDINITEMS_DEFAULT).toBoolean())
                SettingsActivity.KEY_PREF_FLUID_SHOWTIME -> {
                    showTime = preferences.getBoolean(SettingsActivity.KEY_PREF_FLUID_SHOWTIME, app.getString(string.SHOWTIME_DEFAULT).toBoolean())
                    preferences.edit { remove(dateTimeColumnWidthPref) } // Make sure we measure again
                }
                SettingsActivity.KEY_PREF_FLUID_ENABLE_TAGS -> tagsEnabled = preferences.getBoolean(SettingsActivity.KEY_PREF_FLUID_ENABLE_TAGS, false)
            }
        }
        preferences.registerOnSharedPreferenceChangeListener(prefListener)
    }

    // Specific parameters, applicable to just this field
    override fun setupEntryFields() {
        value1.label = preferences.getString(SettingsActivity.KEY_PREF_FLUID_CUSTOM_NAME, "").toString()
        value1.unit = preferences.getString(SettingsActivity.KEY_PREF_FLUID_CUSTOM_UNIT, "").toString()
        value1.template = preferences.getString(SettingsActivity.KEY_PREF_FLUID_CUSTOM_TEMPLATE, app.getString(string.weightHint)) ?: ""

        // After custom was set once the above won't provide the default any longer
        if (value1.label.isEmpty()) value1.label = app.getString(string.fluid)
        if (value1.unit.isEmpty()) value1.unit = preferences.getString(SettingsActivity.KEY_PREF_FLUID_UNIT, app.getString(string.FLUID_UNIT_DEFAULT)).toString()
        if (value1.template.isEmpty()) value1.template   = app.getString(string.fluidHint)
    }

    fun loadThresholds() {
        val defaultThresholds = app.getString(string.FLUID_THRESHOLD_DEFAULT)
        val s = preferences.getString(SettingsActivity.KEY_PREF_FLUID_THRESHOLD, defaultThresholds) + "-" // Add - to string to reflect that it's a lower threshold
        val th = checkThresholds(app, s, defaultThresholds, value1.label)
        lowerThreshold = try { th[0].toFloat() } catch  (_: NumberFormatException) { 0f }

    }

    fun tagCount(tagId: Int = 0, filtered: Boolean = true): Int {
        var string = "SELECT count (*) FROM data WHERE profile_id = ${ActiveProfile.id} AND type = $dataType AND tags "
        string += if (tagId == 0) "= ''" else "LIKE '%$tagId%'"

        if (filtered) {
            getFilter()
            if ((filterStart != 0L) && (filterEnd != 0L)) string += " AND timestamp >= $filterStart AND timestamp <= $filterEnd"
            if ((filterStart == 0L) && (filterEnd != 0L)) string += " AND timestamp <= $filterEnd"
            if ((filterStart != 0L) && (filterEnd == 0L)) string += " AND timestamp >= $filterStart"
        }
        return getInt(string)
    }


    fun getTodayTagSum(tagId: Int): Int {
        val today = SimpleDateFormat(DAY_FORMAT).format(Date().time)
        var string = "SELECT SUM(CAST(value1 as int)) as value1 from data WHERE profile_id = ${ActiveProfile.id} and type=$dataType and value2 = '$today' AND tags "
        string += if (tagId == 0) "LIKE ''" else "LIKE '%$tagId%'"

        return getInt(string)
    }

    // TagId = -1 to ignore tags, = 0 to filter empty tags
    fun getDays(filtered: Boolean, order: String, tagId: Int = -1): MutableList<Data> {
        // Spell out all fields as otherwise the result can't be matched to item = activity crash
        // Make sure to call out all fields explicitly as otherwise the sql statement will crash!!
        //var string = "SELECT _id, timestamp, comment, type, SUM(CAST(value1 as int)) AS value1, value2, value3, value4, attachment, profile_id, category_id, tags FROM data WHERE "
        //var string = "SELECT _id, (strftime('%s', value2 || ' 23:59:59') * 1000) + 999 AS timestamp, comment, type, SUM(CAST(value1 as int)) AS value1, value2, value3, value4, attachment, profile_id, category_id, tags FROM data WHERE "
        var string = "SELECT _id, (strftime('%s', value2) * 1000) + 999 AS timestamp, comment, type, SUM(CAST(value1 as int)) AS value1, value2, value3, value4, attachment, profile_id, category_id, tags FROM data WHERE "
        string += if (blendInItems) " (type = $dataType OR type = ${Tabs.DIARY})"
        else "type=$dataType"

        string += " AND profile_id = ${ActiveProfile.id}"

        if (tagId >= 0) {
            string += " AND tags " + if (tagId == 0) "= ''" else "LIKE '%$tagId%'"
        }

        if (filtered) {
            getFilter()
            if ((filterStart != 0L) && (filterEnd != 0L)) string += " AND timestamp >= $filterStart AND timestamp <= $filterEnd"
            if ((filterStart == 0L) && (filterEnd != 0L)) string += " AND timestamp <= $filterEnd"
            if ((filterStart != 0L) && (filterEnd == 0L)) string += " AND timestamp >= $filterStart"
        }
        // GROUP BY day
        string += " GROUP BY value2 ORDER BY value2 $order"
        return getDataList(string)
    }

    /*fun checkValues(): String {
        // Check empty variables
        if (field1.value.isEmpty()) {
            return app.getString(string.valuesMissing)
        }

        field1.value = field1.value.replace(decimalSeparator, '.')

        //val pattern = "\\d{1," + viewModel.field1.format.length.toString() + "}$"
        val pattern = "\\d{1," + field1.format.length.toString() + "}"
        try {
            if (!Regex(pattern).matches(field1.value)) return app.getString(string.invalidFormat) + "\n" + app.getString(string.configuredFormat, "\"" + field1.format.replace('.', decimalSeparator) + "\"")
        }
        catch (_: Exception) {
            return app.getString(string.unknownError) + "\nRegex: $pattern\nValue: " + field1.value +"\nValue: " + field1.format
        }

        return ""
    }

    fun setField1Parameters() {
        val customName = preferences.getString(SettingsActivity.KEY_PREF_FLUID_CUSTOM_NAME, "").toString()
        if (customName.isEmpty()) {
            customTab = false
            field1.name = app.getString(string.fluid)
            field1.unit = preferences.getString(SettingsActivity.KEY_PREF_FLUID_UNIT, app.getString(string.FLUID_UNIT_DEFAULT)).toString()
            field1.format = app.getString(string.FLUID_THRESHOLD_DEFAULT).replace(Regex("."), "9")
        } else {
            customTab = true
            field1.name = customName
            field1.unit = preferences.getString(SettingsActivity.KEY_PREF_FLUID_CUSTOM_UNIT, "").toString()
            field1.format = preferences.getString(SettingsActivity.KEY_PREF_FLUID_CUSTOM_TEMPLATE, app.getString(string.FLUID_THRESHOLD_DEFAULT).replace(Regex("."), "9")).toString()
        }
        super.setFieldParameters(1)
    }*/
}