/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.fluid

import android.os.Bundle
import android.widget.Toast
import androidx.activity.viewModels
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.safeDrawingPadding
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.preference.PreferenceManager
import com.patrykandpatrick.vico.compose.cartesian.CartesianChartHost
import com.patrykandpatrick.vico.compose.cartesian.axis.rememberBottom
import com.patrykandpatrick.vico.compose.cartesian.axis.rememberStart
import com.patrykandpatrick.vico.compose.cartesian.layer.rememberColumnCartesianLayer
import com.patrykandpatrick.vico.compose.cartesian.layer.rememberLine
import com.patrykandpatrick.vico.compose.cartesian.layer.rememberLineCartesianLayer
import com.patrykandpatrick.vico.compose.cartesian.rememberCartesianChart
import com.patrykandpatrick.vico.compose.cartesian.rememberVicoZoomState
import com.patrykandpatrick.vico.compose.common.ProvideVicoTheme
import com.patrykandpatrick.vico.compose.common.fill
import com.patrykandpatrick.vico.compose.m3.common.rememberM3VicoTheme
import com.patrykandpatrick.vico.core.cartesian.Zoom
import com.patrykandpatrick.vico.core.cartesian.axis.HorizontalAxis
import com.patrykandpatrick.vico.core.cartesian.axis.VerticalAxis
import com.patrykandpatrick.vico.core.cartesian.data.CartesianChartModelProducer
import com.patrykandpatrick.vico.core.cartesian.data.CartesianLayerRangeProvider
import com.patrykandpatrick.vico.core.cartesian.data.lineSeries
import com.patrykandpatrick.vico.core.cartesian.layer.ColumnCartesianLayer.ColumnProvider.Companion.series
import com.patrykandpatrick.vico.core.cartesian.layer.LineCartesianLayer
import com.patrykandpatrick.vico.core.common.component.LineComponent
import com.zell_mbc.medilog.Tabs.FLUID
import com.zell_mbc.medilog.base.ChartActivity
import com.zell_mbc.medilog.base.rememberMarker
import com.zell_mbc.medilog.data.Data
import com.zell_mbc.medilog.preferences.SettingsActivity
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import kotlin.collections.mutableListOf
import com.zell_mbc.medilog.R.string
import com.zell_mbc.medilog.tags.TagsViewModel
import kotlin.getValue

class FluidChartActivity: ChartActivity() {
    // Activities need to be self sufficient because MainActivity might get killed by OS
    val viewModel: FluidViewModel by viewModels()
    val tagsViewModel: TagsViewModel by viewModels()

    override val filename = "WaterChart.jpg"

    var fluid = mutableListOf<Float>()
    var thresholdLine = 0

    private val itemList = mutableListOf<MutableList<Data>>()
    private var tagsEnabled = false
    private var dailyView  = false

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        val preferences = PreferenceManager.getDefaultSharedPreferences(this)
        tagsEnabled      = preferences.getBoolean(SettingsActivity.KEY_PREF_FLUID_ENABLE_TAGS, false)
        showGrid = preferences.getBoolean(SettingsActivity.KEY_PREF_SHOW_FLUID_GRID, true)
        showLegend = preferences.getBoolean(SettingsActivity.KEY_PREF_SHOW_FLUID_LEGEND, false)
        showValues = preferences.getBoolean(SettingsActivity.KEY_PREF_FLUID_SHOW_VALUES, true)
        showMovingAverage = preferences.getBoolean(SettingsActivity.KEY_PREF_FLUID_SHOW_MOVING_AVERAGE, false)
        showThreshold = preferences.getBoolean(SettingsActivity.KEY_PREF_SHOW_FLUID_THRESHOLD, getString(string.SHOW_FLUID_THRESHOLD_DEFAULT).toBoolean())
        if (showThreshold) {
            val temp = "" + preferences.getString(SettingsActivity.KEY_PREF_FLUID_THRESHOLD, "")
            thresholdLine = try { temp.toInt() } catch (_: NumberFormatException) { getString(string.FLUID_THRESHOLD_DEFAULT).toInt() }
        }

        // For the charts only native entries can be considered, no matter what showAllTabs is set to
        val tmp = viewModel.blendInItems
        dailyView = preferences.getBoolean(SettingsActivity.KEY_PREF_FLUID_SUMMARY_CHART, false)

        viewModel.blendInItems = false

        // ######################
        // We essentially support only 3 tags, or noTag + 2 tags
        // ######################
        chartColors = mutableListOf(lineColor1, lineColor2, lineColor3)
        if (tagsEnabled) {
            //legendText.clear()
            //chartColors.clear()
            // No tag set
            val items = if (dailyView) viewModel.getDays(filtered = true, order = "ASC", tagId = 0) else viewModel.getItems("ASC", filtered = true, tagId = 0)
            var i = 0
            if (items.isNotEmpty()) {
                itemList.add(items)
                //chartColors.add(Color(0)) // No tag means no color -> go for black
                if (showValues) legendList.add(legendItem(label = getString(string.noTag), color = chartColors[i++]))
            }
            for (tag in tagsViewModel.getTagList(FLUID)) {
                val items = if (dailyView) viewModel.getDays(filtered = true, order = "ASC", tagId = tag._id) else viewModel.getItems("ASC", filtered = true, tagId = tag._id)
                if (items.isNotEmpty() && i < chartColors.size) {
                    itemList.add(items)
                    //chartColors.add(Color(tag.color.toColorInt()))
                    if (showValues) legendList.add(legendItem(label = tag.tag, color = chartColors[i++])) // Color(tag.color.toColorInt()))) // Todo: Tag colour or chartColor?
                }
            }
        }
        else { // Tags disabled
            if (showValues) legendList.add(legendItem(label = getString(string.fluid), color = chartColors[0]))
            val items = if (dailyView) viewModel.getDays(filtered = true, order = "ASC") else viewModel.getItems("ASC", filtered = true)
            itemList.add(items)
        }
        viewModel.blendInItems = tmp

        if (itemList[0].size < 2) {
            Toast.makeText(this, getString(string.notEnoughDataForChart), Toast.LENGTH_LONG).show()
            finish()
            return // Needs return to avoid the rest to be executed because finish() may not kill the activity fast enough
        }

        if (!(showValues || showLinearTrendline || showMovingAverage)) {
            Toast.makeText(this, getString(string.noValuesActivated), Toast.LENGTH_LONG).show()
            finish()
            return // Needs return to avoid the rest to be executed because finish() may not kill the activity fast enough
        }

        var fTmp: Float
        var i = 0
        // Collect list values and hold min/max values. Always use actual values, the trend lines won't be more extreme
        //for (items in itemList) {
        // Support first 3 tags for now
        for (item in itemList[0]) {
            timestamps.add(i++)
            fTmp = try { item.value1.toFloat() } catch (_: NumberFormatException) { 0f }
            value1List.add(fTmp)
            if (fTmp > chartMax) chartMax = fTmp
            if (fTmp < chartMin) chartMin = fTmp
        }
        if (itemList.size > 1)
            for (item in itemList[1]) {
                timestamps.add(i++)
                fTmp = try { item.value1.toFloat() } catch (_: NumberFormatException) { 0f }
                value2List.add(fTmp)
                if (fTmp > chartMax) chartMax = fTmp
                if (fTmp < chartMin) chartMin = fTmp
            }
        if (itemList.size > 2)
            for (item in itemList[1]) {
                timestamps.add(i++)
                fTmp = try { item.value3.toFloat() } catch (_: NumberFormatException) { 0f }
                value3List.add(fTmp)
                if (fTmp > chartMax) chartMax = fTmp
                if (fTmp < chartMin) chartMin = fTmp
            }

        // Add a visual buffer below
        chartMin -= 10f

        if (showValues) {
            value1Map = itemList[0].associateBy({ scaledTimeStamp(it.timestamp) }, { try { it.value1.toFloat() } catch (_: NumberFormatException) { 0F } })
            if (itemList.size > 1) value2Map = itemList[1].associateBy({ scaledTimeStamp(it.timestamp) }, { try { it.value1.toFloat() } catch (_: NumberFormatException) { 0F } })
            if (itemList.size > 2) value3Map = itemList[2].associateBy({ scaledTimeStamp(it.timestamp) }, { try { it.value1.toFloat() } catch (_: NumberFormatException) { 0F } })
        }

        // Compile moving average values
        if (showMovingAverage) {
            legendList.add(legendItem(label = getString(string.showMovingAverage), color = movingAverageColor))
            val p = preferences.getString(SettingsActivity.KEY_PREF_FLUID_MOVING_AVERAGE_SIZE, "6")
            if (p != null) {
                val period = try { p.toInt() } catch (_: NumberFormatException) { 0 }
                movingAverageValue1List = getMovingAverageFloat(value1List, period)
                var ii = 0
                movingAverageValue1Map = itemList[0].associateBy({ scaledTimeStamp(it.timestamp) }, { try { movingAverageValue1List[ii++] } catch (_: NumberFormatException) { 0F }})

                if (itemList.size > 1) {
                    movingAverageValue2List = getMovingAverageFloat(value2List, period)
                    ii = 0
                    movingAverageValue2Map = itemList[1].associateBy({ scaledTimeStamp(it.timestamp) }, { try { movingAverageValue2List[ii++] } catch (_: NumberFormatException) { 0F } })
                }
                if (itemList.size > 2) {
                    movingAverageValue3List = getMovingAverageFloat(value3List, period)
                    ii = 0
                    movingAverageValue3Map = itemList[2].associateBy({ scaledTimeStamp(it.timestamp) }, { try { movingAverageValue3List[ii++] } catch (_: NumberFormatException) { 0F } })
                }
            }
        }

        // Band aid, todo
        items = itemList[0]

        // Fill data maps
        // Get first record of first tag
        xAxisOffset = itemList[0][0].timestamp
        // Check if there's an earlier one
        for (item in itemList)
            xAxisOffset = if (item[0].timestamp < xAxisOffset) item[0].timestamp else xAxisOffset

        showContent()
    }

    @Composable
    override fun ShowContent() {
        Box(Modifier.safeDrawingPadding()) {
            //chartColors = mutableListOf(Color(0xffb983ff))

            val modelProducer = remember { CartesianChartModelProducer() }
            LaunchedEffect(Unit) {
                withContext(Dispatchers.Default) {
                    modelProducer.runTransaction {
                        lineSeries {
                        if (showValues) {
                            value1Map?.let { series(it.keys, it.values) }
                            value2Map?.let { series(it.keys, it.values) }
                            value3Map?.let { series(it.keys, it.values) }
                        }
                        if (showMovingAverage) {
                            movingAverageValue1Map?.let { series(it.keys, it.values) }
                            movingAverageValue2Map?.let { series(it.keys, it.values) }
                            movingAverageValue3Map?.let { series(it.keys, it.values) }
                        }
                    }
                }
            } }
            ProvideVicoTheme(rememberM3VicoTheme()) {
            LineChart(modelProducer, Modifier.fillMaxHeight())
            }
        }
    }

    @Composable
    private fun LineChart(modelProducer: CartesianChartModelProducer, modifier: Modifier) {
        val lineProvider = LineCartesianLayer.LineProvider.series(
            buildList {
                if (showValues) {
                    add(LineCartesianLayer.rememberLine(areaFill = LineCartesianLayer.AreaFill.single(fill = fill(chartColors[0].copy(alpha = 0.2f))), fill = LineCartesianLayer.LineFill.single(fill = fill(chartColors[0]))))
                    if (value2Map != null) add(LineCartesianLayer.rememberLine(areaFill = LineCartesianLayer.AreaFill.single(fill = fill(chartColors[1].copy(alpha = 0.2f))), fill = LineCartesianLayer.LineFill.single(fill = fill(chartColors[1]))))
                    if (value3Map != null) add(LineCartesianLayer.rememberLine(areaFill = LineCartesianLayer.AreaFill.single(fill = fill(chartColors[2].copy(alpha = 0.2f))), fill = LineCartesianLayer.LineFill.single(fill = fill(chartColors[2]))))
                }
                if (showMovingAverage) {
                    add(LineCartesianLayer.rememberLine(fill = LineCartesianLayer.LineFill.single(fill(chartColors[1])), areaFill = null))
                    add(LineCartesianLayer.rememberLine(fill = LineCartesianLayer.LineFill.single(fill(chartColors[1])), areaFill = null))
                    add(LineCartesianLayer.rememberLine(fill = LineCartesianLayer.LineFill.single(fill(chartColors[1])), areaFill = null))
                }
                if (showLinearTrendline) {
                    add(LineCartesianLayer.rememberLine(stroke = linearTrendLineStroke, fill = LineCartesianLayer.LineFill.single(fill(chartColors[0])), areaFill = null))
                    add(LineCartesianLayer.rememberLine(stroke = linearTrendLineStroke, fill = LineCartesianLayer.LineFill.single(fill(chartColors[1])), areaFill = null))
                    add(LineCartesianLayer.rememberLine(stroke = linearTrendLineStroke, fill = LineCartesianLayer.LineFill.single(fill(chartColors[2])), areaFill = null))
                }
            }
        )

        val span = chartMax - chartMin
        val step = findNiceStep(span.toDouble())
        val start = determineYAxisStart(chartMin)

        CartesianChartHost(
            chart = rememberCartesianChart(
                rememberLineCartesianLayer(lineProvider = lineProvider, rangeProvider = CartesianLayerRangeProvider.fixed(minY = start.toDouble(), maxY = chartMax.toDouble())),
                startAxis =  VerticalAxis.rememberStart(
                    guideline = guideline(), // horizontal lines
                    itemPlacer = remember { VerticalAxis.ItemPlacer.step(step = { step }, shiftTopLines = false) }
                ),
                bottomAxis = HorizontalAxis.rememberBottom(
                    valueFormatter = bottomAxisValueFormatter,
                    guideline = guideline(),
                ),
                legend = if (showLegend) rememberLegend() else null,
                decorations = if (showThreshold) listOf(helperLine("", thresholdLine.toDouble(), chartColors[0])) else listOf(),
                marker = rememberMarker()
            ),
            modelProducer = modelProducer,
            modifier = modifier,
            zoomState = rememberVicoZoomState(zoomEnabled = true, maxZoom = Zoom.max(Zoom.fixed(100f), Zoom.Content), initialZoom = Zoom.min(Zoom.fixed(), Zoom.Content)),
        )
    }

    @Composable
    private fun ColumnChart(modelProducer: CartesianChartModelProducer, modifier: Modifier) {
        val columnProvider = series(
            buildList { for (c in chartColors) add(LineComponent(fill = fill(c))) }
        )

        CartesianChartHost(
            chart = rememberCartesianChart(
                rememberColumnCartesianLayer(columnProvider = columnProvider), //rangeProvider = CartesianLayerRangeProvider.fixed(minY = minY.toDouble(),maxY = maxY.toDouble())),
                startAxis = VerticalAxis.rememberStart(guideline = guideline()),
                bottomAxis = HorizontalAxis.rememberBottom(valueFormatter = bottomAxisValueFormatter, guideline = guideline()),
                legend = if (showLegend) rememberLegend() else null,
                decorations = if (showThreshold) listOf(helperLine("", thresholdLine.toDouble(), chartColors[0])) else listOf(),
                marker = rememberMarker()
            ),
            modelProducer = modelProducer,
            modifier = modifier,
            zoomState = rememberVicoZoomState(zoomEnabled = true, maxZoom = Zoom.max(Zoom.fixed(100f), Zoom.Content), initialZoom = Zoom.min(Zoom.fixed(), Zoom.Content)),
        )
    }
}