/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.bloodpressure

import android.app.Application
import android.content.SharedPreferences
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableFloatStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.compose.ui.graphics.Color
import androidx.core.content.edit
import com.zell_mbc.medilog.R.string
import com.zell_mbc.medilog.Tabs.BLOODPRESSURE
import com.zell_mbc.medilog.data.DataViewModel
import com.zell_mbc.medilog.preferences.SettingsActivity

class BloodPressureViewModel(application: Application): DataViewModel(application, BLOODPRESSURE) {
    override val filterStartPref = "BLOODPRESSUREFILTERSTART"
    override val filterEndPref = "BLOODPRESSUREFILTEREND"
    override val timeFilterModePref =  "BLOODPRESSURE_FILTER_MODE"
    override val rollingFilterValuePref = "BLOODPRESSURE_ROLLING_FILTER_VALUE"
    override val rollingFilterTimeframePref = "BLOODPRESSURE_ROLLING_FILTER_TIMEFRAME"
    override val tagFilterPref = "BLOODPRESSURE_TAG_FILTER"
    override val dateTimeColumnWidthPref = "BLOODPRESSURE_DATE_COLUMN_WIDTH"

    //All preferences that affect UI or app behavior displayed in UI should be here
    // General settings
    override var blendInItems    by mutableStateOf(preferences.getBoolean(SettingsActivity.KEY_PREF_BLOODPRESSURE_BLENDINITEMS, app.getString(string.BLENDINITEMS_DEFAULT).toBoolean()))
    override var highlightValues by mutableStateOf(preferences.getBoolean(SettingsActivity.KEY_PREF_BLOODPRESSURE_HIGHLIGHT_VALUES, app.getString(string.BLOODPRESSURE_HIGHLIGHT_VALUES_DEFAULT).toBoolean()))
    override var showTime        by mutableStateOf(preferences.getBoolean(SettingsActivity.KEY_PREF_BLOODPRESSURE_SHOWTIME, app.getString(string.SHOWTIME_DEFAULT).toBoolean()))
    override var upperThreshold  by mutableFloatStateOf(0f)
    override var lowerThreshold  by mutableFloatStateOf(0f)

    //Tab specific
    var logHeartRhythm by mutableStateOf(preferences.getBoolean(SettingsActivity.KEY_PREF_LOG_HEART_RHYTHM, application.getString(string.LOG_HEARTRHYTHM_DEFAULT).toBoolean()))
    var heartRhythmIssue by  mutableStateOf(false)
    var iconColor by mutableStateOf(Color.Black)

    // Stats page values
    val placeholder = 999

    init {
        setupEntryFields()

        // Register preference change listener
        // Handle updates
        prefListener = SharedPreferences.OnSharedPreferenceChangeListener { _, key ->
            when (key) {
                SettingsActivity.KEY_PREF_BLOODPRESSURE_CUSTOM_NAME1,
                SettingsActivity.KEY_PREF_BLOODPRESSURE_CUSTOM_UNIT,
                SettingsActivity.KEY_PREF_BLOODPRESSURE_CUSTOM_TEMPLATE -> {
                    setupEntryFields() // refresh the fields on any related change
                }
                SettingsActivity.KEY_PREF_LOG_HEART_RHYTHM -> { logHeartRhythm = preferences.getBoolean(SettingsActivity.KEY_PREF_LOG_HEART_RHYTHM, false) }
                SettingsActivity.KEY_PREF_BLOODPRESSURE_BLENDINITEMS -> { blendInItems = preferences.getBoolean(SettingsActivity.KEY_PREF_BLOODPRESSURE_BLENDINITEMS, false) }
                SettingsActivity.KEY_PREF_BLOODPRESSURE_HIGHLIGHT_VALUES -> { highlightValues = preferences.getBoolean(SettingsActivity.KEY_PREF_BLOODPRESSURE_HIGHLIGHT_VALUES, app.getString(string.BLOODPRESSURE_HIGHLIGHT_VALUES_DEFAULT).toBoolean())}
                SettingsActivity.KEY_PREF_BLOODPRESSURE_SHOWTIME -> {
                    showTime = preferences.getBoolean(SettingsActivity.KEY_PREF_BLOODPRESSURE_SHOWTIME, app.getString(string.SHOWTIME_DEFAULT).toBoolean())
                    preferences.edit { remove(dateTimeColumnWidthPref) } // Make sure we measure again
                }
            }
            //Toast.makeText(application, "BloodPressure - BlendIn: $blendInItems, Highlight: $highlightValues, ShowTime: $showTime", Toast.LENGTH_LONG).show()
        }
        preferences.registerOnSharedPreferenceChangeListener(prefListener)
    }

    // Specific parameters, applicable to just this field
    override fun setupEntryFields() {
        value1.label = preferences.getString(SettingsActivity.KEY_PREF_BLOODPRESSURE_CUSTOM_NAME1, "").toString()
        value1.unit = preferences.getString(SettingsActivity.KEY_PREF_BLOODPRESSURE_CUSTOM_UNIT, "").toString()
        value1.template = preferences.getString(SettingsActivity.KEY_PREF_BLOODPRESSURE_CUSTOM_TEMPLATE, "").toString()

        // After custom was set once the above won't provide the default any longer
        if (value1.label.isEmpty()) value1.label = app.getString(string.systolic)
        if (value1.unit.isEmpty()) value1.unit = preferences.getString(SettingsActivity.KEY_PREF_BLOODPRESSURE_UNIT, app.getString(string.BLOODPRESSURE_UNIT_DEFAULT)).toString()
        if (value1.template.isEmpty()) value1.template   = app.getString(string.bpEntryHint)

        value2.label = app.getString(string.diastolic)
        value2.unit = preferences.getString(SettingsActivity.KEY_PREF_BLOODPRESSURE_UNIT, app.getString(string.BLOODPRESSURE_UNIT_DEFAULT)).toString()
        value2.template = app.getString(string.bpEntryHint)

        value3.label = app.getString(string.pulse)
        value3.unit = app.getString(string.pulseUnit)
        value3.template = app.getString(string.bpEntryHint)

        comment.label = app.getString(string.comment)
    }
}
