/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.bloodpressure

import android.content.Context
import androidx.compose.foundation.Image
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.IntrinsicSize
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.text.KeyboardActions
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.outlined.Abc
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.material3.TextField
import androidx.compose.material3.VerticalDivider
import androidx.compose.runtime.Composable
import androidx.compose.runtime.SideEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusDirection
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.ColorFilter
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.PlatformTextStyle
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.font.FontStyle
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.input.KeyboardCapitalization
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.TextUnit
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.zell_mbc.medilog.ActiveProfile
import com.zell_mbc.medilog.Filter.NO_TAGS
import com.zell_mbc.medilog.MainActivity
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.R.string
import com.zell_mbc.medilog.Thresholds
import com.zell_mbc.medilog.base.BaseTab
import com.zell_mbc.medilog.data.Data
import com.zell_mbc.medilog.preferences.SettingsActivity
import com.zell_mbc.medilog.preferences.SettingsActivity.Companion.KEY_PREF_QUICKENTRY
import com.zell_mbc.medilog.scaffold.Screens
import com.zell_mbc.medilog.support.MedilogTheme
import com.zell_mbc.medilog.support.SnackbarDelegate
import com.zell_mbc.medilog.support.checkMinMax
import com.zell_mbc.medilog.support.matchesTemplate
import com.zell_mbc.medilog.support.shouldMoveFocus
import com.zell_mbc.medilog.tags.TagsViewModel
import java.util.Date

class BloodPressureTab(override val label: String?, override val viewModel: BloodPressureViewModel, override val tagsViewModel: TagsViewModel, context: Context, snackbarDelegate: SnackbarDelegate): BaseTab(viewModel, tagsViewModel, context, snackbarDelegate) {
    override var editActivityClass: Class<*> = BloodPressureEditActivity::class.java
    override var infoActivityClass: Class<*> = BloodPressureInfoActivity::class.java
    override var chartActivityClass: Class<*> = BloodPressureChartActivity::class.java

    override val route = Screens.BloodPressure.route
    override val fabEnabled = true
    override val iconActive = R.drawable.ic_blood_pressure_filled
    override val iconInactive = R.drawable.ic_blood_pressure_outlined

    // -------------------------------
    private var bpHelper: BloodPressureHelper = BloodPressureHelper(context)
    private var colorBad = Color.Black // Placeholders
    private var colorGood = Color.Black // Placeholders

    lateinit var multipleMeasurementsDialog: MultipleMeasurementsDialog
    var openMultipleMeasurementsDialog = mutableStateOf(false)

    @Composable
    override fun ShowContent(padding: PaddingValues) {
        super.ShowContent(padding)

        // Global variables
        val quickEntry by viewModel.quickEntryFlow.collectAsState()

        colorBad = MaterialTheme.colorScheme.error
        colorGood = MaterialTheme.colorScheme.primary

        var showComment by remember { mutableStateOf(viewModel.comment.value.isNotEmpty()) }
        val value1Width = measureTextWidth(viewModel.value1.template, MainActivity.fontSize)
        viewModel.iconColor = setIcon()

        MedilogTheme {
            LazyColumn(state = listState, horizontalAlignment = Alignment.Start, modifier = Modifier.fillMaxWidth().padding(start = 8.dp, end = 8.dp)) {
                item {
                    if (quickEntry) {             // Only show DataEntryRow if quick entry is enabled
                        val (textField1, focusDia, focusPulse, focusComment) = remember { FocusRequester.createRefs() }
                        Row(Modifier.fillMaxWidth()) {
                            TextField(
                                value = viewModel.value1.value,
                                onValueChange = {
                                    // 1. Check template
                                    if (matchesTemplate(it, viewModel.value1.template))
                                        // 2. Max/min reached?
                                        if (checkMinMax(field = it, min = 0, max = Thresholds.MAX_SYS)) {
                                            viewModel.value1.value = it // Single source of truth!
                                            // 3. Jump if needed
                                            if (shouldMoveFocus(it, viewModel.value1.template, Thresholds.MAX_SYS))
                                                focusManager?.moveFocus(FocusDirection.Next)
                                            }
                                },
                                colors = textFieldColors,
                                singleLine = true,
                                keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Decimal),
                                textStyle = TextStyle(),
                                label = { Text(text = "${viewModel.value1.label}*", maxLines = 1, overflow = TextOverflow.Ellipsis) },
                                placeholder = { Text("${viewModel.value1.template} ${viewModel.value1.unit}") },
                                modifier = Modifier.weight(1f).focusRequester(textField1).padding(end = 10.dp),
                                keyboardActions = KeyboardActions(onDone = { addItem() })
                            )

                            // Dia
                            TextField(
                                value = viewModel.value2.value,
                                onValueChange = { it ->
                                    // 1. Check template
                                    if (matchesTemplate(it, viewModel.value2.template))
                                    // 2. Max/min reached?
                                        if (checkMinMax(field = it, min = 0, max = Thresholds.MAX_DIA)) {
                                            viewModel.value2.value = it // Single source of truth!
                                            // 3. Jump if needed
                                            if (shouldMoveFocus(it, viewModel.value2.template, Thresholds.MAX_DIA))
                                                focusManager?.moveFocus(FocusDirection.Next)
                                        }
                                },
                                colors = textFieldColors,
                                singleLine = true,
                                keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Decimal),
                                textStyle = TextStyle(),
                                label = { Text(text = "${viewModel.value2.label}*", maxLines = 1, overflow = TextOverflow.Ellipsis) },
                                placeholder = { Text("${viewModel.value2.template} ${viewModel.value2.unit}") },
                                modifier = Modifier.weight(1f).focusRequester(focusDia).padding(end = 10.dp),
                                keyboardActions = KeyboardActions(onDone = { addItem() })
                            )
                            // Pulse
                            TextField(
                                value = viewModel.value3.value,
                                onValueChange = {
                                    showComment = true
                                    // 1. Check template
                                    if (matchesTemplate(it, viewModel.value3.template))
                                    // 2. Max/min reached?
                                        if (checkMinMax(field = it, min = 0, max = Thresholds.MAX_PULSE)) {
                                            viewModel.value3.value = it // Single source of truth!
                                            // 3. Jump if needed
                                            if (shouldMoveFocus(it, viewModel.value3.template, Thresholds.MAX_PULSE))
                                                focusManager?.moveFocus(FocusDirection.Next)
                                        }
                                },
                                singleLine = true,
                                keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Decimal),
                                textStyle = TextStyle(),
                                label = { Text(text = viewModel.value3.label, maxLines = 1, overflow = TextOverflow.Ellipsis) },
                                placeholder = { Text("${viewModel.value3.template} ${viewModel.value3.unit}") },
                                modifier = Modifier.weight(1f).focusRequester(focusPulse).padding(end = 10.dp),
                                keyboardActions = KeyboardActions(onDone = { addItem() }),
                                colors = textFieldColors
                            )

                            if (viewModel.logHeartRhythm) {
                                Image(painter = painterResource(R.drawable.ic_baseline_warning_24), contentDescription = "Heart Rhythm State",
                                    modifier = Modifier.weight(0.5f).padding(start = 16.dp).align(Alignment.CenterVertically).clickable {
                                        viewModel.heartRhythmIssue = !viewModel.heartRhythmIssue
                                        viewModel.iconColor = setIcon()
                                        }, colorFilter = ColorFilter.tint(viewModel.iconColor),)
                            }
                        } // Row

                        if (showComment) {
                            // Comment
                            TextField(
                                value = viewModel.comment.value,
                                colors = textFieldColors,
                                onValueChange = { viewModel.comment.value = it },
                                singleLine = true,
                                textStyle = TextStyle(),
                                label = { Text(text = viewModel.comment.label) },
                                keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Text, capitalization = KeyboardCapitalization.Sentences),
                                modifier = Modifier.fillMaxWidth().focusRequester(focusComment),
                                trailingIcon = { IconButton(onClick = {
                                    openTextTemplatesDialog.value = true })
                                { Icon(Icons.Outlined.Abc,contentDescription = null) } },
                                keyboardActions = KeyboardActions(onDone = { addItem() }))
                        }
                        SideEffect {
                            // Set cursor to first field after addItem completed
                            if (activateFirstField) {
                                textField1.requestFocus()
                                activateFirstField = false }
                        }

                        // Dialog section
                        if (openTextTemplatesDialog.value) {
                            textTemplateDialog.ShowDialog(viewModel.dataType, setShowDialog = { openTextTemplatesDialog.value = it }) {
                                viewModel.comment.value += it
                                focusComment.requestFocus()
                            }
                            ShowKeyboard() // Todo: Not working reliably
                        }
                        if (openMultipleMeasurementsDialog.value) multipleMeasurementsDialog.ShowDialog(setShowDialog = { openMultipleMeasurementsDialog.value = it })
                        //if (viewModel.openSearchDialog.value) SearchDialog(setShowDialog = { viewModel.openSearchDialog.value = it })
                    }
                }
                // Add key= to give compose a stable handle in case of filter changes during recompositions
                items(dataList.value, key = { it._id }) { item ->
                    HorizontalDivider(color = MaterialTheme.colorScheme.secondaryContainer, thickness = 1.dp) // Lines starting from the top
                    Row(modifier = Modifier.fillMaxWidth().height(IntrinsicSize.Min).width(IntrinsicSize.Min).clickable { selection.value = item }, verticalAlignment = Alignment.CenterVertically)
                    {
                        ShowRow(item, value1Width)
                    }

                    if (selection.value != null) {
                        ItemClicked(selection.value!!._id)
                        selection.value = null
                    }
                }
            }
        }
    }

    @Composable
    fun ShowRow(item: Data, value1Width: Dp) {
        ResizableDateColumn(formatDateString(item.timestamp))
        //Text(formatDateString(item.timestamp), Modifier.padding(end = cellPadding.dp, top = rowPadding.dp, bottom = rowPadding.dp).width(dateColumnWidth), color = MaterialTheme.colorScheme.primary, fontSize = MainActivity.fontSize.sp)
        //VerticalDivider(color = MaterialTheme.colorScheme.secondaryContainer, modifier = Modifier.width(1.dp).fillMaxHeight()) // Vertical separator

        var fontStyle = FontStyle.Normal
        var textColor = MaterialTheme.colorScheme.primary
        var fontWeight = FontWeight.Normal

        // A diary item blended in
        if (item.type != viewModel.dataType) {
            fontStyle = FontStyle.Italic
            item.value1 = ""
            item.value2 = ""
            item.value3 = ""
            // Comment
            val s = item.comment + if (item.attachment.isNotEmpty()) " $PAPERCLIP" else ""
            Text(s, Modifier.padding(start = cellPadding.dp), fontWeight = fontWeight, color = textColor, fontSize = MainActivity.fontSize.sp, fontStyle = fontStyle)
            return
        }

        if (viewModel.logHeartRhythm) {
            // Check value
            var heartRhythm = -1
            if (item.value4.isNotEmpty()) {
                heartRhythm = try {
                    item.value4.toInt()
                } catch (_: NumberFormatException) {
                    0
                }
            }
            // Show image only if value > 0
            val s = if (heartRhythm > 0) stringResource(id = string.warningSign)
            else "   "
            Text(s, Modifier.padding(start = cellPadding.dp, end = cellPadding.dp), color = MaterialTheme.colorScheme.primary, fontSize = MainActivity.fontSize.sp)
            VerticalDivider(color = MaterialTheme.colorScheme.secondaryContainer, modifier = Modifier.width(1.dp).fillMaxHeight()) // Vertical separator
        }
        // Sys
        if (item.value1.isNotEmpty() && viewModel.highlightValues) {
            when (bpHelper.sysGrade(item.value1)) {
                bpHelper.hyperGrade3 -> {
                    textColor = MaterialTheme.colorScheme.error
                    fontWeight = FontWeight.Bold
                }
                bpHelper.hyperGrade2 -> textColor = MaterialTheme.colorScheme.error
                bpHelper.hyperGrade1 -> textColor = MainActivity.AMBER
                bpHelper.hypotension -> textColor = MainActivity.AMBER
            }
        }
        Text(item.value1, Modifier.padding(start = cellPadding.dp, end = cellPadding.dp).width(value1Width), textAlign = TextAlign.Center, fontWeight = fontWeight, color = textColor, fontSize = MainActivity.fontSize.sp, fontStyle = fontStyle)
        VerticalDivider(color = MaterialTheme.colorScheme.secondaryContainer, modifier = Modifier.width(1.dp).fillMaxHeight()) // Vertical separator

        // Dia
        textColor = MaterialTheme.colorScheme.primary
        fontWeight = FontWeight.Normal
        if (item.value2.isNotEmpty() && viewModel.highlightValues) {
            when (bpHelper.diaGrade(item.value2)) {
                bpHelper.hyperGrade3 -> {
                    textColor = MaterialTheme.colorScheme.error
                    fontWeight = FontWeight.Bold
                }
                bpHelper.hyperGrade2 -> textColor = MaterialTheme.colorScheme.error
                bpHelper.hyperGrade1 -> textColor = MainActivity.AMBER
                bpHelper.hypotension -> textColor = MainActivity.AMBER
            }
        }
        Text(item.value2, Modifier.padding(start = cellPadding.dp, end = cellPadding.dp).width(value1Width), textAlign = TextAlign.Center, fontWeight = fontWeight, color = textColor, fontSize = MainActivity.fontSize.sp, fontStyle = fontStyle)
        textColor = MaterialTheme.colorScheme.primary
        fontWeight = FontWeight.Normal
        VerticalDivider(color = MaterialTheme.colorScheme.secondaryContainer, modifier = Modifier.width(1.dp).fillMaxHeight()) // Vertical separator

        // Pulse
        Text(item.value3,
            Modifier
                .padding(start = cellPadding.dp, end = cellPadding.dp)
                .width(value1Width), textAlign = TextAlign.Center, fontWeight = fontWeight, color = textColor, fontSize = MainActivity.fontSize.sp, fontStyle = fontStyle)
        VerticalDivider(color = MaterialTheme.colorScheme.secondaryContainer, modifier = Modifier
            .width(1.dp)
            .fillMaxHeight()) // Vertical separator

        // Comment
        val s = item.comment + if (item.attachment.isNotEmpty()) " $PAPERCLIP" else ""
        Text(s, Modifier.padding(start = cellPadding.dp), fontWeight = fontWeight, color = textColor, fontSize = MainActivity.fontSize.sp, fontStyle = fontStyle,
            style = TextStyle(lineHeight = TextUnit.Unspecified, platformStyle = PlatformTextStyle(includeFontPadding = false)) // style is for removing the padding between multiline text
        )
    }

    override fun addItem() {
        // If standard entry is active create a data element and launch the editActivity
        if (!preferences.getBoolean(KEY_PREF_QUICKENTRY, true)) {
            startEditing(-1) // Indicate that this is about a new item
            return
        }

        // ###########################
        // Checks
        // ###########################
        // Close keyboard so errors are visible
        hideKeyboard()

        // Safely convert string values
        var validInput = matchesTemplate(viewModel.value1)
        if (!validInput) {
            val message =  context.getString(string.invalidFormat) + "\n" + context.getString(string.configuredFormat, viewModel.value1.template)
            snackbarDelegate.showSnackbar(message)
            return
        }
        validInput = matchesTemplate(viewModel.value2)
        if (!validInput) {
            val message =  context.getString(string.invalidFormat) + "\n" + context.getString(string.configuredFormat, viewModel.value1.template)
            snackbarDelegate.showSnackbar(message)
            return
        }
        if (viewModel.value3.value.isNotEmpty()) { // Pulse
            validInput = matchesTemplate(viewModel.value3)
            if (!validInput) {
                val message = context.getString(string.invalidFormat) + "\n" + context.getString(string.configuredFormat, viewModel.value1.template)
                snackbarDelegate.showSnackbar(message)
                return
            }
        }

        val item = Data(0, timestamp = Date().time, comment = viewModel.comment.value, type = viewModel.dataType, value1 = viewModel.value1.value, value2 = viewModel.value2.value, value3 = viewModel.value3.value, value4 = if (viewModel.heartRhythmIssue) "1" else "0", "", tags = NO_TAGS, category_id = -1, profile_id = ActiveProfile.id)
        viewModel.upsertBlocking(item)

        val multipleMeasurementsTimeWindow = "" + preferences.getString(SettingsActivity.KEY_PREF_BLOODPRESSURE_MULTI_MEASUREMENTS_WINDOW, context.getString(string.BLOODPRESSURE_MULTI_MEASUREMENTS_WINDOW_DEFAULT))
        if (multipleMeasurementsTimeWindow.toInt() > 0) {
            multipleMeasurementsDialog = MultipleMeasurementsDialog(context, viewModel)
            openMultipleMeasurementsDialog.value = true
        }

        if ((viewModel.filterEnd > 0L) && (viewModel.filterEnd < item.timestamp)) snackbarDelegate.showSnackbar(context.getString(string.filteredOut))

        cleanUpAfterAddItem()
        viewModel.heartRhythmIssue = false
        viewModel.iconColor = setIcon()
    }

    // Helper functions
    fun setIcon(): Color {
        //viewModel.heartRhythmIssue = !viewModel.heartRhythmIssue
        return if (viewModel.heartRhythmIssue) colorBad else colorGood
    }
}