/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */
package com.zell_mbc.medilog.base

import android.content.Context
import android.content.Intent
import android.content.SharedPreferences
import android.widget.Toast
import androidx.compose.foundation.Image
import androidx.compose.foundation.clickable
import androidx.compose.foundation.gestures.detectDragGesturesAfterLongPress
import androidx.compose.foundation.isSystemInDarkTheme
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.lazy.LazyListState
import androidx.compose.foundation.lazy.rememberLazyListState
import androidx.compose.material3.LocalTextStyle
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.material3.TextFieldColors
import androidx.compose.material3.TextFieldDefaults
import androidx.compose.material3.VerticalDivider
import androidx.compose.runtime.Composable
import androidx.compose.runtime.MutableState
import androidx.compose.runtime.State
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusManager
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.ColorFilter
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.platform.LocalFocusManager
import androidx.compose.ui.platform.LocalSoftwareKeyboardController
import androidx.compose.ui.platform.SoftwareKeyboardController
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.text.AnnotatedString
import androidx.compose.ui.text.LinkAnnotation
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.TextLinkStyles
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.rememberTextMeasurer
import androidx.compose.ui.text.withLink
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.core.content.edit
import androidx.preference.PreferenceManager
import com.zell_mbc.medilog.MainActivity.Companion.Delimiter
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.R.string
import com.zell_mbc.medilog.data.Data
import com.zell_mbc.medilog.data.DataViewModel
import com.zell_mbc.medilog.dialogs.toHexString
import com.zell_mbc.medilog.texttemplates.TextTemplateDialog
import com.zell_mbc.medilog.support.SnackbarDelegate
import com.zell_mbc.medilog.support.getCorrectedDateFormat
import com.zell_mbc.medilog.tags.TagsDialog
import java.text.DateFormat
import java.util.Locale
import java.util.regex.Pattern
import androidx.core.net.toUri
import com.zell_mbc.medilog.MainActivity.Companion.fontSize
import com.zell_mbc.medilog.debug.DebugLog
import com.zell_mbc.medilog.debug.DebugOverlay
import com.zell_mbc.medilog.support.link
import com.zell_mbc.medilog.support.onAttachmentClick
import com.zell_mbc.medilog.tags.TagsViewModel
import java.text.SimpleDateFormat

interface NavigationTab {
    val route: String
    val label: String?
    val iconActive: Int
    val iconInactive: Int
    val fabEnabled: Boolean get() = true
    val viewModel: DataViewModel

    fun addItem()
    fun showInfoScreen(context: Context)
    fun showChartScreen(context: Context)
    fun startEditing(index: Int) // Launch edit activity

    @Composable fun ShowContent(padding: PaddingValues)
    }

abstract class BaseTab(vm: DataViewModel, open val tagsViewModel: TagsViewModel, val context: Context, val snackbarDelegate: SnackbarDelegate): NavigationTab {
    abstract var editActivityClass: Class<*>
    abstract var infoActivityClass: Class<*>
    abstract var chartActivityClass: Class<*>

    override val viewModel = vm
    override val fabEnabled: Boolean get() = true
    abstract override val route: String

    lateinit var dataList: State<List<Data>>
    lateinit var listState: LazyListState

    lateinit var selection: MutableState<Data?>
    lateinit var textFieldColors: TextFieldColors

    var keyboardController: SoftwareKeyboardController? = null
    var focusManager: FocusManager? = null

    val preferences: SharedPreferences = PreferenceManager.getDefaultSharedPreferences(context)
    val dateFormat: DateFormat = getCorrectedDateFormat(context)

    val is24Hour = android.text.format.DateFormat.is24HourFormat(context)
    val timeFormat: DateFormat = if (is24Hour) {
        SimpleDateFormat("HH:mm", Locale.getDefault()) // 24-hour
    } else {
        SimpleDateFormat("hh:mm a", Locale.getDefault()) // 12-hour
    }
    //private val timeFormat: DateFormat = DateFormat.getTimeInstance(DateFormat.SHORT, Locale.getDefault())

    //var quickEntry: Boolean = true
    var selectedField = ""

    var value1Width = 0.dp
    var value2Width = 0.dp
    var value3Width = 0.dp
    //var dateColumnWidth2 = 0.dp


    val PAPERCLIP =  "\uD83D\uDCCE"

    val cellPadding = 5 // Padding before and after a text in the grid
    var rowPadding = 1 // Padding tab list rows (top & bottom)
    var activateFirstField = false

    // Dialog control variables
    var showTagsDialog = mutableStateOf(false)
    var openTextTemplatesDialog = mutableStateOf(false)

    var textTemplateDialog: TextTemplateDialog
    var tagsDialog: TagsDialog

    val urlPattern: Pattern = Pattern.compile(
        "(?:^|[\\W])((ht|f)tp(s?):\\/\\/|www\\.)"
                + "(([\\w\\-]+\\.){1,}?([\\w\\-.~]+\\/?)*"
                + "[\\p{Alnum}.,%_=?&#\\-+()\\[\\]\\*$~@!:/{};']*)",
        Pattern.CASE_INSENSITIVE or Pattern.MULTILINE or Pattern.DOTALL
    )


    init {
        textTemplateDialog = TextTemplateDialog(context)
        tagsDialog = TagsDialog(context)
    }

    @Composable
    open fun PreShowContent(padding: PaddingValues) {
        val debugMode by DebugLog.debugMode.collectAsState(initial = false)
        if (debugMode) DebugOverlay() { ShowContent(padding = padding) }
        else ShowContent(padding = padding)
    }

    @Composable
    override fun ShowContent(padding: PaddingValues) {
        keyboardController = LocalSoftwareKeyboardController.current
        focusManager = LocalFocusManager.current
        textFieldColors = TextFieldDefaults.colors(unfocusedContainerColor = Color.Transparent, focusedContainerColor = Color.Transparent)

        // Recreate the query flow whenever blendInItems changes
        dataList = remember(viewModel.blendInItems) { viewModel.query() }.collectAsState(initial = emptyList())

        selection = remember { mutableStateOf(null) }
        listState = rememberLazyListState()

        // If time/date can be set per tab, columnWidth needs to be tab specific
        val initialWidth = preferences.getFloat(viewModel.dateTimeColumnWidthPref, -1f)
        viewModel.dateColumnWidthState.floatValue = if (initialWidth > 0f) initialWidth else measureDateString(viewModel.showTime, fontSize)
    }

    override fun showInfoScreen(context: Context) {
        val items = viewModel.getItems("ASC", filtered = true)
        if (items.isNotEmpty()) {
            val intent = Intent(context, infoActivityClass)
            context.startActivity(intent)
        }
        else Toast.makeText(context, context.getString(string.warning) + " " + context.getString(string.noDataToShow), Toast.LENGTH_LONG).show()
    }

    override fun showChartScreen(context: Context) {
        val items = viewModel.getItems("ASC", filtered = true)
        if (items.size < 2) {
            Toast.makeText(context, context.getString(string.notEnoughDataForChart), Toast.LENGTH_LONG).show()
            return
        }

        val intent = Intent(context, chartActivityClass)
        context.startActivity(intent)
    }

    @Composable
    fun ResizableDateColumn(dateString: String, textColor: Color = MaterialTheme.colorScheme.primary, minWidth: Dp = 50.dp, maxWidth: Dp = 200.dp, ) {
        val density = LocalDensity.current
        val widthDp = viewModel.dateColumnWidthState.floatValue.dp

        // Date column
        Text(text = dateString, modifier = Modifier.width(widthDp).padding(end = 4.dp, top = 2.dp, bottom = 2.dp), color = textColor, fontSize = fontSize.sp)

        // Draggable divider with long press
        VerticalDivider(color = MaterialTheme.colorScheme.secondaryContainer,
            modifier = Modifier.width(8.dp) // wider touch target
                .fillMaxHeight()
                .pointerInput(Unit) {
                    detectDragGesturesAfterLongPress(
                        onDrag = { change, dragAmount ->
                            change.consume()
                            val deltaDp = (dragAmount.x / density.density).dp
                            val newWidth = (viewModel.dateColumnWidthState.floatValue.dp + deltaDp).coerceIn(minWidth, maxWidth)
                            viewModel.dateColumnWidthState.floatValue = newWidth.value
                        },
                        onDragEnd = { preferences.edit { putFloat(viewModel.dateTimeColumnWidthPref, viewModel.dateColumnWidthState.floatValue) } }
                    )
                }
        )
    }

    // Helper functions
    // Measure width of date string
    @Composable
    fun measureDateString(showTime: Boolean = true, fontSize: Int): Float {
        val sampleString = when {
            !showTime -> "12/31/99"
            !is24Hour -> "12/31/99 - 11:59 PM"
            else -> "12/31/99 - 11:59"
        }

        val density = LocalDensity.current
        val textMeasurer = rememberTextMeasurer()
        val textStyle = LocalTextStyle.current.copy(fontSize = fontSize.sp)
        val padding = 8
        val widthPx = remember(sampleString, fontSize, textStyle) {
            textMeasurer.measure(text = sampleString, style = textStyle).size.width.toFloat()
        }
        return with(density) { (widthPx + padding).toDp().value }  // convert pixels → dp
    }

    @Composable
    fun measureTextWidth(text: String, fontSize: Int): Dp {
        val density = LocalDensity.current
        val textMeasurer = rememberTextMeasurer()
        val textStyle = LocalTextStyle.current.copy(fontSize = fontSize.sp)
        val padding = 0
        val widthPx = remember(text, fontSize, textStyle) {
            textMeasurer.measure(text = text, style = textStyle).size.width.toFloat()
        }
        return with(density) { (widthPx + padding).toDp() }  // convert pixels → dp
    }

    @Composable
    fun measureTextWidth2(text: String, fontSize: Int): Dp {
        // Remember the text measurer once per composition
        val textMeasurer = rememberTextMeasurer()
        val density = LocalDensity.current

        // Recalculate width only if text or fontSize changes
        return remember(text, fontSize) {
            with(density) {
                val textLayoutResult = textMeasurer.measure(text = text, style = TextStyle(fontSize = fontSize.sp))
                val padding = 8.dp
                textLayoutResult.size.width.toDp() + padding
            }
        }
    }

    @Composable
    fun ItemClicked(selectedId: Int) {
        val editItem = viewModel.getItem(selectedId)

        // Are we dealing with a genuine item or one which is blended in?
        if (editItem != null) {
            if (editItem.type != viewModel.dataType) {
                snackbarDelegate.showSnackbar(context.getString(string.blendedItem))
                return
            }
        }
        else {
            snackbarDelegate.showSnackbar("Error: Item with id $selectedId not found!")
            return
        }

        val openDialog = remember { mutableStateOf(true) }

        if (selectedField == "comment") {
            selectedField = ""
            val urlMatcher = urlPattern.matcher(editItem.comment)
            var matchStart: Int
            var matchEnd: Int
            var url: String
            if (urlMatcher.find()) {
                matchStart = urlMatcher.start(1)
                matchEnd = urlMatcher.end()
                url = editItem.comment.substring(matchStart, matchEnd)

                val intent = Intent(Intent.ACTION_VIEW)
                intent.data = url.toUri()
                context.startActivity(intent)
            }
            else if (openDialog.value) startEditing(selectedId)
        }
        else if (openDialog.value) startEditing(selectedId)
                //showDialog.value = false

        openDialog.value = false
    }

    override fun startEditing(index: Int) {
        val intent = Intent(context, editActivityClass)
        intent.putExtra("editItemIndex", index)
        context.startActivity(intent)
    }

    fun hideKeyboard() {
        /*        val v = MainActivity.mediLog.currentFocus
                if (v != null) {
                    val imm = MainActivity.mediLog.getSystemService(AppCompatActivity.INPUT_METHOD_SERVICE) as InputMethodManager
                    imm.hideSoftInputFromWindow(v.windowToken, 0)
                }*/
    }

    // Displays the icon matching the file extension
    @Composable
    fun HandleAttachment(attachment: String) {
        val colorFilter = if (isSystemInDarkTheme()) ColorFilter.tint(Color.White) else null
        Image(
            modifier = Modifier.padding(start = cellPadding.dp).clickable { onAttachmentClick(context, attachment) },
            painter = painterResource(id = getDrawable(attachment)), contentDescription = "Attachment icon",
            colorFilter = colorFilter)
    }

    @Composable
    fun HideKeyboard() {
        val controller = LocalSoftwareKeyboardController.current
        controller?.hide()
        /*val v = context.currentFocus
        if (v != null) {
            val imm = activity?.getSystemService(AppCompatActivity.INPUT_METHOD_SERVICE) as InputMethodManager
            imm.hideSoftInputFromWindow(v.windowToken, 0)
        }*/
    }

    fun showKeyboard() {
        /*val v = context.currentFocus
        if (v != null) {
            val imm = activity?.getSystemService(AppCompatActivity.INPUT_METHOD_SERVICE) as InputMethodManager
            v.requestFocus()
            imm.showSoftInput(v, 0) // Todo: Doesn't work reliably
        }*/
    }

    @Composable
    fun ShowKeyboard() {
        val controller = LocalSoftwareKeyboardController.current
        controller?.show()
    }

    fun formatDateString(m: Long): String = if (viewModel.showTime) dateFormat.format(m) + " - " + timeFormat.format(m) else dateFormat.format(m)

    // Post addItem actions
    fun cleanUpAfterAddItem() {
        // Clean up UI
        viewModel.value1.value = ""
        viewModel.value2.value = ""
        viewModel.value3.value = ""
        viewModel.value4.value = ""
        viewModel.comment.value = ""
        viewModel.tagIds = ""

        hideKeyboard()
        activateFirstField = true //<- Potentially make this a setting? Annoying if you add only one item at a time
    }

    @Composable
    fun compileAnnotatedString(source: String): AnnotatedString {
        val urlMatcher = urlPattern.matcher(source)
        var matchStart: Int
        var matchEnd: Int
        var url: String
        var retString: AnnotatedString
        if (urlMatcher.find()) {
            matchStart = urlMatcher.start(1)
            matchEnd = urlMatcher.end()
            url = source.substring(matchStart, matchEnd)

            retString = buildAnnotatedString {
                append(source.substring(0, matchStart))
                withLink(LinkAnnotation.Url(url = url, TextLinkStyles(style = SpanStyle(color = MaterialTheme.colorScheme.link)))) { append(url) }
                append(source.substring(matchEnd))
            }
        }
        else retString = buildAnnotatedString { append(source) }
        return retString
    }

    // Take tags, retrieve the associated color and return the color code
    fun getTagColor(tags: String): String {
        if (tags.isBlank()) return ""

        var colorString = ""

        // Todo: Pick color of first tag until I can think of a better way to show multiple tags
        val tagStrings = tags.split(Delimiter.TAG)
        if (tagStrings[0].isNotEmpty()) {
            val tmpColorString = tagsViewModel.getColor(tagStrings[0])
            if (tmpColorString != Color.Transparent.toHexString()) colorString = tmpColorString // Don't touch color if no tag color is set
        }
        return colorString
    }

    // Returns the correct icon based on file type
    fun getDrawable(fileName: String): Int {
        return when {
            fileName.contains(".jpg") || fileName.contains(".png") -> R.drawable.baseline_image_24
            fileName.contains(".pdf") -> R.drawable.baseline_picture_as_pdf_24
            else -> R.drawable.ic_baseline_attach_file_24
        }
    }
}
