/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog

import android.app.Dialog
import android.content.ActivityNotFoundException
import android.content.Context
import android.content.Intent
import android.content.Intent.*
import android.content.SharedPreferences
import android.content.res.Configuration
import android.graphics.Color.RED
import android.net.Uri
import android.os.Build
import android.os.Bundle
import android.text.InputType
import android.text.InputType.TYPE_TEXT_VARIATION_PASSWORD
import android.text.format.DateUtils
import android.view.*
import android.view.inputmethod.InputMethodManager
import android.widget.Button
import android.widget.EditText
import android.widget.TextView
import android.widget.Toast
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.activity.result.contract.ActivityResultContracts
import androidx.activity.viewModels
import androidx.appcompat.app.AppCompatActivity
import androidx.appcompat.app.AppCompatDelegate.*
import androidx.biometric.BiometricPrompt
import androidx.biometric.BiometricPrompt.PromptInfo
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.systemBarsPadding
import androidx.compose.foundation.layout.wrapContentSize
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowRight
import androidx.compose.material.icons.automirrored.outlined.Send
import androidx.compose.material.icons.filled.Add
import androidx.compose.material.icons.filled.AreaChart
import androidx.compose.material.icons.filled.Info
import androidx.compose.material.icons.outlined.Download
import androidx.compose.material.icons.outlined.FilterList
import androidx.compose.material.icons.outlined.FilterListOff
import androidx.compose.material.icons.outlined.FolderZip
import androidx.compose.material.icons.outlined.Info
import androidx.compose.material.icons.outlined.OpenInBrowser
import androidx.compose.material.icons.outlined.Person
import androidx.compose.material.icons.outlined.Settings
import androidx.compose.material.icons.outlined.Upgrade
import androidx.compose.material.icons.outlined.Upload
import androidx.compose.material.icons.rounded.MoreVert
import androidx.compose.material.icons.rounded.Search
import androidx.compose.material3.BottomAppBarDefaults
import androidx.compose.material3.DropdownMenu
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.FloatingActionButton
import androidx.compose.material3.FloatingActionButtonDefaults
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.MediumTopAppBar
import androidx.compose.material3.NavigationBar
import androidx.compose.material3.NavigationBarItem
import androidx.compose.material3.Scaffold
import androidx.compose.material3.SnackbarHost
import androidx.compose.material3.SnackbarHostState
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.runtime.Composable
import androidx.compose.runtime.MutableState
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableIntStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.input.nestedscroll.nestedScroll
import androidx.compose.ui.platform.LocalSoftwareKeyboardController
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import androidx.core.content.edit
import androidx.core.net.toUri
import androidx.core.view.WindowCompat
import androidx.documentfile.provider.DocumentFile
import androidx.fragment.app.FragmentActivity
import androidx.lifecycle.viewModelScope
import androidx.navigation.NavController
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.composable
import androidx.navigation.compose.currentBackStackEntryAsState
import androidx.navigation.compose.rememberNavController
import androidx.preference.PreferenceManager
import com.google.android.material.color.DynamicColors
import com.google.android.material.dialog.MaterialAlertDialogBuilder
import com.google.android.material.snackbar.Snackbar
import com.zell_mbc.medilog.R.*
import com.zell_mbc.medilog.Tabs.BLOODPRESSURE
import com.zell_mbc.medilog.Tabs.DIARY
import com.zell_mbc.medilog.Tabs.DOCUMENTS
import com.zell_mbc.medilog.Tabs.FLUID
import com.zell_mbc.medilog.Tabs.GLUCOSE
import com.zell_mbc.medilog.Tabs.OXIMETRY
import com.zell_mbc.medilog.Tabs.TEMPERATURE
import com.zell_mbc.medilog.Tabs.WEIGHT
import com.zell_mbc.medilog.base.BasePdf
import com.zell_mbc.medilog.bloodpressure.BloodPressurePdf
import com.zell_mbc.medilog.bloodpressure.BloodPressureTab
import com.zell_mbc.medilog.bloodpressure.BloodPressureViewModel
import com.zell_mbc.medilog.data.Backup
import com.zell_mbc.medilog.data.DataViewModel
import com.zell_mbc.medilog.data.ImportActivity
import com.zell_mbc.medilog.data.TextTemplates
import com.zell_mbc.medilog.diary.DiaryPdf
import com.zell_mbc.medilog.diary.DiaryTab
import com.zell_mbc.medilog.diary.DiaryViewModel
import com.zell_mbc.medilog.documents.DocumentsTab
import com.zell_mbc.medilog.documents.DocumentsViewModel
import com.zell_mbc.medilog.fluid.FluidPdf
import com.zell_mbc.medilog.fluid.FluidTab
import com.zell_mbc.medilog.fluid.FluidViewModel
import com.zell_mbc.medilog.glucose.GlucosePdf
import com.zell_mbc.medilog.glucose.GlucoseTab
import com.zell_mbc.medilog.glucose.GlucoseViewModel
import com.zell_mbc.medilog.billing.BillingManager
import com.zell_mbc.medilog.billing.ThankYouForPurchaseDialog
import com.zell_mbc.medilog.billing.UnlimitedUpgradeDialog
import com.zell_mbc.medilog.billing.UpgradeProductInfo
import com.zell_mbc.medilog.billing.BillingUnavailableDialog
import com.zell_mbc.medilog.billing.BillingManagerImpl
import com.zell_mbc.medilog.debug.DebugLog
import com.zell_mbc.medilog.dialogs.DatabaseImportDialog
import com.zell_mbc.medilog.dialogs.DatabaseShareDialog
import com.zell_mbc.medilog.onboarding.OnboardingPage
import com.zell_mbc.medilog.oximetry.OximetryPdf
import com.zell_mbc.medilog.oximetry.OximetryTab
import com.zell_mbc.medilog.oximetry.OximetryViewModel
import com.zell_mbc.medilog.preferences.SettingsActivity
import com.zell_mbc.medilog.preferences.SettingsActivity.Companion.KEY_PREF_APP_THEME
import com.zell_mbc.medilog.preferences.SettingsActivity.Companion.KEY_PREF_AUTO_BACKUP
import com.zell_mbc.medilog.preferences.SettingsActivity.Companion.KEY_PREF_BACKUP_WARNING
import com.zell_mbc.medilog.preferences.SettingsActivity.Companion.KEY_PREF_FORCE_REAUTHENTICATION
import com.zell_mbc.medilog.profiles.ProfilesViewModel
import com.zell_mbc.medilog.scaffold.MediLogSearchBar
import com.zell_mbc.medilog.scaffold.NotYetAuthorized
import com.zell_mbc.medilog.scaffold.Screens
import com.zell_mbc.medilog.about.AboutScreen
import com.zell_mbc.medilog.base.NavigationTab
import com.zell_mbc.medilog.dialogs.ChangelogDialog
import com.zell_mbc.medilog.profiles.PreferenceSync
import com.zell_mbc.medilog.profiles.PreferenceSync.migrateLegacyProfilePrefs
import com.zell_mbc.medilog.profiles.ProfilesScreen
import com.zell_mbc.medilog.scaffold.fullScreenRoutes
import com.zell_mbc.medilog.settings.SettingsViewModel
import com.zell_mbc.medilog.support.AppStart
import com.zell_mbc.medilog.support.BiometricHelper
import com.zell_mbc.medilog.support.MedilogTheme
import com.zell_mbc.medilog.support.SnackbarDelegate
import com.zell_mbc.medilog.support.getAllTabMeta
import com.zell_mbc.medilog.support.getVersionCode
import com.zell_mbc.medilog.support.loadEnabledTabs
import com.zell_mbc.medilog.support.setDefaults
import com.zell_mbc.medilog.tags.TagsViewModel
import com.zell_mbc.medilog.temperature.TemperaturePdf
import com.zell_mbc.medilog.temperature.TemperatureTab
import com.zell_mbc.medilog.temperature.TemperatureViewModel
import com.zell_mbc.medilog.texttemplates.TextTemplatesViewModel
import com.zell_mbc.medilog.weight.WeightPdf
import com.zell_mbc.medilog.weight.WeightTab
import com.zell_mbc.medilog.weight.WeightViewModel
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.io.*
import java.net.HttpURLConnection
import java.net.URL
import java.net.URLEncoder
import java.text.DecimalFormatSymbols
import java.util.*
import java.util.UUID.randomUUID
import java.util.concurrent.Executor
import java.util.concurrent.Executors
import java.util.concurrent.TimeUnit
import kotlin.concurrent.thread
import kotlin.getValue

class MainActivity : AppCompatActivity() {
    // ViewModels
    val weightViewModel: WeightViewModel by viewModels()
    val bloodPressureViewModel: BloodPressureViewModel by viewModels()
    val diaryViewModel: DiaryViewModel by viewModels()
    val fluidViewModel: FluidViewModel by viewModels()
    val glucoseViewModel: GlucoseViewModel by viewModels()
    val temperatureViewModel: TemperatureViewModel by viewModels()
    val oximetryViewModel: OximetryViewModel by viewModels()
    val documentsViewModel: DocumentsViewModel by viewModels()

    val profilesViewModel: ProfilesViewModel by viewModels()
    val textTemplatesViewModel: TextTemplatesViewModel by viewModels()
    val tagsViewModel:TagsViewModel by viewModels()

    private lateinit var preferences: SharedPreferences
    lateinit var viewModel:DataViewModel

    // Billing stuff for Google Play
    private val unlimitedRecords  = mutableStateOf(false)
    private val showThankYouDialogState = mutableStateOf(false)
    private val showBillingUnavailableDialogState = mutableStateOf(false)
    val billingManager = BillingManagerProvider.instance     // Billing/Licensing stuff, only relevant for GooglePlay version, empty for FOSS
    object BillingManagerProvider { val instance: BillingManager by lazy { BillingManagerImpl() } }     // Use a singleton to avoid recreating every time MainActivity get's called

    private val enabledTabsState = mutableStateOf<List<Int>>(emptyList())
    //private var activeProfile: Profiles? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        enableEdgeToEdge() //This will include/color the top Android info bar
        super.onCreate(savedInstanceState)

        medilogApplicationContext = applicationContext as MediLog

        // Callback functions! Not executed just yet, these get called from Billing manager
        billingManager.showBillingUnavailableDialog = { show -> showBillingUnavailableDialogState.value = show }
        billingManager.showThankYouDialog = { show -> showThankYouDialogState.value = show }
        billingManager.setUnlimitedRecords = { value -> unlimitedRecords.value = value }

        preferences = PreferenceManager.getDefaultSharedPreferences(this)

        // Load profile so the individual tabs can properly initialize
        val ap = if (profilesViewModel.count() == 0L) {
            // First launch: create new profile
            profilesViewModel.createNewProfile().also {
                preferences.edit { putInt(ACTIVE_PROFILE_KEY, it._id) }
            }
        } else {
            // Load last used or fall back to first
            val id = preferences.getInt(ACTIVE_PROFILE_KEY, UNSET)
            profilesViewModel.get(id) ?: profilesViewModel.get(profilesViewModel.getFirst()).also {
                if (id == UNSET) { // Only save if no profile was stored
                    preferences.edit { putInt(ACTIVE_PROFILE_KEY, it?._id ?: UNSET) }
                }
            }
        }
        ActiveProfile.id = ap?._id ?: UNSET
        ActiveProfile.name = ap?.name ?: ""

        // Migrate legacy preferences, delete after a few releases
        migrateLegacyProfilePrefs(this)

        // Initial load
        enabledTabsState.value = loadEnabledTabs(preferences)

        // Watch for changes
        val listener = SharedPreferences.OnSharedPreferenceChangeListener { _, key ->
            if (key == ACTIVE_TABS_KEY) {
                enabledTabsState.value = loadEnabledTabs(preferences)
            }
        }
        preferences.registerOnSharedPreferenceChangeListener(listener)

        // Check license
        //preferences.edit { putBoolean(UnlimitedRecords, false) } // Activating this line will force billing checks
        checkLicense()

        val window = this@MainActivity.window

        viewModel = weightViewModel
        if (preferences.getBoolean(SettingsActivity.KEY_PREF_BLOCK_SCREENSHOTS, false)) window.setFlags(WindowManager.LayoutParams.FLAG_SECURE, WindowManager.LayoutParams.FLAG_SECURE)

        userFeedbackLevel = preferences.getInt("USER_FEEDBACK_SETTING", 0) // Drives level of feedback (nothing / standard / debug)

        // Check DynamicColor
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
            dynamicColorOn = preferences.getBoolean(SettingsActivity.KEY_PREF_DYNAMIC_COLOR, false)
            if (dynamicColorOn) DynamicColors.applyToActivitiesIfAvailable(application) // Required for view based screens
        }

        // fontSize = either what's in the preferences or default
        val defaultFontSize = getString(string.TEXT_SIZE_DEFAULT)
        fontSize = preferences.getString(SettingsActivity.KEY_PREF_TEXT_SIZE, defaultFontSize)?.toInt() ?: defaultFontSize.toInt()

        val theme = preferences.getString(KEY_PREF_APP_THEME, getString(string.DARK_MODE_DEFAULT))
        val themes = resources.getStringArray(array.themeValues)

        when (theme) {
            themes[1] -> setDefaultNightMode(MODE_NIGHT_NO)
            themes[2] -> setDefaultNightMode(MODE_NIGHT_YES)
            themes[3] -> setDefaultNightMode(MODE_NIGHT_AUTO_BATTERY)
            else -> setDefaultNightMode(MODE_NIGHT_FOLLOW_SYSTEM)
        }

        // ********************* Debug code **************************
        // Force crash for debug purposes
        //throw RuntimeException("This is a forced crash for debug purposes");
        /*
        var sql = "SELECT * FROM data WHERE type==3 AND LENGTH(value1) > 1"
        val liveAttachments = viewModel.getDataList(sql)
        for (item in liveAttachments) {
            if (item.type == 3) {
                item.value1 = ""
                item.value2 = ""
                item.value3 = ""
                item.value4 = ""
                viewModel.upsert(item)
            }
        }
*/
        // ********************* /Debug code **************************

        preferences.edit {
            // Initialize delimiter on first run based on locale
            var delimiter = preferences.getString(SettingsActivity.KEY_PREF_DELIMITER, "")
            if (delimiter.equals("")) {
                val currentLanguage = Locale.getDefault().displayLanguage
                delimiter = if (currentLanguage.contains("Deutsch")) ";" else ","

                putString(SettingsActivity.KEY_PREF_DELIMITER, delimiter)
                apply()
            }
            putBoolean("encryptedAttachments", true)
        }
        checkBackup()

        // deprecated after API 30
        //if (Build.VERSION.SDK_INT < Build.VERSION_CODES.R)
        window.setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_ADJUST_RESIZE) // Required so Navbar and FAB move up when keyboard is visible

        // Migrate old templates, added with v3.0, delete after a few versions
        val oldTemplates = preferences.getString("etTextTemplates", "") + ""
        if (oldTemplates.isNotEmpty()) {
            val t = oldTemplates.split(",")
            val dt = viewModel.dataType // Pick first active tab, not ideal but hey
            t.forEach {
                textTemplatesViewModel.upsert(TextTemplates(0, dt, it))
            }
            preferences.edit { putString("etTextTemplates", "") }
        }

        // Check user feedback
       if (userFeedbackLevel > 0) {
            val lastRun = preferences.getLong("LAST_FEEDBACK_RUN", 0L)
            if (!DateUtils.isToday(lastRun)) userFeedback(viewModel, this) // Check if it ran already today
        }

        if ((applicationContext as MediLog).appStart == AppStart.FIRST_TIME_APP) setDefaults(this)

        // Todo: Does removing this break edgeToEdge?
        WindowCompat.setDecorFitsSystemWindows(window, true)
        setContent {
            StartCompose()
        }
    }


    private fun checkBackup() {
        // Is the DB empty? If yes, don't bother with backups
        // Consider everything below 5 entries to be simple tests
        val records = viewModel.count()
        if (records < 5) return

        // Did we check backup status already today? If yes, don't bother again
        if (DateUtils.isToday(preferences.getLong("LAST_BACKUP_CHECK", 0L))) return // Check if it ran already today

        var editor = preferences.edit()
        editor.putLong("LAST_BACKUP_CHECK", Date().time).apply()

        val lastBackup = preferences.getLong("LAST_BACKUP", 0L)
        val backupAgeMS = Date().time - lastBackup
        val backupAgeDays = TimeUnit.DAYS.convert(backupAgeMS, TimeUnit.MILLISECONDS)

        // Auto Backup
        // Check if autoBackup is enabled
        val autoBackup = preferences.getString(KEY_PREF_AUTO_BACKUP, getString(string.AUTO_BACKUP_DEFAULT))
        if (!autoBackup.isNullOrBlank()) {
            val autoBackupDays = try { autoBackup.toInt() } catch (_: NumberFormatException) { getString(string.AUTO_BACKUP_DEFAULT).toInt() }
            if (backupAgeDays >= autoBackupDays) {
                // Check if everything is in place for auto backups
                val uriString = preferences.getString(SettingsActivity.KEY_PREF_BACKUP_URI, "")
                if (uriString.isNullOrEmpty()) {
                    Toast.makeText(this, getString(string.missingBackupLocation), Toast.LENGTH_LONG).show()
                    return
                }

                // Valid location?
                val uri = uriString.toUri()
                val dFolder = DocumentFile.fromTreeUri(this, uri)
                if (dFolder == null) {
                    Toast.makeText(this, getString(string.invalidBackupLocation), Toast.LENGTH_LONG).show()
                    return
                }

                // Password
                var zipPassword = preferences.getString(SettingsActivity.KEY_PREF_PASSWORD, "")
                if (zipPassword.isNullOrEmpty()) zipPassword = ""

                writeBackup(uri, zipPassword, true)

                preferences.edit() { putLong("LAST_BACKUP_CHECK", Date().time) }
            }
        }
        else {
            // Backup warning, needs to go last so users don't get flooded with messages
            val backupWarningDays: Int
            // Check if backupWarning is enabled
            val backupWarning = preferences.getString(KEY_PREF_BACKUP_WARNING, getString(string.BACKUP_WARNING_DEFAULT))
            if (!backupWarning.isNullOrEmpty()) {
                backupWarningDays = try { backupWarning.toInt() } catch (_: NumberFormatException) { getString(string.BACKUP_WARNING_DEFAULT).toInt() }
                if (backupAgeDays >= backupWarningDays) {
                    val message = getString(string.backupOverdue) + ".  " + when (lastBackup) {
                        0L -> getString(string.backupOverdueMsg1, records)
                        else -> getString(string.backupOverdueMsg2, records, backupAgeDays)
                    }
                    Toast.makeText(this, message, Toast.LENGTH_LONG).show()
                }
            }
        }
    }

    fun writeBackup(uri: Uri, zipPassword: String, autoBackup: Boolean = false) {
        /*if (count() == 0) {
            Toast.makeText(app, app.getString(string.emptyDatabase), Toast.LENGTH_LONG).show()
            return
        }*/
         weightViewModel.viewModelScope.launch(Dispatchers.IO) {
            Backup(this@MainActivity,  application, uri, zipPassword).exportZIPFile(autoBackup)
        }
    }

    override fun onStart() {
        super.onStart()
        checkTimeout()
    }

    private fun checkTimeout() {
        if (!preferences.getBoolean(SettingsActivity.KEY_PREF_BIOMETRIC, false)) {
            medilogApplicationContext.authenticated = true
            authenticated.value = true
            return
        }

        val tmp = "" + preferences.getString(KEY_PREF_FORCE_REAUTHENTICATION, getString(string.FORCE_REAUTHENTICATION_DEFAULT))
        val threshold = 60000 * (tmp.toLongOrNull() ?: getString(string.FORCE_REAUTHENTICATION_DEFAULT).toLong())

        // Check timer
        val now = Calendar.getInstance().timeInMillis
        val then = preferences.getLong("STOPWATCH", 0L)
        val diff = now - then

        // First check if authentication is enabled
        // If yes, check if re-authentication needs to be forced = if more than threshold milliseconds passed after last onStop event was executed
        if (diff > threshold) {
            medilogApplicationContext.authenticated = false
            authenticated.value = false
        }

        val biometricHelper = BiometricHelper(this)
        val canAuthenticate = biometricHelper.canAuthenticate(true)
        if (!medilogApplicationContext.authenticated && canAuthenticate == 0) {
            // No idea why I need this
            val newExecutor: Executor = Executors.newSingleThreadExecutor()
            val activity: FragmentActivity = this
            val myBiometricPrompt = BiometricPrompt(activity, newExecutor, object : BiometricPrompt.AuthenticationCallback() {

                override fun onAuthenticationError(errorCode: Int, errString: CharSequence) {
                    super.onAuthenticationError(errorCode, errString)
                    Snackbar
                        .make(findViewById(android.R.id.content), getString(string.retryActionText), 30000)
                        .setAction(getString(string.retryAction)) { onStart() }
                        .setActionTextColor(RED)
                        .show()
                }

                override fun onAuthenticationSucceeded(result: BiometricPrompt.AuthenticationResult) {
                    super.onAuthenticationSucceeded(result)
                    medilogApplicationContext.authenticated = true
                    authenticated.value = true
                    //runOnUiThread { binding.viewPager.visibility = View.VISIBLE }
                }
            })
            // Todo: Fix deprecation
            if (Build.VERSION.SDK_INT > 29) {
                val promptInfo = PromptInfo.Builder()
                    .setTitle(getString(string.appName))
                    .setDeviceCredentialAllowed(true)  // Allow to use pin as well
                    .setSubtitle(getString(string.biometricLogin))
                    .setConfirmationRequired(false)
                    .build()
                myBiometricPrompt.authenticate(promptInfo)
            } else {
                // Pin fallback Disabled for now until bug https://issuetracker.google.com/issues/142740104 is fixed)
                val promptInfo = PromptInfo.Builder()
                    .setTitle(getString(string.appName))
                    .setNegativeButtonText(getString(string.cancel))
                    .setSubtitle(getString(string.biometricLogin))
                    .setConfirmationRequired(false)
                    .build()
                myBiometricPrompt.authenticate(promptInfo)
            }
        }
    }

    private val backupLauncher = registerForActivityResult(ActivityResultContracts.StartActivityForResult()) { result ->
        if (result.resultCode == RESULT_OK) {
            // There are no request codes
            val data: Intent? = result.data
            data ?: return@registerForActivityResult // No data = nothing to do, shouldn't happen

            var backupFolderUri: Uri? = null

            // Check Uri is sound
            try {
                backupFolderUri = data.data
            } catch (_: Exception) {
                Toast.makeText(this, this.getString(string.eSelectDirectory) + " " + data, Toast.LENGTH_LONG).show()
            }
            if (backupFolderUri == null) {
                Toast.makeText(this, this.getString(string.eSelectDirectory) + " " + data, Toast.LENGTH_LONG).show()
                return@registerForActivityResult
            }

            val zipPassword = preferences.getString(SettingsActivity.KEY_PREF_PASSWORD, "")
            if (zipPassword.isNullOrEmpty()) {
                val customDialog = Dialog(this)
                customDialog.setContentView(layout.input_dialog)
                customDialog.window?.setLayout(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.WRAP_CONTENT)
                val title: TextView = customDialog.findViewById(id.tvTitle)
                title.text = getString(string.enterPassword)
                val txtInput: TextView = customDialog.findViewById(id.etInput)
                val positiveButton: Button = customDialog.findViewById(id.btPositive)
                val negativeButton: Button = customDialog.findViewById(id.btNegative)
                positiveButton.text = getString(string.submit)
                negativeButton.text = getString(string.cancel)
                txtInput.hint = getString(string.password)
                txtInput.inputType = InputType.TYPE_CLASS_TEXT or TYPE_TEXT_VARIATION_PASSWORD
                positiveButton.setOnClickListener {
                    writeBackup(backupFolderUri, txtInput.text.toString())
                    customDialog.dismiss()
                }
                negativeButton.setOnClickListener {
                    customDialog.dismiss()
                }
                customDialog.show()
            } else writeBackup(backupFolderUri, zipPassword)
        }
    }


    private val restoreLauncher = registerForActivityResult(ActivityResultContracts.StartActivityForResult()) { result ->
        if (result.resultCode == RESULT_OK) {
            // There are no request codes
            val data: Intent? = result.data
            data ?: return@registerForActivityResult // No data = nothing to do, shouldn't happen

            var backupFileUri: Uri? = null
            try {
                backupFileUri = data.data
            } catch (_: Exception) {
                Toast.makeText(this, this.getString(string.eSelectDirectory) + " " + data, Toast.LENGTH_LONG).show()
            }

            backupFileUri ?: return@registerForActivityResult

            // Check if user needs to be warned
            val existingRecords = viewModel.count()
            if (existingRecords > 0) {
                MaterialAlertDialogBuilder(this)
                    .setTitle(resources.getString(string.warning))
                    .setMessage(resources.getString(string.thisIsGoing, existingRecords) + "\n" + getString(string.doYouReallyWantToContinue))
                    .setPositiveButton(resources.getString(string.yes)) { _, _ ->
                        launchRestoreIntent(backupFileUri)
                    }
                    .setNegativeButton(resources.getString(string.cancel)) { _, _ ->
                        // Respond to negative button press
                    }
                    .show()
            } else launchRestoreIntent(backupFileUri)
        }
    }


    private val importLauncher = registerForActivityResult(ActivityResultContracts.StartActivityForResult()) { result ->
        if (result.resultCode == RESULT_OK) {
            // There are no request codes
            val data: Intent? = result.data
            data ?: return@registerForActivityResult // No data = nothing to do, shouldn't happen

            var importFileUri: Uri? = null
            try {
                importFileUri = data.data
            } catch (_: Exception) {
                Toast.makeText(this, this.getString(string.eSelectDirectory) + " " + data, Toast.LENGTH_LONG).show()
            }

            importFileUri ?: return@registerForActivityResult
            launchRestoreIntent(importFileUri)
        }
    }

    private fun getPdfClass(dataType: Int): BasePdf? {
        val pdfClass = when (dataType) {
            BLOODPRESSURE -> BloodPressurePdf(bloodPressureViewModel, this@MainActivity)
            WEIGHT -> WeightPdf(weightViewModel, this@MainActivity)
            DIARY -> DiaryPdf(diaryViewModel, this@MainActivity)
            GLUCOSE -> GlucosePdf(glucoseViewModel, this@MainActivity)
            FLUID -> FluidPdf(fluidViewModel, tagsViewModel, this@MainActivity)
            TEMPERATURE -> TemperaturePdf(temperatureViewModel, this@MainActivity)
            OXIMETRY -> OximetryPdf(oximetryViewModel, this@MainActivity)
            else -> null
        }
        return pdfClass
    }

    private fun openFile(viewModel: DataViewModel, type: String) {
        if (viewModel.getSize(true) == 0) {
            Toast.makeText(this, getString(string.emptyTable) + " " + getString(string.pdfCantCreate), Toast.LENGTH_LONG).show()
            return
        }

        var uri: Uri? = null
        when (type) {
            IntentTypes.PDF -> {
                val pdfHandler = getPdfClass(viewModel.dataType)
                pdfHandler?.createPdfDocument()
                uri = pdfHandler?.savePdfDocument()
            }

            IntentTypes.CSV -> {
                uri = viewModel.writeCsvFile(true)
            }
        }

        if (uri != null) {
            val intent = Intent(ACTION_VIEW)
            //intent.type = type
            intent.setDataAndType(uri, type)
            intent.flags = FLAG_GRANT_READ_URI_PERMISSION
            val chooser = createChooser(intent, null)
            try {
                startActivity(chooser)
            } catch (e: ActivityNotFoundException) {
                Toast.makeText(this, getString(string.eShareError) + ": " + e.localizedMessage, Toast.LENGTH_LONG).show()
            }
        }
    }

    private fun sendPdfFile(viewModel: DataViewModel, type: String, zipPassword: String = "") {
        if (viewModel.getSize(true) == 0) {
            Toast.makeText(this, getString(string.emptyTable) + " " + getString(string.pdfCantCreate), Toast.LENGTH_LONG).show()
            return
        }

        val intent = Intent(ACTION_SEND)
        intent.type = type
        intent.flags = FLAG_GRANT_READ_URI_PERMISSION
        intent.putExtra(EXTRA_TEXT, getString(string.sendText))

        intent.putExtra(EXTRA_SUBJECT, "MediLog " + getString(string.data))
        var uri: Uri? = null
        val pdfHandler = getPdfClass(viewModel.dataType)
        val pdfDoc = pdfHandler?.createPdfDocument()
        when (type) {
            IntentTypes.ZIP -> uri = viewModel.getZIP(pdfDoc, zipPassword)
            IntentTypes.CSV -> uri = viewModel.writeCsvFile(true)
            IntentTypes.PDF -> uri = viewModel.savePdfDocument(pdfDoc)
        }

        if ((uri != null) && (packageManager.resolveActivity(intent, 0) != null)) {
            intent.putExtra(EXTRA_STREAM, uri)
            try {
                startActivity(createChooser(intent, getString(string.sendText)))
            } catch (e: Exception) {
                Toast.makeText(this, getString(string.eShareError) + ": " + e.localizedMessage, Toast.LENGTH_LONG).show()
            }
        }
    }

    private fun sendCsvFile(viewModel: DataViewModel) {
        val type = IntentTypes.CSV
        val intent = Intent(ACTION_SEND)
        intent.type = type
                    intent.flags = FLAG_GRANT_READ_URI_PERMISSION
        intent.putExtra(EXTRA_TEXT, getString(string.sendText))

        intent.putExtra(EXTRA_SUBJECT, "MediLog " + getString(string.data))
        var uri: Uri? = null
        when (type) {
            IntentTypes.CSV -> uri = viewModel.writeCsvFile(true)
        }

        if ((uri != null) && (packageManager.resolveActivity(intent, 0) != null)) {
            intent.putExtra(EXTRA_STREAM, uri)
            try {
                startActivity(createChooser(intent, getString(string.sendText)))
            } catch (e: Exception) {
                Toast.makeText(this, getString(string.eShareError) + ": " + e.localizedMessage, Toast.LENGTH_LONG).show()
            }
        }
    }

    private fun getSendZipPassword(viewModel: DataViewModel) {
        val customDialog = Dialog(this)
        customDialog.setContentView(layout.input_dialog)
        customDialog.window?.setLayout(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.WRAP_CONTENT)
        val title: TextView = customDialog.findViewById(id.tvTitle)
        title.text = getString(string.enterPassword)
        val txtInput: TextView = customDialog.findViewById(id.etInput)
        val positiveButton: Button = customDialog.findViewById(id.btPositive)
        val negativeButton: Button = customDialog.findViewById(id.btNegative)
        positiveButton.text = getString(string.submit)
        negativeButton.text = getString(string.cancel)
        txtInput.hint = getString(string.password)
        txtInput.inputType = InputType.TYPE_CLASS_TEXT or TYPE_TEXT_VARIATION_PASSWORD
        positiveButton.setOnClickListener {
            sendPdfFile(viewModel, IntentTypes.ZIP, txtInput.text.toString())
            customDialog.dismiss()
        }
        negativeButton.setOnClickListener {
            customDialog.dismiss()
        }
        customDialog.show()
    }


    private fun launchRestoreIntent(backupFileUri: Uri) {
        // Analyse file
        val cr = contentResolver

        when (val mimeType = cr.getType(backupFileUri)) {
            IntentTypes.ZIP -> {
                val zipPassword = preferences.getString(SettingsActivity.KEY_PREF_PASSWORD, "")
                tmpPin = preferences.getString(DATABASE_PIN_KEY, "") ?: "" // Keep the current pin so it survives a restore which might replace the pin
                if (zipPassword.isNullOrEmpty()) getOpenZipPassword(backupFileUri)
                else {
                    val intent = Intent(application, ImportActivity::class.java)
                    intent.putExtra("URI", backupFileUri.toString())
                    intent.putExtra("PWD", zipPassword)
                    startActivity(intent)
                }
            }

            "application/csv",
            "text/csv",
            "text/comma-separated-values" -> {
                val intent = Intent(application, ImportActivity::class.java)
                intent.putExtra("URI", backupFileUri.toString())
                intent.putExtra("PWD", "SIDELOAD")
                startActivity(intent)
            }

            else -> Toast.makeText(this, "Don't know how to handle file type: $mimeType", Toast.LENGTH_LONG).show()
        }
    }


    private fun getOpenZipPassword(zipUri: Uri) {
        val customDialog = Dialog(this)
        customDialog.setContentView(layout.input_dialog)
        customDialog.window?.setLayout(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.WRAP_CONTENT)
        val title: TextView = customDialog.findViewById(id.tvTitle)
        title.text = getString(string.enterPassword)
        val txtInput: TextView = customDialog.findViewById(id.etInput)
        val positiveButton: Button = customDialog.findViewById(id.btPositive)
        val negativeButton: Button = customDialog.findViewById(id.btNegative)
        positiveButton.text = getString(string.submit)
        negativeButton.text = getString(string.cancel)
        txtInput.hint = getString(string.password)
        txtInput.inputType = InputType.TYPE_CLASS_TEXT or TYPE_TEXT_VARIATION_PASSWORD
        positiveButton.setOnClickListener {
            val intent = Intent(application, ImportActivity::class.java)
            intent.putExtra("URI", zipUri.toString())
            intent.putExtra("PWD", txtInput.text.toString())
            //Toast.makeText(this, "0: Before startActivity", Toast.LENGTH_LONG).show()
            TimeUnit.SECONDS.sleep(2L)
            startActivity(intent)
            customDialog.dismiss()
        }
        negativeButton.setOnClickListener {
            customDialog.dismiss()
        }
        customDialog.show()
    }

// #####################################################
// Application independent code
// #####################################################
// https://dev.to/ahmmedrejowan/hide-the-soft-keyboard-and-remove-focus-from-edittext-in-android-ehp
override fun dispatchTouchEvent(event: MotionEvent): Boolean {
    if (event.action == MotionEvent.ACTION_DOWN) {
        val v: View? = currentFocus
        if (v is EditText) {
            val outRect = android.graphics.Rect()
            v.getGlobalVisibleRect(outRect)
            if (!outRect.contains(event.rawX.toInt(), event.rawY.toInt())) {
                v.clearFocus()
                val imm = this.getSystemService(INPUT_METHOD_SERVICE) as InputMethodManager?
                imm!!.hideSoftInputFromWindow(v.windowToken, 0)
            }
        }
    }
    return super.dispatchTouchEvent(event)
}

    public override fun onStop() {
        super.onStop()
        // Save active tab
        preferences.edit(commit = true) {
            putInt("crash_count", 0)
            putLong("STOPWATCH", Calendar.getInstance().timeInMillis) // Set Re-authentication timer
        }
    }

    // Scaffold and other UI stuff
    @OptIn(ExperimentalMaterial3Api::class)
    @Composable
    fun StartCompose() {
        var showOverflowMenu by remember { mutableStateOf(false) }
        val snackbarHostState = remember { SnackbarHostState() }
        val snackbarDelegate = SnackbarDelegate(snackbarHostState, rememberCoroutineScope())
        //var activeTabRoute = preferences.getString( ACTIVE_TAB_KEY, "weight_route") ?: "weight_route"
        var activeTabRoute = try { preferences.getString(ACTIVE_TAB_KEY, "weight_route") } catch (_: ClassCastException) { "weight_route" } ?: "weight_route"  // In case stored value is wrong type → overwrite it

        val allTabMeta = getAllTabMeta(this)

        val tabs: List<NavigationTab> = listOfNotNull(
            WeightTab(allTabMeta[WEIGHT], weightViewModel, tagsViewModel, this@MainActivity, snackbarDelegate).takeIf { WEIGHT in enabledTabsState.value },
            BloodPressureTab(allTabMeta[BLOODPRESSURE], bloodPressureViewModel, tagsViewModel, this@MainActivity, snackbarDelegate).takeIf { BLOODPRESSURE in enabledTabsState.value },
            DiaryTab(allTabMeta[DIARY], diaryViewModel, tagsViewModel, this@MainActivity, snackbarDelegate).takeIf { DIARY in enabledTabsState.value },
            FluidTab(allTabMeta[FLUID], fluidViewModel, tagsViewModel, this@MainActivity, snackbarDelegate).takeIf { FLUID in enabledTabsState.value },
            TemperatureTab(allTabMeta[TEMPERATURE], temperatureViewModel, tagsViewModel, this@MainActivity, snackbarDelegate).takeIf { TEMPERATURE in enabledTabsState.value },
            OximetryTab(allTabMeta[OXIMETRY], oximetryViewModel, tagsViewModel, this@MainActivity, snackbarDelegate).takeIf { OXIMETRY in enabledTabsState.value },
            GlucoseTab(allTabMeta[GLUCOSE], glucoseViewModel, tagsViewModel, this@MainActivity, snackbarDelegate).takeIf { GLUCOSE in enabledTabsState.value },
            DocumentsTab(allTabMeta[DOCUMENTS], documentsViewModel, tagsViewModel, this@MainActivity, snackbarDelegate).takeIf { DOCUMENTS in enabledTabsState.value },
        )
        if (activeTabRoute !in tabs.map { it.route }) {
            activeTabRoute = tabs.firstOrNull()?.route ?: Screens.Weight.route
        }

        // Check authentication status before showing stuff
        authenticated.value = medilogApplicationContext.authenticated
        val go = remember { authenticated }
        if (!go.value) {
            NotYetAuthorized()
            return
        }

        var searchBarOpen by remember { mutableStateOf(false) }

        val navController = rememberNavController()
        val navBackStackEntry by navController.currentBackStackEntryAsState()
        val currentRoute = navBackStackEntry?.destination?.route // Holds the current route, auto updates!
        val currentTab = tabs.find { it.route == currentRoute }
        val isDocuments = currentTab is DocumentsTab
        //(applicationContext as MediLog).appStart = AppStart.FIRST_TIME_APP
        MedilogTheme {
            if ((applicationContext as MediLog).appStart == AppStart.FIRST_TIME_APP) {
                Surface(modifier = Modifier.fillMaxSize().systemBarsPadding(), // <-- handles top/bottom insets
                    color = MaterialTheme.colorScheme.background) {
                    OnboardingPage(context = medilogApplicationContext, setValue = { (applicationContext as MediLog).appStart = it })
                }
            }
            else {
                var showChangelog by remember { mutableStateOf((applicationContext as MediLog).appStart == AppStart.FIRST_TIME_VERSION) }
                if (showChangelog) {
                    ChangelogDialog(onDismissRequest = {
                        showChangelog = false
                        (applicationContext as MediLog).appStart = AppStart.NORMAL
                    })
                }

                val controller = LocalSoftwareKeyboardController.current
                val header = medilogApplicationContext.getString(string.appName) + if (profilesViewModel.count() > 1L) " - " + (ActiveProfile.name) else ""
                val scrollBehavior = TopAppBarDefaults.enterAlwaysScrollBehavior()

                Scaffold(
                    modifier = Modifier.nestedScroll((scrollBehavior.nestedScrollConnection)),
                    snackbarHost = { SnackbarHost(snackbarHostState) },
                    topBar = {
                        if (currentRoute !in fullScreenRoutes) {
                        if (searchBarOpen)
                            MediLogSearchBar(currentTab?.viewModel!!, setShowDialog = { searchBarOpen = it }, application) {
                                if (it > 0) currentTab.startEditing(it)
                            }
                        else MediumTopAppBar(
                            title = { Text(header, maxLines = 1, // Prevents wrapping
                                            overflow = TextOverflow.Ellipsis) },
                            actions = {
                                IconButton(onClick = {
                                    val intent = Intent(this@MainActivity, FilterActivity::class.java)
                                    intent.putExtra(ACTIVE_TAB_KEY, currentTab?.viewModel?.dataType ?: -1) //viewModel.dataType)
                                    startActivity(intent)
                                }) { Icon(imageVector = if (currentTab?.viewModel?.filterActive() ?: false) Icons.Outlined.FilterList else Icons.Outlined.FilterListOff, contentDescription = "Filter") }
                                IconButton(onClick = { searchBarOpen = true }) { Icon(imageVector = Icons.Rounded.Search, contentDescription = "Search") }
                                //Spacer(modifier = Modifier.width(10.dp))
                                IconButton(enabled = !isDocuments, onClick = { currentTab?.showInfoScreen(this@MainActivity) }) { Icon(imageVector = Icons.Default.Info, contentDescription = "Info Screen") }
                                IconButton(enabled = !isDocuments, onClick = { currentTab?.showChartScreen(this@MainActivity) }) { Icon(imageVector = Icons.Default.AreaChart, contentDescription = "Chart Screen") }
                                IconButton(onClick = { showOverflowMenu = !showOverflowMenu }) { Icon(imageVector = Icons.Rounded.MoreVert, contentDescription = "Overflow") }
                            },
                            scrollBehavior = scrollBehavior,
                            modifier = Modifier.nestedScroll(scrollBehavior.nestedScrollConnection)
                        )
                        }
                    },
                    bottomBar = {
                        if (tabs.size > 1 && currentRoute !in fullScreenRoutes) {
                            NavigationBar {
                                tabs.forEach { tab ->
                                    NavigationBarItem(
                                        selected = tab.route == currentRoute,
                                        onClick = {
                                            if (tab.route != currentRoute) {
                                                navController.navigate(tab.route) { launchSingleTop = true }
                                                preferences.edit { putString(ACTIVE_TAB_KEY, tab.route) }
                                            }
                                        },
                                        icon = { Icon(painter = painterResource(if (tab.route == currentRoute) tab.iconActive else tab.iconInactive), contentDescription = tab.label) },
                                        label = { Text(tab.label ?: "") }
                                    )
                                }
                            }
                        }
                    },
                    floatingActionButton = {
                        if (currentRoute !in fullScreenRoutes) {
                        FloatingActionButton(
                            onClick = {
                                controller?.hide()
                                val currentTab = tabs.find { it.route == currentRoute } // find the active tab
                                currentTab?.addItem()
                            },
                            containerColor = BottomAppBarDefaults.bottomAppBarFabColor,
                            elevation = FloatingActionButtonDefaults.elevation()
                        ) {
                            Icon(Icons.Filled.Add, "Add item")
                        }
                    }
                   },
                )
                { paddingValues ->
                    NavHost(navController = navController, startDestination = activeTabRoute, modifier = Modifier.padding(paddingValues)) {
                        // Tab routes
                        tabs.forEach { tab ->
                            composable(tab.route) { tab.ShowContent(paddingValues) }
                        }

                        // One-off screens
                        composable(Screens.About.route) {
                            AboutScreen(onBack = { navController.popBackStack() })
                        }
                        composable(Screens.Profiles.route) {
                            ProfilesScreen(
                                onProfileActivated = {
                                    if (it != ActiveProfile.id) {
                                        PreferenceSync.switchProfile(this@MainActivity, ActiveProfile.id, it)
                                        // Restart MainActivity to make sure changes are picked up right away
                                        val intent = Intent(this@MainActivity, MainActivity::class.java)
                                        intent.addFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP or Intent.FLAG_ACTIVITY_NEW_TASK)
                                        startActivity(intent)
                                    }
                                    else navController.popBackStack() // Do nothing but return
                                    //profilesViewModel.setActiveProfile(it)
      //                              navController.popBackStack()
                                },
                                onBack = { navController.popBackStack() }
                            )
                        }
                    }
                }
                if (showOverflowMenu) OverflowMenu(navController = navController, onDismissRequest = { showOverflowMenu = false }, expanded = true, viewModel = currentTab!!.viewModel)
            }
        }
    }

    @Composable
    fun OverflowMenu(onDismissRequest: (Boolean) -> Unit, expanded: Boolean, navController: NavController, viewModel: DataViewModel) {
        var showUpgradeDialog by remember { mutableStateOf(false) }
        var productDetails by remember { mutableStateOf<UpgradeProductInfo?>(null) }

        var showCsvMenu by remember { mutableStateOf(false) }
        var showPdfMenu by remember { mutableStateOf(false) }
        var showDataMenu by remember { mutableStateOf(false) }
        var selectedMenu = ""
        val defaultBackground = MaterialTheme.colorScheme.surfaceContainer
        val selectedBackground = MaterialTheme.colorScheme.surfaceContainerHighest

        Box(modifier = Modifier.fillMaxWidth().wrapContentSize(Alignment.TopEnd)) {
            DropdownMenu(expanded = expanded, { onDismissRequest(false) }) {
                DropdownMenuItem(
                    text = { Text(this@MainActivity.getString(string.action_settings), style = TextStyle.Default) },
                    leadingIcon = { Icon(Icons.Outlined.Settings, contentDescription = null) },
                    onClick = {
                        val intent = Intent(this@MainActivity, SettingsActivity::class.java)
                        this@MainActivity.startActivity(intent)
                        onDismissRequest(false)
                    }
                )
                DropdownMenuItem(
                    modifier = Modifier.background(if(this@MainActivity.getString(string.actionPDF) == selectedMenu) selectedBackground else defaultBackground),
                    text = { Text(this@MainActivity.getString(string.actionPDF), style = TextStyle.Default) },
                    trailingIcon = { Icon(Icons.AutoMirrored.Filled.ArrowRight, contentDescription = null) },
                    //colors = MenuItemColors,
                    onClick = {
                        showPdfMenu = true
                        selectedMenu = this@MainActivity.getString(string.actionPDF)
                    }
                )
                DropdownMenuItem(
                    modifier = Modifier.background(if(this@MainActivity.getString(string.action_dataManagement) == selectedMenu) selectedBackground else defaultBackground),
                    text = { Text(this@MainActivity.getString(string.action_dataManagement), style = TextStyle.Default) },
                    trailingIcon = { Icon(Icons.AutoMirrored.Filled.ArrowRight, contentDescription = null) },
                    onClick = {
                        showDataMenu = true
                        selectedMenu = this@MainActivity.getString(string.action_dataManagement)
                    }
                )
                DropdownMenuItem(
                    text = { Text(this@MainActivity.getString(string.action_profiles), style = TextStyle.Default) },
                    leadingIcon = { Icon(Icons.Outlined.Person, contentDescription = null) },
                    onClick = {
                        navController.navigate(Screens.Profiles.route)
                        onDismissRequest(false)
                    }
                )
                if (flavour == AppFlavours.PLAY && !unlimitedRecords.value)
                    DropdownMenuItem(
                      text = { Text(stringResource(R.string.upgrade), style = TextStyle.Default) },
                        leadingIcon = { Icon(Icons.Outlined.Upgrade, contentDescription = null) },
                        onClick = {
                            billingManager.queryUpgradeProduct(this@MainActivity) { details ->
                                productDetails = details
                            }
                            showUpgradeDialog = true
                            //onDismissRequest(false)
                        }
                    )
                DropdownMenuItem(
                    text = { Text(this@MainActivity.getString(string.action_about), style = TextStyle.Default) },
                    leadingIcon = { Icon(Icons.Outlined.Info, contentDescription = null) },
                    onClick = {
                        navController.navigate(Screens.About.route)
                        onDismissRequest(false) // close the menu if needed
                    }
                )

                if (showCsvMenu) CsvMenu(onDismissRequest = {
                    showCsvMenu = false // Close submenu
                    onDismissRequest(false) // Close Overflow menu
                    }, expanded = true, viewModel = viewModel)
                if (showPdfMenu) PdfMenu(onDismissRequest = {
                    showPdfMenu = false // Close submenu
                    onDismissRequest(false) // Close Overflow menu
                    }, true, viewModel = viewModel)
                if (showDataMenu) DataMenu(onDismissRequest = {
                    showDataMenu = false // Close submenu
                    onDismissRequest(false) // Close Overflow menu
                    }, true)

                if (showUpgradeDialog) UnlimitedUpgradeDialog(
                        show = showUpgradeDialog,
                        productDetails = productDetails,
                        onPurchaseClick = { pd ->
                            billingManager.launchPurchaseFlow(this@MainActivity)
                            showUpgradeDialog = false
                        },
                        onRestoreClick = { checkLicense() },
                        onDismiss = { showUpgradeDialog = false }
                )
                if (showThankYouDialogState.value) ThankYouForPurchaseDialog(onDismiss = { showThankYouDialogState.value = false })
                if (showBillingUnavailableDialogState.value) BillingUnavailableDialog(onDismiss = { showBillingUnavailableDialogState.value = false })
            }
        }
    }

    @Composable
    fun CsvMenu(onDismissRequest: (Boolean) -> Unit, expanded: Boolean, viewModel: DataViewModel) {
        Box(modifier = Modifier.fillMaxWidth().wrapContentSize(Alignment.TopEnd)) {
            DropdownMenu(
                expanded = expanded,
                { onDismissRequest(false) },
            ) {
                DropdownMenuItem(
                    text = { Text(this@MainActivity.getString(string.actionOpenCSV), style = TextStyle.Default) },
                    leadingIcon = { Icon(Icons.Outlined.OpenInBrowser, contentDescription = null) },
                    onClick = {
                        //if (viewModel.getSize(true) == 0) Toast.makeText(this, getString(string.noDataToExport), Toast.LENGTH_LONG).show()
                        openFile(viewModel,IntentTypes.CSV)
                        onDismissRequest(false)
                    }
                )
                DropdownMenuItem(
                    text = { Text(this@MainActivity.getString(string.actionSendCSV), style = TextStyle.Default) },
                    leadingIcon = { Icon(Icons.AutoMirrored.Outlined.Send, contentDescription = null) },
                    onClick = {
                        //if (viewModels[binding.tabLayout.selectedTabPosition].getSize(true) == 0) Toast.makeText(this, getString(string.noDataToExport), Toast.LENGTH_LONG).show()
                        sendCsvFile(viewModel)
                        onDismissRequest(false)
                    }
                )
            }
        }
    }

    @Composable
    fun PdfMenu(onDismissRequest: (Boolean) -> Unit, expanded: Boolean, viewModel: DataViewModel) {
        Box(modifier = Modifier.fillMaxWidth().wrapContentSize(Alignment.TopEnd)) {
            DropdownMenu(expanded = expanded, { onDismissRequest(false) }) {
                DropdownMenuItem(
                    text = { Text(this@MainActivity.getString(string.actionOpenPDF), style = TextStyle.Default) },
                    leadingIcon = { Icon(Icons.Outlined.OpenInBrowser, contentDescription = null) },
                    onClick = {
                        //if (viewModels[binding.tabLayout.selectedTabPosition].getSize(true) == 0) Toast.makeText(this, getString(string.noDataToExport), Toast.LENGTH_LONG).show()
                        openFile(viewModel, IntentTypes.PDF)
                        onDismissRequest(false)
                    }
                )
                DropdownMenuItem(
                    text = { Text(this@MainActivity.getString(string.actionSendPDF), style = TextStyle.Default) },
                    leadingIcon = { Icon(Icons.AutoMirrored.Outlined.Send, contentDescription = null) },
                    onClick = {
                        //if (viewModels[binding.tabLayout.selectedTabPosition].getSize(true) == 0) Toast.makeText(this, getString(string.noDataToExport), Toast.LENGTH_LONG).show()
                        sendPdfFile(viewModel, IntentTypes.PDF)
                        onDismissRequest(false)
                    }
                )
                DropdownMenuItem(
                    text = { Text(this@MainActivity.getString(string.actionSendZIP), style = TextStyle.Default) },
                    leadingIcon = { Icon(Icons.Outlined.FolderZip, contentDescription = null) },
                    onClick = {
                        //if (viewModels[binding.tabLayout.selectedTabPosition].getSize(true) == 0) Toast.makeText(this, getString(string.noDataToExport), Toast.LENGTH_LONG).show()
                         val zipPassword = preferences.getString(SettingsActivity.KEY_PREF_PASSWORD, "")
                         if (zipPassword.isNullOrEmpty()) getSendZipPassword(viewModel)
                         else sendPdfFile(viewModel, IntentTypes.ZIP, zipPassword)
                         onDismissRequest(false)
                    }
                )
            }
        }
    }

    @Composable
    fun DataMenu(
        onDismissRequest: (Boolean) -> Unit,
        expanded: Boolean
    ) {
        var showSendDbDialog by remember { mutableStateOf(false) }
        var showReceiveDbDialog by remember { mutableStateOf(false) }
        val debugMode by DebugLog.debugMode.collectAsState(initial = false)

        Box(
            modifier = Modifier
                .fillMaxWidth()
                .wrapContentSize(Alignment.TopEnd)
        ) {
            DropdownMenu(
                expanded = expanded,
                { onDismissRequest(false) }
            ) {
                DropdownMenuItem(
                    text = { Text(this@MainActivity.getString(string.action_restore), style = TextStyle.Default) },
                    leadingIcon = { Icon(Icons.Outlined.OpenInBrowser, contentDescription = null) },
                    onClick = {
                        val intent = Intent(ACTION_GET_CONTENT)
                        intent.type = "application/zip"
                        restoreLauncher.launch(createChooser(intent, getString(string.selectFile)))
                        onDismissRequest(false)
                    }
                )
                DropdownMenuItem(
                    text = { Text(this@MainActivity.getString(string.action_backup), style = TextStyle.Default) },
                    leadingIcon = { Icon(Icons.AutoMirrored.Outlined.Send, contentDescription = null) },
                    onClick = {
                        if (viewModel.count() == 0) {
                            //Toast.makeText(this, getString(string.noDataToExport), Toast.LENGTH_LONG).show()
                            onDismissRequest(false)
                        }

                        // Pick backup folder
                        val intent = Intent(ACTION_OPEN_DOCUMENT_TREE)
                        intent.addCategory(CATEGORY_DEFAULT)
                        intent.addFlags(FLAG_GRANT_PERSISTABLE_URI_PERMISSION)
                        intent.addFlags(FLAG_GRANT_READ_URI_PERMISSION)
                        intent.addFlags(FLAG_GRANT_WRITE_URI_PERMISSION)

                        backupLauncher.launch(createChooser(intent, getString(string.selectDirectory)))
                        onDismissRequest(false)
                    }
                )
                DropdownMenuItem(
                    text = { Text(this@MainActivity.getString(string.action_import), style = TextStyle.Default) },
                    leadingIcon = { Icon(Icons.Outlined.FolderZip, contentDescription = null) },
                    onClick = {
                        // Pick import folder
                        val intent = Intent(ACTION_GET_CONTENT)
                        intent.type = "text/*"
                        importLauncher.launch(createChooser(intent, getString(string.selectFile)))
                        onDismissRequest(false)
                    }
                )
                // Only show if debug mode is enabled
                if (debugMode)  {
                    HorizontalDivider(thickness = 1.dp)
                    DropdownMenuItem(
                        text = { Text(this@MainActivity.getString(string.action_send_db), style = TextStyle.Default) },
                        leadingIcon = { Icon(Icons.Outlined.Download, contentDescription = null) },
                        onClick = { showSendDbDialog =true }
                    )
                    DropdownMenuItem(
                        text = { Text(this@MainActivity.getString(string.action_receive_db), style = TextStyle.Default) },
                        leadingIcon = { Icon(Icons.Outlined.Upload, contentDescription = null) },
                        onClick = { showReceiveDbDialog =true }
                    )
                }
            }

            if (showSendDbDialog) {
                DatabaseShareDialog(
                    dbDir = this@MainActivity.filesDir ?: File(""),
                    onDismiss = { showSendDbDialog = false }
                )
            }
            if (showReceiveDbDialog) {
                DatabaseImportDialog(
                    databaseName = databaseName,
                    onDismissRequest = { showReceiveDbDialog = false },
                    onRefreshActivity = {
                        // This will restart the activity, forcing UI/data reload
                        // After successful restore and dialog OK:
                            this@MainActivity.let { activity ->
                                // Restart activity to force complete reload
                                val intent = activity.intent
                                activity.finish()
                                activity.startActivity(intent)
                            }
                        }
                )
            }
        }
    }

    fun checkLicense() {
        if (resources.getInteger(integer.flavour) == AppFlavours.FOSS) {
            unlimitedRecords.value = true // Always true for FOSS
            preferences.edit { putBoolean(UnlimitedRecords, true) }
            return
        }
        unlimitedRecords.value  = preferences.getBoolean(UnlimitedRecords, false) // Check if upgrade has been purchased already
        if (!unlimitedRecords.value) { // Never upgraded or a new install
            billingManager.isLicensedUser(this) // This function will update unlimitedRecords.value asynchronously if user is licensed
        }
    }

//*****************************************************************************
    companion object {
        lateinit var medilogApplicationContext: MediLog
        var authenticated = mutableStateOf(false)
        var flavour = 0

        lateinit var feedbackString: MutableState<String>  // Global variable so we can set it in multiple places

        //const val DEBUGTAG: String = "MediLog-Debug-Tag"
        //const val MEDILOG_ERROR: String = "MediLog-Debug-Tag"

        object DatePattern {
            const val DATE_TIME = "yyyy-MM-dd HH:mm:ss"
            const val DATE = "yyyy-MM-dd"
        }

        object Delimiter {
            const val THRESHOLD = "-"
            const val STRING = '"'
            const val TAG = ":"
        }

        var tmpPin = "" // Used to handle db pin changes during imports

        const val NOT_SET = -1
        val AMBER = Color(0xFFFF8500) //Color(0xFFADFF01)
        const val decimalSeparators = ".," // Allow both , and . as input
        const val DAY_FORMAT = "yyyy-MM-dd"

        object TextSize {
            const val MIN = "10"
            const val MAX = "40"
        }

        // Constants to check against if an entry is valid

        data class TableFields(val n: String) {
            var name = n
            var pos = NOT_SET
        }

        // Initialise table vars
        val dataFields = arrayListOf(
            TableFields("timestamp"),
            TableFields("profile_id"),
            TableFields("tags"),
            TableFields("category_id"), // Can go once DB schema is updated
            TableFields("type"),
            TableFields("value1"),
            TableFields("value2"),
            TableFields("value3"),
            TableFields("value4"),
            TableFields("comment"),
            TableFields("attachment")
        )

        val templateFields = arrayListOf(
            TableFields("_id"),
            TableFields("type"),
            TableFields("template")
        )

        val tagFields = arrayListOf(
            TableFields("_id"),
            TableFields("type"),
            TableFields( "tag"),
            TableFields(    "color")
        )

        val profileFields = arrayListOf(
            TableFields("_id"),
            TableFields("name"),
            TableFields("description")
        )

        val settingsFields = arrayListOf(
            TableFields("profile_id"),
            TableFields("_key"),
            TableFields("value")
        )

        // Retrieve decimal separator for current language
        val decimalSeparator = DecimalFormatSymbols.getInstance().decimalSeparator
        val modifyDecimalSeparator = (decimalSeparator.compareTo('.') != 0)

 //     // Initialize, real value is set in mainActivity
        var userFeedbackLevel = 0
        //var debugMode = false
        //var dynamicColorOn = false
        var dynamicColorOn by mutableStateOf(false)
        var fontSize by mutableIntStateOf(12)


    fun resetReAuthenticationTimer(context: Context?) {
            if (context == null) return

            // Reset Stopwatch, after all we are still in MediLog
            val preferences = PreferenceManager.getDefaultSharedPreferences(context)

            preferences.edit {
                putLong("STOPWATCH", Calendar.getInstance().timeInMillis) // Set Re-authentication timer
            }
        }

        fun userFeedback(viewModel: DataViewModel, context: Context?):String {
            if (context == null) return ""

            val versionCode = getVersionCode(context)

            val delimiter = ":"
            var tabs = "" //delimiter
            for (tab in getAllTabMeta(context)) {
                tabs += viewModel.getSize(tab.key).toString() + delimiter
            }
            val sql = "SELECT count(*) FROM data WHERE attachment != ''"
            val attachments = viewModel.getInt(sql)

            val preferences = PreferenceManager.getDefaultSharedPreferences(context)
            val devID = preferences.getString("DEVICE_ID", randomUUID().toString())
            val data = "uuid=" + devID + delimiter + "build=" + versionCode + delimiter + "tabs=" + tabs + attachments //tabs.substring(0, tabs.length-1) // Cut off trailing delimiter

            val reqParam = URLEncoder.encode(data, "UTF-8")
            val mURL = URL("https://medilog.zell-mbc.com") //context.getString(string.userFeedbackUrl)) // + reqParam)

            thread {
                var localFeedbackString: String
                val https = mURL.openConnection() as HttpURLConnection
                https.requestMethod = "POST"
                https.doOutput = true

                val editor = preferences.edit()
                try {
                    val wr = OutputStreamWriter(https.outputStream)
                    wr.write(reqParam)
                    wr.flush()
                    val ret = https.responseCode // Required to close the POST request
                    editor.putString("DEVICE_ID", devID)
                    editor.putLong("LAST_FEEDBACK_RUN", Date().time)
                    localFeedbackString = if (ret == 200) {
                        editor.putString("USER_FEEDBACK_STRING", data)
                        data
                    } else {
                        // Save error message
                        editor.putString("USER_FEEDBACK_STRING", "Error: $ret")
                        ret.toString()
                    }
                  wr.close()
                }
                catch (e: Exception) {
                    // Save error message
                    editor.putString("USER_FEEDBACK_STRING", "Exception: $e")
                    localFeedbackString = e.toString()
                    //Log.d("Return: Exception", e.toString())
                }
                editor.apply()
                if (::feedbackString.isInitialized) feedbackString.value = localFeedbackString // Only initialized if Feedback UI is shown, ignore otherwise
            }
            return data
        }

        fun isDarkThemeOn(context: Context) = when (context.resources.configuration.uiMode and
            Configuration.UI_MODE_NIGHT_MASK) {
            Configuration.UI_MODE_NIGHT_YES -> true
            Configuration.UI_MODE_NIGHT_NO -> false
            Configuration.UI_MODE_NIGHT_UNDEFINED -> false
            else -> false
        }
    }
}

