/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.temperature

import android.os.Bundle
import android.widget.Toast
import androidx.activity.viewModels
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.safeDrawingPadding
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import com.patrykandpatrick.vico.compose.cartesian.CartesianChartHost
import com.patrykandpatrick.vico.compose.cartesian.axis.rememberBottom
import com.patrykandpatrick.vico.compose.cartesian.axis.rememberStart
import com.patrykandpatrick.vico.compose.cartesian.layer.rememberLine
import com.patrykandpatrick.vico.compose.cartesian.layer.rememberLineCartesianLayer
import com.patrykandpatrick.vico.compose.cartesian.rememberCartesianChart
import com.patrykandpatrick.vico.compose.cartesian.rememberVicoZoomState
import com.patrykandpatrick.vico.compose.common.ProvideVicoTheme
import com.patrykandpatrick.vico.compose.common.fill
import com.patrykandpatrick.vico.compose.m3.common.rememberM3VicoTheme
import com.patrykandpatrick.vico.core.cartesian.Zoom
import com.patrykandpatrick.vico.core.cartesian.axis.HorizontalAxis
import com.patrykandpatrick.vico.core.cartesian.axis.VerticalAxis
import com.patrykandpatrick.vico.core.cartesian.data.CartesianChartModelProducer
import com.patrykandpatrick.vico.core.cartesian.data.CartesianLayerRangeProvider
import com.patrykandpatrick.vico.core.cartesian.data.lineSeries
import com.patrykandpatrick.vico.core.cartesian.layer.LineCartesianLayer
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.R.string
import com.zell_mbc.medilog.base.ChartActivity
import com.zell_mbc.medilog.base.rememberMarker
import com.zell_mbc.medilog.preferences.SettingsActivity
import com.zell_mbc.medilog.preferences.SettingsActivity.Companion.KEY_PREF_TEMPERATURE_CUSTOM_ICON
import com.zell_mbc.medilog.support.checkThresholds
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import kotlin.getValue

class TemperatureChartActivity: ChartActivity() {
    // Activities need to be self sufficient because MainActivity might get killed by OS
    private val viewModel: TemperatureViewModel by viewModels()
    override val filename = "TemperatureChart.jpg"

    private var upperThresholdLine = 0f
    private var lowerThresholdLine = 0f

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        supportActionBar?.setIcon(preferences.getInt(KEY_PREF_TEMPERATURE_CUSTOM_ICON,R.drawable.ic_outline_device_thermostat_24))

        showGrid = preferences.getBoolean(SettingsActivity.KEY_PREF_TEMPERATURE_SHOW_GRID, true)
        showLegend = preferences.getBoolean(SettingsActivity.KEY_PREF_TEMPERATURE_SHOW_LEGEND, false)
        showValues = preferences.getBoolean(SettingsActivity.KEY_PREF_TEMPERATURE_SHOW_VALUES, true)
        showMovingAverage = preferences.getBoolean(SettingsActivity.KEY_PREF_TEMPERATURE_SHOW_MOVING_AVERAGE, false)
        showThreshold = preferences.getBoolean(SettingsActivity.KEY_PREF_TEMPERATURE_SHOW_THRESHOLDS, false)
        val itemUnit = "" + preferences.getString(SettingsActivity.KEY_PREF_TEMPERATURE_UNIT, getString(string.TEMPERATURE_CELSIUS))
        if (showThreshold) {
            val defaultThresholds = if (itemUnit == getString(string.TEMPERATURE_CELSIUS)) getString(string.TEMPERATURE_THRESHOLDS_CELSIUS_DEFAULT) else getString(string.TEMPERATURE_THRESHOLDS_FAHRENHEIT_DEFAULT)
            val s = "" + preferences.getString(SettingsActivity.KEY_PREF_TEMPERATURE_THRESHOLDS, defaultThresholds)
            val th = checkThresholds(this, s, defaultThresholds, getString(string.temperature))
            lowerThresholdLine = try { th[0].toFloat() } catch  (_: NumberFormatException) { 0f }
            upperThresholdLine = try { th[1].toFloat() } catch  (_: NumberFormatException) { 0f }
        }

        // For the charts only native diary entries can be considered, no matter what showAllTabs is set to
        val tmp = viewModel.blendInItems
        viewModel.blendInItems = false
        items = viewModel.getItems("ASC", filtered = true)
        viewModel.blendInItems = tmp

        if (items.size < 2) {
            Toast.makeText(this, getString(string.notEnoughDataForChart), Toast.LENGTH_LONG).show()
            finish()
            return // Needs return to avoid the rest to be executed because finish() may not kill the activity fast enough
        }
        if (!(showValues || showMovingAverage)) {
            Toast.makeText(this, getString(string.noValuesActivated), Toast.LENGTH_LONG).show()
            finish()
            return // Needs return to avoid the rest to be executed because finish() may not kill the activity fast enough
        }
        // Prepare legend
        if (showValues) legendList.add(legendItem(label = getString(string.temperature), color = lineColor1))

        // Collect list values and hold min/max values
        var fTmp: Float
        var i = 0
        for (item in items) {
            timestamps.add(i++)
            fTmp = try { item.value1.toFloat() } catch (_: NumberFormatException) { 0f }
            value1List.add(fTmp)
            if (fTmp > chartMax) chartMax = fTmp
            if (fTmp < chartMin) chartMin = fTmp // Not needed because value2Min will be smaller
        }
        // Add a bit of top and bottom filling
        val buffer = if (itemUnit == getString(string.TEMPERATURE_CELSIUS)) 1 else 2
        chartMin = ((chartMin.toInt()).toFloat())- buffer
        chartMax += buffer

        // Compile moving average values
        if (showMovingAverage) {
            legendList.add(legendItem(label = getString(string.showMovingAverage), color = movingAverageColor))
            val p = preferences.getString(SettingsActivity.KEY_PREF_TEMPERATURE_MOVING_AVERAGE_SIZE, "6")
            if (p != null) {
                val period = try { p.toInt() } catch (_: NumberFormatException) { 0 }
                movingAverageValue1List = getMovingAverageFloat(value1List, period)
            }
        }
        // Compile linear trendline values
        if (showLinearTrendline) {
            legendList.add(legendItem(label = getString(string.trendline), color = linearTrendlineColor))
            linearTrendlineValue1List = getLinearTrendlineFloat(value1List)
        }

        // Fill data maps
        if (showValues) value1Map = items.associateBy({ scaledTimeStamp(it.timestamp) }, { try { it.value1.toFloat() } catch (_: NumberFormatException) { 0f } })
        if (showLinearTrendline) {
            var ii = 0
            linearTrendlineValue1Map = items.associateBy({ scaledTimeStamp(it.timestamp) }, { try { linearTrendlineValue1List[ii++] } catch (_: NumberFormatException) { 0F } })
        }
        if (showMovingAverage) {
            var ii = 0
            movingAverageValue1Map = items.associateBy({ scaledTimeStamp(it.timestamp) }, { try { movingAverageValue1List[ii++] } catch (_: NumberFormatException) { 0F }})
        }

        showContent()
    }

    @Composable
    override fun ShowContent() {
        Box(Modifier.safeDrawingPadding()) {
            chartColors = mutableListOf(Color(0xffb983ff))

            val modelProducer = remember { CartesianChartModelProducer() }
            LaunchedEffect(Unit) {
                withContext(Dispatchers.Default) {
                    modelProducer.runTransaction {
                        lineSeries {
                            if (showValues) {
                                value1Map?.let { series(it.keys, it.values) }
                                value2Map?.let { series(it.keys, it.values) }
                                value3Map?.let { series(it.keys, it.values) }
                            }
                            if (showMovingAverage) {
                                movingAverageValue1Map?.let { series(it.keys, it.values) }
                                movingAverageValue2Map?.let { series(it.keys, it.values) }
                                movingAverageValue3Map?.let { series(it.keys, it.values) }
                            }
                            if (showLinearTrendline) {
                                linearTrendlineValue1Map?.let { series(it.keys, it.values) }
                                linearTrendlineValue2Map?.let { series(it.keys, it.values) }
                                linearTrendlineValue3Map?.let { series(it.keys, it.values) }
                            }
                        }
                    }
                }
            }
            ProvideVicoTheme(rememberM3VicoTheme()) {
                ComposeChart(modelProducer, Modifier.fillMaxHeight())
            }
        }
    }

    @Composable
    private fun ComposeChart(modelProducer: CartesianChartModelProducer, modifier: Modifier) {
        val marker = rememberMarker()
        val lineProvider = LineCartesianLayer.LineProvider.series(
            buildList {
                if (showValues) add(LineCartesianLayer.rememberLine(areaFill = LineCartesianLayer.AreaFill.single(fill = fill(fillColor1)), fill = LineCartesianLayer.LineFill.single(fill = fill(lineColor1))))
                if (showMovingAverage) add(LineCartesianLayer.rememberLine(fill = LineCartesianLayer.LineFill.single(fill(movingAverageColor)), areaFill = null))
                if (showLinearTrendline) add(LineCartesianLayer.rememberLine(stroke = linearTrendLineStroke, fill = LineCartesianLayer.LineFill.single(fill(linearTrendlineColor)), areaFill = null))
            }
        )
        val span = chartMax - chartMin
        val step = findNiceStep(span.toDouble())

        CartesianChartHost(
            chart = rememberCartesianChart(
                rememberLineCartesianLayer(lineProvider = lineProvider, rangeProvider = CartesianLayerRangeProvider.fixed(minY = chartMin.toDouble(), maxY = chartMax.toDouble())),
                startAxis =  VerticalAxis.rememberStart(
                    guideline = guideline(), // horizontal lines
                    itemPlacer = remember { VerticalAxis.ItemPlacer.step(step = { step }, shiftTopLines = false) }
                ),
                bottomAxis = HorizontalAxis.rememberBottom(
                    valueFormatter = bottomAxisValueFormatter,
                    guideline = guideline(),
                ),
                legend = if (showLegend) rememberLegend() else null,
                decorations = if (showThreshold) listOf(helperLine("", upperThresholdLine.toDouble(), lineColor1), helperLine("", lowerThresholdLine.toDouble(), lineColor1)) else listOf(),
                marker = marker,
            ),
            modelProducer = modelProducer,
            modifier = modifier,
            zoomState = rememberVicoZoomState(zoomEnabled = true, maxZoom = Zoom.max(Zoom.fixed(100f), Zoom.Content), initialZoom = Zoom.min(Zoom.fixed(), Zoom.Content)),
        )
    }
}
