/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.screens

import androidx.compose.foundation.text.selection.SelectionContainer

import android.content.ClipData
import android.content.ClipboardManager
import android.content.Context
import android.content.Intent
import android.widget.Toast
import androidx.compose.foundation.Image
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.ContentCopy
import androidx.compose.material.icons.filled.OpenInBrowser
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import com.zell_mbc.medilog.R
import androidx.core.net.toUri
import com.zell_mbc.medilog.ProjectDetails
import com.zell_mbc.medilog.about.ShowContent

// --- Helper Functions ---

fun openUrl(context: Context, url: String) {
    try {
        val intent = Intent(Intent.ACTION_VIEW, url.toUri())
        context.startActivity(intent)
    } catch (e: Exception) {
        Toast.makeText(context, "Cannot open link.", Toast.LENGTH_SHORT).show()
    }
}

fun copyToClipboard(context: Context, clipboardManager: ClipboardManager, label: String, text: String) {
    val clip = ClipData.newPlainText(label, text)
    clipboardManager.setPrimaryClip(clip)
    Toast.makeText(context, "$label copied to clipboard!", Toast.LENGTH_SHORT).show()
}

// --- Main Composable Screen ---
@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun DonationScreen(onBack: () -> Unit) {
    val context = LocalContext.current
    // Get the system's ClipboardManager service
    val clipboardManager = context.getSystemService(Context.CLIPBOARD_SERVICE) as ClipboardManager

    Scaffold(
        topBar = {
            TopAppBar(title = { Text(stringResource(R.string.supportOurWork)) },
                navigationIcon = { IconButton(onClick = onBack) { Icon(Icons.AutoMirrored.Filled.ArrowBack, contentDescription = "Back") } },
            )
        }
    ) { paddingValues ->
        Column(
            modifier = Modifier.padding(paddingValues).fillMaxSize().verticalScroll(rememberScrollState()).padding(16.dp),
            horizontalAlignment = Alignment.CenterHorizontally
        ) {
            Text(
                text = stringResource(R.string.donationThankYou),
                style = MaterialTheme.typography.bodyLarge, color = MaterialTheme.colorScheme.onSurfaceVariant, modifier = Modifier.padding(bottom = 24.dp)
            )

            // --- Bank Account Donation ---
            DonationSectionCard(title = stringResource(R.string.bankTransfer)) {
                // Pre-generated QR Code Image
                Image(
                    painter = painterResource(id = R.drawable.ic_sepa_qr_medilog),
                    contentDescription = "SEPA QR Code for Donation",
                    modifier = Modifier.size(200.dp).align(Alignment.CenterHorizontally).padding(bottom = 16.dp))

                val epcLink = "bank://singlepaymentsepa?IBAN=DE04500105170098703540&Name=Thomas Zell&Amount=10.00&Currency=EUR&Purpose=MediLog"
                DonationLinkCard(
                    title = "European (EPC) payment link",
                    url = epcLink,
                    logoContent = { Icon(painter = painterResource(id = R.drawable.ic_epc_logo), contentDescription = "EPC Logo", modifier = Modifier.size(28.dp), tint = Color.Unspecified)                    }
                )

                // Bank Details
                Spacer(modifier = Modifier.height(16.dp))
                HorizontalDivider(color = MaterialTheme.colorScheme.secondaryContainer, thickness = 1.dp)
                Spacer(modifier = Modifier.height(16.dp))
                Text("Bank Details:")
                DonationDetailRow(label = "IBAN", value = ProjectDetails.iban) {
                    copyToClipboard(context, clipboardManager, "IBAN", ProjectDetails.iban)
                }
                Spacer(modifier = Modifier.height(8.dp))
                DonationDetailRow(label = "BIC", value = ProjectDetails.bic) {
                    copyToClipboard(context, clipboardManager, "BIC", ProjectDetails.bic)
                }
                Spacer(modifier = Modifier.height(8.dp))
                DonationDetailRow(label = stringResource(R.string.accountHolder), value = ProjectDetails.accountHolder) {
                    copyToClipboard(context, clipboardManager, "Account Holder", ProjectDetails.accountHolder)
                }
            }

            Spacer(modifier = Modifier.height(24.dp))

            // --- Online Services ---
            DonationSectionCard(title = stringResource(R.string.onlineServices)) {
                DonationLinkCard(
                    title = "Liberapay",
                    url = ProjectDetails.liberapayUrl,
                    logoContent = { Icon(painter = painterResource(id = R.drawable.ic_liberapay), contentDescription = "Liberapay Logo", modifier = Modifier.size(28.dp), tint = Color.Unspecified)                    }
                )
                Spacer(modifier = Modifier.height(12.dp))
                DonationLinkCard(
                    title = "PayPal",
                    url = ProjectDetails.payPalUrl,
                    logoContent = { Icon(painter = painterResource(id = R.drawable.ic_paypal_icon), contentDescription = "PayPal Logo", modifier = Modifier.size(28.dp), tint = Color.Unspecified)                    }
                )
            }

            Spacer(modifier = Modifier.height(24.dp))

            // --- Cryptocurrency ---
            DonationSectionCard(title = "Cryptocurrency") {
                DonationLinkCard(title = "Bitcoin", url = ProjectDetails.bitcoinAddressUrl,
                    logoContent = { Icon(painter = painterResource(id = R.drawable.ic_bitcoin), contentDescription = "Bitcoin Logo", modifier = Modifier.size(28.dp), tint = Color.Unspecified)                    }
                )
             }
        }
    }
}

// --- Reusable Composable Elements ---

@Composable
fun DonationSectionCard(title: String, content: @Composable ColumnScope.() -> Unit) {
    Card(
        modifier = Modifier.fillMaxWidth(),
        colors = CardDefaults.cardColors(
            containerColor = MaterialTheme.colorScheme.surfaceContainerHigh // Use a lighter shade for sections
        )
    ) {
        Column(modifier = Modifier.padding(16.dp)) {
            Text(
                text = title, style = MaterialTheme.typography.titleLarge, fontWeight = FontWeight.Bold,
                color = MaterialTheme.colorScheme.onSurface, modifier = Modifier.padding(bottom = 12.dp)
            )
            content()
        }
    }
}

@Composable
fun DonationDetailRow(label: String, value: String, onCopyClick: () -> Unit) {
    Row(modifier = Modifier.fillMaxWidth(), horizontalArrangement = Arrangement.SpaceBetween, verticalAlignment = Alignment.CenterVertically) {
        Column(modifier = Modifier.weight(1f)) {
            Text(label, style = MaterialTheme.typography.bodySmall, color = MaterialTheme.colorScheme.onSurfaceVariant)
            SelectionContainer { // Allows long-press to select text
                Text(value, style = MaterialTheme.typography.bodyMedium, fontWeight = FontWeight.Medium)
            }
        }
        IconButton(onClick = onCopyClick) {
            Icon(
                imageVector = Icons.Default.ContentCopy,
                contentDescription = "Copy $label",
                tint = MaterialTheme.colorScheme.primary
            )
        }
    }
}

@Composable
fun DonationLinkCard(title: String, url: String, logoContent: @Composable () -> Unit) {
    val context = LocalContext.current
    Card(
        modifier = Modifier.fillMaxWidth().clickable { openUrl(context, url) },
        colors = CardDefaults.cardColors(containerColor = MaterialTheme.colorScheme.surface),
        elevation = CardDefaults.cardElevation(defaultElevation = 1.dp)
    ) {
        Row(modifier = Modifier.fillMaxWidth().padding(16.dp), verticalAlignment = Alignment.CenterVertically, horizontalArrangement = Arrangement.SpaceBetween) {
            Row(verticalAlignment = Alignment.CenterVertically) {
                // Logo placeholder: You can replace this with Coil/Glide image or a vector drawable
                Box(
                    modifier = Modifier.size(40.dp),
                    contentAlignment = Alignment.Center
                ) {
                    logoContent()
                }
                Spacer(modifier = Modifier.width(16.dp))
                Text(text = title, style = MaterialTheme.typography.titleMedium, fontWeight = FontWeight.SemiBold)
            }
            Icon(imageVector = Icons.Default.OpenInBrowser, contentDescription = "Open $title link", tint = MaterialTheme.colorScheme.primary
            )
        }
    }
}