/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.profiles

import android.app.Application
import android.content.Context
import android.content.SharedPreferences
import android.os.Handler
import android.os.Looper
import android.widget.Toast
import androidx.lifecycle.AndroidViewModel
import androidx.lifecycle.viewModelScope
import androidx.preference.PreferenceManager
import com.zell_mbc.medilog.MainActivity.Companion.Delimiter
import com.zell_mbc.medilog.MainActivity.Companion.profileFields
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.data.MediLogDB
import com.zell_mbc.medilog.data.Profiles
import com.zell_mbc.medilog.preferences.SettingsActivity
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking
import java.io.File

class ProfilesViewModel(application: Application): AndroidViewModel(application) {
    val app = application

    @JvmField
    val preferences: SharedPreferences = PreferenceManager.getDefaultSharedPreferences(app)

    @JvmField
    val separator = preferences.getString(SettingsActivity.KEY_PREF_DELIMITER, ",")
    private val lineSeparator = System.lineSeparator()

    private val dao = MediLogDB.getDatabase(app).profilesDao()

    // viewModelScope is ok because dao.upsert is a suspend function, w/o suspend -> crash
    fun upsert(profile: Profiles) = viewModelScope.launch(Dispatchers.IO) { dao.upsert(profile) }
    //fun update(p: Profiles)       = viewModelScope.launch(Dispatchers.IO) { dao.updateProfile(p._id, p.name, p.description) }

    fun delete(id: Int) = viewModelScope.launch(Dispatchers.IO) {
        val dataDao = MediLogDB.getDatabase(app).dataDao()
        val db = MediLogDB.getDatabase(app)

        db.runInTransaction {
            dao.delete(id) // Delete item from profiles table
            dataDao.deleteProfile(id) // Delete all data entries associated with this profile
            // The other tables are profile independent
        }
        val prefsDir = File(app.applicationInfo.dataDir, "shared_prefs")  // navigate to shared_prefs folder
        if (prefsDir.exists() && prefsDir.isDirectory) {
            val profileFile = File(prefsDir, "profile_$id.xml")
            if (profileFile.exists()) profileFile.delete()
        }
    }

    // DB functions
    fun getAllRecords(): Flow<List<Profiles>> = dao.getAllRecords()

    fun get(id: Int): Profiles? {
        var p: Profiles? = null
        runBlocking {
            val j = launch(Dispatchers.IO) { p = dao.getItem(id) }
            j.join()
        }
        return p
    }


    fun count(): Long {
        var count = 0L
        runBlocking {
            val j = launch(Dispatchers.IO) {
                count = dao.count()
            }
            j.join()
        }
        return count
    }

    // Wait for the new id to be returned
    fun upsertBlocking(p: Profiles): Long {
        var id = -2L
        runBlocking {
            val j = launch(Dispatchers.IO) { id = dao.upsert(p) }
            j.join()
        }
        if (id == -2L) id = p._id.toLong() // If this is an update return current id

        return id
    }

    // Deletes all records
    fun deleteAll() {
        runBlocking {
            val j = launch(Dispatchers.IO) {
                MediLogDB.getDatabase(app).runInTransaction {
                    dao.deleteAll()
                    dao.resetPrimaryKey()
                }
            }
            j.join()
        }
    }

    fun backup():List<Profiles> {
        var items = emptyList<Profiles>()
        runBlocking {
            val j = launch(Dispatchers.IO) {
                items = dao.backup()
            }
            j.join()
        }
        return items
    }

    // Return Id of first profile
    fun getFirst(): Int {
        var profileId = -1
        runBlocking {
            val j = launch(Dispatchers.IO) {
                val item = dao.getFirst()
                if (item != null) profileId = item._id
            }
            j.join()
        }

        return profileId // > 0) profileId else -1
    }

    //--------------------------------
    // Profile specific

    // Create new profile with default values
    fun createNewProfile(name: String = app.getString(R.string.userNameDefault), description: String = ""): Profiles {
        val profile = Profiles(_id = 0) //

        profile.name = name
        profile.description = description
        val newId = upsertBlocking(profile).toInt()
        profile._id = newId
        return profile
    }

    fun dataToCSV(items: List<Profiles>): String {
        if (items.isEmpty()) {
            // This will always be called from a coroutine, hence the looper is required
            Handler(Looper.getMainLooper()).post { Toast.makeText(app, app.getString(R.string.noDataToExport), Toast.LENGTH_LONG).show() }
            return ""
        }

        val sb = StringBuilder()

        val tableFields = profileFields  //getTableColumnNames("profiles", app)

    // Compile header line
        var line = ""
        for (field in tableFields) {
            line += field.name + separator
        }
        line += lineSeparator

       // val line2 = "_id$separator name$separator height$separator sex$separator dob$separator weight_threshold$separator fluid_threshold$separator active_tabs$separator comment$separator$lineSeparator"
        sb.append(line)
        //Log.d("Backup", line)

        for (item in items) {
            line = ""
            // Not elegant but this makes sure the field order always matches the header line
            for (field in tableFields) {
                when (field.name) {
                    "_id"                    -> line += item._id.toString() + separator
                    "name"                   -> line += item.name + separator
                    "description"            -> {
                        if (item.description.contains("\n")) item.description = item.description.replace("\n","\\n")  // Look for line breaks
                        if (item.description.contains(Delimiter.STRING)) item.description = item.description.replace(Delimiter.STRING.toString(),"&quot;")  // Look for "
                        line += Delimiter.STRING + item.description + Delimiter.STRING + separator
                    }
                }
            }
            line += lineSeparator
            sb.append(line)
        }
        return sb.toString()
    }
}