/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.onboarding

import android.content.Context
import androidx.activity.compose.LocalActivity
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.pager.HorizontalPager
import androidx.compose.foundation.pager.rememberPagerState
import androidx.compose.foundation.shape.CircleShape
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.core.content.edit
import androidx.preference.Preference
import androidx.preference.PreferenceManager
import com.zell_mbc.medilog.ACTIVE_TABS_KEY
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.dialogs.TabsSelectionDialog
import com.zell_mbc.medilog.support.AppStart
import kotlinx.coroutines.launch

//@OptIn(ExperimentalPagerApi::class)
@Composable
fun OnboardingPage(modifier: Modifier = Modifier, context: Context, setValue: (AppStart) -> Unit) {
    val context = LocalContext.current
    val activity = LocalActivity.current

    val pages = listOf(
        OnboardingModel(
            title = context.getString(R.string.disclaimer),
            description = context.getString(R.string.disclaimerText),
            imageRes = R.drawable.ic_disclaimer),
        OnboardingModel(
            title = context.getString(R.string.onboardingTabs),
            description = context.getString(R.string.onboardingTabsDescription),
            imageRes = R.drawable.ic_tabs),
        OnboardingModel(
            title = context.getString(R.string.onboardingBackup),
            description = context.getString(R.string.onboardingBackupDescription),
            imageRes = R.drawable.ic_backup)
        )

    val pagerState = rememberPagerState(pageCount = { pages.size } )
    val coroutineScope = rememberCoroutineScope()
    var showTabDialog by remember { mutableStateOf(false) }

    Column(modifier = modifier.fillMaxSize().padding(16.dp)) {
        HorizontalPager(state = pagerState, modifier = Modifier.weight(1f).fillMaxWidth()) { page -> OnboardingItem(pages[page]) }
        Row(horizontalArrangement = Arrangement.SpaceBetween, verticalAlignment = Alignment.CenterVertically, modifier = Modifier.fillMaxWidth().padding(10.dp)
        ) {
            val textColor = if (pagerState.currentPage == 0) MaterialTheme.colorScheme.onPrimary else MaterialTheme.colorScheme.primary //Color(0xFFAAAAAA) else Color(0xFF333333)
            Text(
                context.getString(R.string.previous), style = TextStyle(color = textColor, fontSize = 14.sp, fontWeight = FontWeight.Normal),
                 modifier = Modifier.clickable {
                     val previousPage = pagerState.currentPage - 1
                     coroutineScope.launch { pagerState.animateScrollToPage(previousPage) }
                 }
            )

            Row(horizontalArrangement = Arrangement.Center, modifier = Modifier.weight(1f)) {
                repeat(pages.size) { index ->
                    val isSelected = pagerState.currentPage == index
                    Box(
                        modifier = Modifier
                        .padding(4.dp)
                        .width(if (isSelected) 18.dp else 8.dp)
                        .height(if (isSelected) 8.dp else 8.dp)
                        .border(
                            width = 1.dp,
                            color = Color(0xFF707784),
                            shape = RoundedCornerShape(10.dp)
                        )
                        .background(color = if (isSelected) Color(0xFF3B6C64) else Color(0xFFFFFFFF), shape = CircleShape)
                    )
                }
            }
            val s  = if (pagerState.currentPage < pages.size -1) context.getString(R.string.next) else context.getString(R.string.done)
            Text(s, style = TextStyle(color = MaterialTheme.colorScheme.primary, fontSize = 14.sp, fontWeight = FontWeight.Normal),
                 modifier = Modifier.clickable {
                     if (pagerState.currentPage < pages.size -1) {
                         val nextPage = pagerState.currentPage + 1
                         coroutineScope.launch { pagerState.animateScrollToPage(nextPage) }
                     }
                     else {
                         showTabDialog = true
                     }
                 }
            )
            var selectedTabs by remember { mutableStateOf<List<Int>>(emptyList()) }

            TabsSelectionDialog(
                showDialog = showTabDialog,
                onDismiss = {
                    setValue(AppStart.NORMAL)
                    showTabDialog = false
                    val intent = activity?.intent
                    activity?.finish()
                    //overridePendingTransition(0, 0)
                    activity?.startActivity(intent)
                }, // Not interested? Get all of them as set higher up
                onConfirm = {
                    setValue(AppStart.NORMAL)
                    selectedTabs = it
                    var tabString = ""
                    selectedTabs.forEach { tabString += "$it," }
                    val preferences = PreferenceManager.getDefaultSharedPreferences(context)
                    preferences.edit { putString(ACTIVE_TABS_KEY, tabString) }
                    showTabDialog = false

                    val intent = activity?.intent
                    activity?.finish()
                    activity?.startActivity(intent)
                }
            )

        }
    }
}
