/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.documents

import android.os.Bundle
import androidx.activity.compose.setContent
import androidx.activity.viewModels
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.imePadding
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.material3.TextField
import androidx.compose.material3.TextFieldColors
import androidx.compose.material3.TextFieldDefaults
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.input.KeyboardCapitalization
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.unit.dp
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.Tabs
import com.zell_mbc.medilog.base.EditActivity
import kotlin.getValue

class DocumentsEditActivity : EditActivity() {
    override val viewModel: DocumentsViewModel by viewModels()
    override val dataType = Tabs.DOCUMENTS

    // Todo: When loading a new document and comment is empty, add Document Name as description
    override fun onCreate(savedInstanceState: Bundle?) {
        attachementsEnabled = true

        super.onCreate(savedInstanceState)
        setContent { StartCompose() }
    }

    @Composable
    override fun ShowContent() {
        Column(modifier = Modifier
            .padding(start = 8.dp, end = 8.dp)
            .imePadding()) {
            DateTimeBlock()
            val textFieldColors: TextFieldColors = TextFieldDefaults.colors(unfocusedContainerColor = Color.Transparent, focusedContainerColor = Color.Transparent)

            Row(Modifier.fillMaxWidth(), verticalAlignment = Alignment.CenterVertically) {
                Row(Modifier.fillMaxWidth()) {
                    // Comment
                    TextField(
                        modifier = Modifier.fillMaxWidth(),
                        value = viewModel.comment.value,
                        colors = textFieldColors,
                        onValueChange = { viewModel.comment.value = it },
                        label = { Text(text = viewModel.comment.label, style = MaterialTheme.typography.bodyMedium) },
                        keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Text, capitalization = KeyboardCapitalization.Sentences),
                    )
                }
            }
            Text("")
            if (attachmentPresent()) {
                Row(modifier = Modifier.align(Alignment.CenterHorizontally)) {
                    AttachmentBlock()
                }
            }
        }
  //      }

        // Dialogs
        if (showTagsDialog)
            tagsDialog.ShowDialog(viewModel.dataType, selectedTags = viewModel.tagIds, setShowDialog = { showTagsDialog = it }) {
                viewModel.tagIds = it
            }

        if (showTextTemplatesDialog)
            textTemplateDialog.ShowDialog(viewModel.dataType, setShowDialog = { showTextTemplatesDialog = it }) { viewModel.comment.value += it }

        if (showDatePickerDialog) OpenDatePickerDialog()
        if (showTimePickerDialog) OpenTimePickerDialog()    }


    override fun saveItem() {
        // Check empty variables
        if (viewModel.comment.value.isEmpty()) {
            hideKeyboard()
            snackbarDelegate.showSnackbar(getString(R.string.descriptionMissing))
            return
        }
        super.saveItem()
        finish() // Close current window / activity
    }
}