/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

/*
 */

package com.zell_mbc.medilog.diary

import android.app.Application
import android.content.SharedPreferences
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableIntStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.core.content.edit
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.R.string
import com.zell_mbc.medilog.Tabs
import com.zell_mbc.medilog.data.DataViewModel
import com.zell_mbc.medilog.preferences.SettingsActivity

class DiaryViewModel(application: Application): DataViewModel(application, Tabs.DIARY) {
    override val filterStartPref = "DIARYFILTERSTART"
    override val filterEndPref = "DIARYFILTEREND"
    override val timeFilterModePref =  "DIARY_FILTER_MODE"
    override val rollingFilterValuePref = "DIARY_ROLLING_FILTER_VALUE"
    override val rollingFilterTimeframePref = "DIARY_ROLLING_FILTER_TIMEFRAME"
    override val tagFilterPref = "DIARYTAGFILTER"
    override val dateTimeColumnWidthPref = "DIARY_DATE_COLUMN_WIDTH"

    //All preferences that affect UI or app behavior displayed in UI should be here
    // General settings
    override var blendInItems    by mutableStateOf(preferences.getBoolean(SettingsActivity.KEY_PREF_DIARY_BLENDINITEMS, app.getString(string.BLENDINITEMS_DEFAULT).toBoolean()))
    override var highlightValues = false
    override var showTime        by mutableStateOf(preferences.getBoolean(SettingsActivity.KEY_PREF_DIARY_SHOWTIME, app.getString(string.SHOWTIME_DEFAULT).toBoolean()))
    override var upperThreshold  = 0f
    override var lowerThreshold  = 0f

    var currentEmoji by mutableIntStateOf( R.drawable.ic_ok )
    var goodSmiley by mutableStateOf("")
    var notGoodSmiley by mutableStateOf( "")
    var badSmiley by mutableStateOf("")

    var colorsInitialized = false

    init {
        setupEntryFields()
        setupEmojis()

        // Register preference change listener
        // Handle updates
        prefListener = SharedPreferences.OnSharedPreferenceChangeListener { _, key ->
            when (key) {
                SettingsActivity.KEY_PREF_QUICKENTRY -> quickEntry = preferences.getBoolean(SettingsActivity.KEY_PREF_QUICKENTRY, true)
                SettingsActivity.KEY_PREF_DIARY_SHOWTIME -> {
                    showTime = preferences.getBoolean(SettingsActivity.KEY_PREF_DIARY_SHOWTIME, app.getString(string.SHOWTIME_DEFAULT).toBoolean())
                    preferences.edit { remove(dateTimeColumnWidthPref) } // Make sure we measure again
                }
                SettingsActivity.KEY_PREF_DIARY_SHOWGOOD,
                SettingsActivity.KEY_PREF_DIARY_SHOWNOTGOOD,
                SettingsActivity.KEY_PREF_DIARY_SHOWBAD -> setupEmojis()
                SettingsActivity.KEY_PREF_DIARY_BLENDINITEMS -> { blendInItems = preferences.getBoolean(SettingsActivity.KEY_PREF_DIARY_BLENDINITEMS, app.getString(string.BLENDINITEMS_DEFAULT).toBoolean()) }
                SettingsActivity.KEY_PREF_DIARY_CUSTOM_NAME -> {
                    setupEntryFields() // refresh the fields on any related change
                }
            }
        }
        preferences.registerOnSharedPreferenceChangeListener(prefListener)
    }

    // Specific parameters, applicable to just this field
    override fun setupEntryFields() {
        comment.label = preferences.getString(SettingsActivity.KEY_PREF_DIARY_CUSTOM_NAME, "").toString()

        // After custom was set once the above won't provide the default any longer
        if (comment.label.isEmpty()) comment.label = app.getString(string.diary)
    }

    fun setupEmojis() {
        notGoodSmiley = if (preferences.getBoolean(SettingsActivity.KEY_PREF_DIARY_SHOWNOTGOOD, true)) app.getString(R.string.notGoodSmileyEmojy) else " "
        badSmiley = if (preferences.getBoolean(SettingsActivity.KEY_PREF_DIARY_SHOWBAD, true)) app.getString(R.string.badSmileyEmojy) else " "
        goodSmiley = if (preferences.getBoolean(SettingsActivity.KEY_PREF_DIARY_SHOWGOOD, false)) "😀︎" else " "
    }
}
