/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.data

import android.content.Context
import android.net.Uri
import android.util.Log
import androidx.preference.PreferenceManager
import com.zell_mbc.medilog.databaseName
import com.zell_mbc.medilog.support.LAST_APP_VERSION
import java.io.File
import java.io.FileOutputStream

fun backupDatabase(context: Context) {
    val dbName = databaseName
    val dbFile = context.getDatabasePath(dbName)
    val preferences = PreferenceManager.getDefaultSharedPreferences(context)
    val lastVersionCode = preferences.getInt(LAST_APP_VERSION, -1)
    val backupFile = File(context.filesDir, dbFile.name + "-$lastVersionCode.backup")

    val db = MediLogDB.getDatabase(context)

    try {
        // Close DB to flush and release resources
        db.closeInstance()

        if (!dbFile.exists()) {
            Log.e("Backup", "Database file does not exist.")
            return
        }

        dbFile.copyTo(backupFile, overwrite = true)
        Log.i("Backup", "Database backed up to ${backupFile.absolutePath}")

    } catch (e: Exception) {
        Log.e("Backup", "Backup failed", e)
    } finally {
        // Reopen DB
        MediLogDB.getDatabase(context)
    }
}

fun restoreDatabaseFromUri(context: Context, databaseName: String, backupUri: Uri, onResult: (success: Boolean) -> Unit) {
    val dbFile = context.getDatabasePath(databaseName)
    try {
        // Close existing DB instance before overwriting
        val db = MediLogDB.getDatabase(context)
        db.closeInstance()

        context.contentResolver.openInputStream(backupUri)?.use { inputStream ->
            FileOutputStream(dbFile).use { outputStream ->
                inputStream.copyTo(outputStream)
            }
        } ?: throw IllegalArgumentException("Unable to open input stream from URI")

        onResult(true)
    } catch (e: Exception) {
        e.printStackTrace()
        onResult(false)
    } finally {
        // Reopen the DB after restore
        MediLogDB.getDatabase(context)
    }
}
