/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.bloodpressure

import android.os.Bundle
import androidx.activity.compose.setContent
import androidx.activity.viewModels
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.RowScope
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.safeDrawingPadding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.base.InfoActivity
import kotlin.getValue

class BloodPressureInfoActivity: InfoActivity() {
    private val viewModel: BloodPressureViewModel by viewModels()
    val column1Weight = .4f
    val column2Weight = .2f
    val column3Weight = .2f
    val column4Weight = .2f

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        supportActionBar?.setIcon(R.drawable.ic_bloodpressure)

        totalData = collectData(viewModel, this, totalData)
        setContent { StartCompose() }
    }

    @Composable
    fun RowScope.TableCell(text: String, weight: Float, color: Color = MaterialTheme.colorScheme.primary, align:TextAlign = TextAlign.Center) {
        Text(text = text, Modifier.weight(weight), color = color, textAlign = align)
    }

    @Composable
    fun ShowRow(s: String) {
        val row = s.split("|")
        when {
            row[0].contains("header") -> { // header line
                Text(text = row[1], Modifier.padding(cellPadding.dp), color = MaterialTheme.colorScheme.onPrimaryContainer)
                HorizontalDivider(thickness = 1.dp)
            }
            row[0].contains("footer") -> {
                Text(text = row[1], Modifier.padding(cellPadding.dp), color = MaterialTheme.colorScheme.onPrimaryContainer, style = MaterialTheme.typography.bodySmall)
            }
            else ->
                Row(Modifier.fillMaxWidth()) {
                    TableCell(text = row[0], weight = column1Weight, align = TextAlign.Start)
                    TableCell(text = row[1], weight = column2Weight)
                    TableCell(text = row[2], weight = column3Weight)
                    TableCell(text = row[3], weight = column4Weight)
                }
        }
    }

    @Composable
    override fun ShowContent() {
        Surface(
            shape = RoundedCornerShape(8.dp),
            modifier = Modifier.safeDrawingPadding().fillMaxWidth()) {
            Column(modifier = Modifier.padding(8.dp)) {
                Text("")
                //HeaderBlock(R.drawable.ic_bloodpressure)
                // Each cell of a column must have the same weight.
                val column1Weight = .4f
                val column2Weight = .2f
                val column3Weight = .2f
                val column4Weight = .2f

                Row(Modifier.background(MaterialTheme.colorScheme.primaryContainer)) {
                    TableCell(text = "", weight = column1Weight, MaterialTheme.colorScheme.onPrimaryContainer)
                    TableCell(text = getString(R.string.monthLabel), weight = column2Weight, MaterialTheme.colorScheme.onPrimaryContainer)
                    TableCell(text = getString(R.string.threeMonthLabel), weight = column3Weight, MaterialTheme.colorScheme.onPrimaryContainer)
                    TableCell(text = getString(R.string.annualLabel), weight = column4Weight, MaterialTheme.colorScheme.onPrimaryContainer)
                }
                LazyColumn(modifier = Modifier.fillMaxSize(1F)  ) {
                    items(totalData) { item ->
                        ShowRow(item)
                    }
                }
            }
        }
    }
}