/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.bloodpressure

import android.os.Bundle
import android.widget.Toast
import androidx.activity.viewModels
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.safeDrawingPadding
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import com.patrykandpatrick.vico.compose.cartesian.CartesianChartHost
import com.patrykandpatrick.vico.compose.cartesian.axis.rememberBottom
import com.patrykandpatrick.vico.compose.cartesian.axis.rememberStart
import com.patrykandpatrick.vico.compose.cartesian.layer.rememberLine
import com.patrykandpatrick.vico.compose.cartesian.layer.rememberLineCartesianLayer
import com.patrykandpatrick.vico.compose.cartesian.rememberCartesianChart
import com.patrykandpatrick.vico.compose.cartesian.rememberVicoZoomState
import com.patrykandpatrick.vico.compose.common.fill
import com.patrykandpatrick.vico.core.cartesian.Zoom
import com.patrykandpatrick.vico.core.cartesian.axis.HorizontalAxis
import com.patrykandpatrick.vico.core.cartesian.axis.VerticalAxis
import com.patrykandpatrick.vico.core.cartesian.data.CartesianChartModelProducer
import com.patrykandpatrick.vico.core.cartesian.data.CartesianLayerRangeProvider
import com.patrykandpatrick.vico.core.cartesian.data.lineSeries
import com.patrykandpatrick.vico.core.cartesian.decoration.HorizontalLine
import com.patrykandpatrick.vico.core.cartesian.layer.LineCartesianLayer
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.R.string
import com.zell_mbc.medilog.base.ChartActivity
import com.zell_mbc.medilog.base.rememberMarker
import com.zell_mbc.medilog.preferences.SettingsActivity
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import kotlin.getValue
import kotlin.math.abs
import kotlin.math.min

class BloodPressureChartActivity: ChartActivity() {
    private val viewModel: BloodPressureViewModel by viewModels()
    override val filename = "BloodPressureChart.jpg"

    // Tab specific settings
    var showPulse = false
    //private var showRibbonChart = false

    private var SYS = 0
    private var DIA = 1
    private var PULSE = 2

    private var sysLine = 0
    private var sysLabel = ""
    private var diaLine = 0
    private var diaLabel = ""

    private var minY = 999.9
    private var maxY = 0.1

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        showLinearTrendline = preferences.getBoolean(SettingsActivity.KEY_PREF_SHOW_BLOODPRESSURE_LINEAR_TRENDLINE, false)
        showPulse = preferences.getBoolean(SettingsActivity.KEY_PREF_SHOWPULSE, false)
        showGrid = preferences.getBoolean(SettingsActivity.KEY_PREF_SHOW_BLOODPRESSURE_GRID, true)
        showLegend = preferences.getBoolean(SettingsActivity.KEY_PREF_SHOW_BLOODPRESSURE_LEGEND, false)
        showValues = preferences.getBoolean(SettingsActivity.KEY_PREF_BLOODPRESSURE_SHOW_VALUES, true)
        showMovingAverage = preferences.getBoolean(SettingsActivity.KEY_PREF_BLOODPRESSURE_SHOW_MOVING_AVERAGE, false)
        showThreshold = preferences.getBoolean(SettingsActivity.KEY_PREF_SHOW_BLOODPRESSURE_THRESHOLD, false)
       // showRibbonChart = preferences.getBoolean(SettingsActivity.KEY_PREF_BLOODPRESSURE_RIBBON_CHART, false)

        val bpHelper = BloodPressureHelper(this)

        if (!(showValues || showLinearTrendline || showMovingAverage)) {
            Toast.makeText(this, getString(string.noValuesActivated), Toast.LENGTH_LONG).show()
            finish()
            return // Needs return to avoid the rest to be executed because finish() may not kill the activity fast enough
        }

        // Prepare legend
        if (showValues) {
            legendList.add(legendItem(label = getString(string.systolic), color = lineColor1))
            legendList.add(legendItem(label = getString(string.diastolic), color = lineColor2))
            if (showPulse) legendList.add(legendItem(label = getString(string.pulse), color = lineColor3))
        }

        if (showMovingAverage) {
            legendList.add(legendItem(label = getString(R.string.showMovingAverage), color = lineColor1))
            legendList.add(legendItem(label = getString(string.showMovingAverage), color = lineColor2))
        }
        if (showLinearTrendline) {
            legendList.add(legendItem(label = getString(string.showLinearTrendline), color = lineColor1))
        }

        // For the charts only native entries can be considered, no matter what showAllTabs is set to
        val tmp = viewModel.blendInItems
        viewModel.blendInItems = false
        items = viewModel.getItems("ASC", filtered = true)
        viewModel.blendInItems = tmp

        if (items.size < 2) {
            Toast.makeText(this, getString(string.notEnoughDataForChart), Toast.LENGTH_LONG).show()
            finish()
            return // Needs return to avoid the rest to be executed because finish() may not kill the activity fast enough
        }

        // Collect list values and hold min/max values
        var fTmp: Float
        var i = 0
        for (item in items) {
            timestamps.add(i++)
            fTmp = try { item.value1.toFloat() } catch (_: NumberFormatException) { 0f }
            value1List.add(fTmp)
            if (fTmp > chartMax) chartMax = fTmp
            //if (fTmp < chartMin) chartMin = fTmp // Not needed because value2Min will be smaller

            fTmp = try { item.value2.toFloat() } catch (_: NumberFormatException) { 0f }
            value2List.add(fTmp)
            //if (fTmp > chartMax) chartMax = fTmp  // Not needed because value2Max will be higher
            if (fTmp < value2Min) value2Min = fTmp

            if (showPulse) { fTmp = try { item.value3.toFloat() } catch (_: NumberFormatException) { 0f }
                value3List.add(fTmp)
                if (fTmp > value3Max) value3Max = fTmp
                if (fTmp < value3Min) value3Min = fTmp
            }
        }

        // Compile moving average values
        if (showMovingAverage) {
            val p = preferences.getString(SettingsActivity.KEY_PREF_BLOODPRESSURE_MOVING_AVERAGE_SIZE, getString(string.BLOODPRESSURE_MOVING_AVERAGE_SIZE_DEFAULT))
            if (p != null) {
                val period = try {
                    p.toInt()
                } catch (_: NumberFormatException) {
                    0
                }
                movingAverageValue1List = getMovingAverageFloat(value1List, period)
                movingAverageValue2List = getMovingAverageFloat(value2List, period)
                if (showPulse) movingAverageValue3List = getMovingAverageFloat(value3List, period)
            }
        }
        // Compile linear trendline values
        if (showLinearTrendline) {
            linearTrendlineValue1List = getLinearTrendlineFloat(value1List)
            linearTrendlineValue2List = getLinearTrendlineFloat(value2List)
            if (showPulse) linearTrendlineValue3List = getLinearTrendlineFloat(value3List)
        }

        xAxisOffset = items[0].timestamp

        // Fill data maps
        if (showValues) {
            value1Map = items.associateBy({ scaledTimeStamp(it.timestamp) }, { try { it.value1.toFloat() } catch (_: NumberFormatException) { 0f } })
            value2Map = items.associateBy({ scaledTimeStamp(it.timestamp) }, { try { it.value2.toFloat() } catch (_: NumberFormatException) { 0f } })
            if (showPulse) value3Map = items.associateBy({ scaledTimeStamp(it.timestamp) }, { try { it.value3.toFloat() } catch (_: NumberFormatException) { 0f } })
        }

        if (showLinearTrendline) {
            var ii = 0
            linearTrendlineValue1Map = items.associateBy({ scaledTimeStamp(it.timestamp) }, { try { linearTrendlineValue1List[ii++] } catch (_: NumberFormatException) { 0F } })
            ii = 0
            linearTrendlineValue2Map = items.associateBy({ scaledTimeStamp(it.timestamp) }, { try { linearTrendlineValue2List[ii++] } catch (_: NumberFormatException) { 0F } })
            if (showPulse) {
                ii = 0
                linearTrendlineValue3Map = items.associateBy({ scaledTimeStamp(it.timestamp) }, { try { linearTrendlineValue3List[ii++] } catch (_: NumberFormatException) { 0F } })
            }
        }

        if (showMovingAverage) {
            var ii = 0
            movingAverageValue1Map = items.associateBy({ scaledTimeStamp(it.timestamp) }, { try { movingAverageValue1List[ii++] } catch (_: NumberFormatException) { 0F }})
            ii = 0
            movingAverageValue2Map = items.associateBy({ scaledTimeStamp(it.timestamp) }, { try { movingAverageValue2List[ii++] } catch (_: NumberFormatException) { 0F } })
            if (showPulse) {
                ii = 0
                movingAverageValue3Map = items.associateBy({ scaledTimeStamp(it.timestamp) }, { try { movingAverageValue3List[ii++] } catch (_: NumberFormatException) { 0F } })
            }
        }

        // Show the grade line next to max sys value
        val sysGrades = listOf(
            abs(bpHelper.hyperGrade1Sys - chartMax),
            abs(bpHelper.hyperGrade2Sys - chartMax),
            abs(bpHelper.hyperGrade3Sys - chartMax)
        )

        sysLine = 0
        sysLabel = getString(string.systolic) + ", " +  getString(string.grade)
        when (sysGrades.min())    {
            sysGrades[0] -> {
                sysLine = bpHelper.hyperGrade1Sys
                sysLabel += " 1"
            }
            sysGrades[1] -> {
                sysLine = bpHelper.hyperGrade1Sys
                sysLabel += " 2"
            }
            else         -> {
                sysLine = bpHelper.hyperGrade1Sys
                sysLabel += " 3"
            }
        }

        val diaGrades = listOf(
            abs(bpHelper.hyperGrade1Sys - value2Max),
            abs(bpHelper.hyperGrade2Sys - value2Max),
            abs(bpHelper.hyperGrade3Sys - value2Max)
        )

        diaLine = 0
        diaLabel = getString(string.diastolic) + ", " +  getString(string.grade)
        when (diaGrades.min())    {
            diaGrades[0] -> {
                diaLine = bpHelper.hyperGrade1Dia
                diaLabel += " 1"
            }
            diaGrades[1] -> {
                diaLine = bpHelper.hyperGrade2Dia
                diaLabel += " 2"
            }
            else         -> {
                diaLine = bpHelper.hyperGrade3Dia
                diaLabel += " 3"
            }
        }
        chartMin = (if (showPulse) min(value2Min, value3Min) else value2Min) - 5f
        chartMax = chartMax + 5f
        showContent()
    }

    @Composable
    override fun ShowContent() {
        Box(Modifier.safeDrawingPadding()) {
            // Sys and Dia thresholds
            val decorationList = if (showThreshold) listOf(
                helperLine("", sysLine.toDouble(), lineColor1),
                helperLine("", diaLine.toDouble(), lineColor2),
            ) else listOf()

            val modelProducer = remember { CartesianChartModelProducer() }
            LaunchedEffect(Unit) {
                withContext(Dispatchers.Default) {
                    modelProducer.runTransaction {
                        lineSeries {
                            if (showValues) {
                                value1Map?.let { series(it.keys, it.values) }
                                value2Map?.let { series(it.keys, it.values) }
                                value3Map?.let { series(it.keys, it.values) }
                            }
                            if (showMovingAverage) {
                                movingAverageValue1Map?.let { series(it.keys, it.values) }
                                movingAverageValue2Map?.let { series(it.keys, it.values) }
                                movingAverageValue3Map?.let { series(it.keys, it.values) }
                            }
                            if (showLinearTrendline) {
                                linearTrendlineValue1Map?.let { series(it.keys, it.values) }
                                linearTrendlineValue2Map?.let { series(it.keys, it.values) }
                                linearTrendlineValue3Map?.let { series(it.keys, it.values) }
                            }
                        }
                    }
                }
            }
            ComposeChart(modelProducer, Modifier.fillMaxHeight(), decorationList)
        }
    }

    @Composable
    private fun ComposeChart(
        modelProducer: CartesianChartModelProducer,
        modifier: Modifier, decorationList: List<HorizontalLine>) {

        val marker = rememberMarker()
        //val backgroundFill = if (isSystemInDarkTheme()) Color(0xff000000) else Color(0xffffffff)

        val lineProvider = LineCartesianLayer.LineProvider.series(
            buildList {
                if (showValues) {
                    add(LineCartesianLayer.rememberLine(areaFill = LineCartesianLayer.AreaFill.single(fill = fill(fillColor1)), fill = LineCartesianLayer.LineFill.single(fill = fill(lineColor1))))
                    /*     if (showRibbonChart) add(
                    LineCartesianLayer.rememberLine(fill = LineCartesianLayer.LineFill.single(com.patrykandpatrick.vico.compose.common.fill(chartColors[DIA])), areaFill = LineCartesianLayer.AreaFill.single(
                        com.patrykandpatrick.vico.compose.common.fill(
                            backgroundFill
                        )
                    )) )
                else */
                    add(LineCartesianLayer.rememberLine(areaFill = LineCartesianLayer.AreaFill.single(fill = fill(fillColor2)), fill = LineCartesianLayer.LineFill.single(fill(lineColor2))))
                    if (showPulse) add(LineCartesianLayer.rememberLine(areaFill = LineCartesianLayer.AreaFill.single(fill = fill(fillColor3)), fill = LineCartesianLayer.LineFill.single(fill(lineColor3))))
                }
                if (showMovingAverage) {
                    add(LineCartesianLayer.rememberLine(fill = LineCartesianLayer.LineFill.single(fill(chartColors[SYS])), areaFill = null))
                    add(LineCartesianLayer.rememberLine(fill = LineCartesianLayer.LineFill.single(fill(chartColors[DIA])), areaFill = null))
                    if (showPulse) add(LineCartesianLayer.rememberLine(fill = LineCartesianLayer.LineFill.single(fill(chartColors[PULSE])), areaFill = null))
                }
                if (showLinearTrendline) {
                    add(LineCartesianLayer.rememberLine(stroke = linearTrendLineStroke, fill = LineCartesianLayer.LineFill.single(fill(chartColors[SYS])), areaFill = null))
                    add(LineCartesianLayer.rememberLine(stroke = linearTrendLineStroke, fill = LineCartesianLayer.LineFill.single(fill(chartColors[DIA])), areaFill = null))
                    if (showPulse) add(LineCartesianLayer.rememberLine(stroke = linearTrendLineStroke, fill = LineCartesianLayer.LineFill.single(fill(chartColors[PULSE])), areaFill = null))
                }
            }
        )

        val span = chartMax - chartMin
        val step = findNiceStep(span.toDouble())

        CartesianChartHost(
            chart = rememberCartesianChart(
                rememberLineCartesianLayer(lineProvider = lineProvider, rangeProvider = CartesianLayerRangeProvider.fixed(minY = chartMin.toDouble(), maxY = chartMax.toDouble())),
                startAxis =  VerticalAxis.rememberStart(
                    guideline = guideline(), // horizontal lines
                    itemPlacer = remember { VerticalAxis.ItemPlacer.step(step = { step }, shiftTopLines = false) }
                ),
                bottomAxis = HorizontalAxis.rememberBottom(
                    valueFormatter = bottomAxisValueFormatter, // Convert values back to proper dates
                    guideline = guideline()
                ),
                legend = if (showLegend) rememberLegend() else null,
                decorations = decorationList,
                marker = marker,
            ),
            modelProducer = modelProducer,
            modifier = modifier,
            zoomState = rememberVicoZoomState(zoomEnabled = true, maxZoom = Zoom.max(Zoom.fixed(100f), Zoom.Content), initialZoom = Zoom.min(Zoom.fixed(), Zoom.Content))
        )
    }
}