/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.oximetry

import android.app.Application
import android.content.SharedPreferences
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableFloatStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.core.content.edit
import com.zell_mbc.medilog.Tabs
import com.zell_mbc.medilog.R.string
import com.zell_mbc.medilog.Thresholds.MAX_OXYGEN
import com.zell_mbc.medilog.data.DataViewModel
import com.zell_mbc.medilog.preferences.SettingsActivity
import kotlin.text.toBoolean

class OximetryViewModel(application: Application): DataViewModel(application, Tabs.OXIMETRY) {
    override val filterStartPref = "OXIMETRYFILTERSTART"
    override val filterEndPref = "OXIMETRYFILTEREND"
    override val timeFilterModePref =  "OXIMETRY_FILTER_MODE"
    override val rollingFilterValuePref = "OXIMETRY_ROLLING_FILTER_VALUE"
    override val rollingFilterTimeframePref = "OXIMETRY_ROLLING_FILTER_TIMEFRAME"
    override val tagFilterPref = "OXIMETRY_TAG_FILTER"
    override val dateTimeColumnWidthPref = "OXIMETRY_DATE_COLUMN_WIDTH"

    //All preferences that affect UI or app behavior displayed in UI should be here
    // General settings
    override var blendInItems    by mutableStateOf(preferences.getBoolean(SettingsActivity.KEY_PREF_OXIMETRY_BLENDINITEMS, app.getString(string.BLENDINITEMS_DEFAULT).toBoolean()))
    override var highlightValues by mutableStateOf(preferences.getBoolean(SettingsActivity.KEY_PREF_OXIMETRY_HIGHLIGHT_VALUES, app.getString(string.OXIMETRY_HIGHLIGHT_VALUES_DEFAULT).toBoolean()))
    override var showTime        by mutableStateOf(preferences.getBoolean(SettingsActivity.KEY_PREF_OXIMETRY_SHOWTIME, app.getString(string.SHOWTIME_DEFAULT).toBoolean()))
    override var upperThreshold  by mutableFloatStateOf(0f)
    override var lowerThreshold  by mutableFloatStateOf(0f)

    init {
        setupEntryFields()
        loadThresholds()
        // Handle updates
        prefListener = SharedPreferences.OnSharedPreferenceChangeListener { _, key ->
            when (key) {
                SettingsActivity.KEY_PREF_QUICKENTRY -> quickEntry = preferences.getBoolean(SettingsActivity.KEY_PREF_QUICKENTRY, true)
                SettingsActivity.KEY_PREF_OXIMETRY_CUSTOM_NAME,
                SettingsActivity.KEY_PREF_OXIMETRY_CUSTOM_UNIT,
                SettingsActivity.KEY_PREF_OXIMETRY_CUSTOM_TEMPLATE -> setupEntryFields() // refresh the fields on any related change }
                SettingsActivity.KEY_PREF_OXIMETRY_HIGHLIGHT_VALUES -> highlightValues = preferences.getBoolean(SettingsActivity.KEY_PREF_OXIMETRY_HIGHLIGHT_VALUES, app.getString(string.OXIMETRY_HIGHLIGHT_VALUES_DEFAULT).toBoolean())
                SettingsActivity.KEY_PREF_OXIMETRY_THRESHOLDS -> loadThresholds()
                SettingsActivity.KEY_PREF_OXIMETRY_SHOWTIME -> {
                    showTime = preferences.getBoolean(SettingsActivity.KEY_PREF_OXIMETRY_SHOWTIME, app.getString(string.SHOWTIME_DEFAULT).toBoolean())
                    preferences.edit { remove(dateTimeColumnWidthPref) } // Make sure we measure again
                }
                SettingsActivity.KEY_PREF_OXIMETRY_BLENDINITEMS -> blendInItems = preferences.getBoolean(SettingsActivity.KEY_PREF_OXIMETRY_BLENDINITEMS, app.getString(string.BLENDINITEMS_DEFAULT).toBoolean())
            }
        }
        preferences.registerOnSharedPreferenceChangeListener(prefListener)         // Register preference change listener
    }

    fun loadThresholds(){
        val thresholdString = preferences.getString(SettingsActivity.KEY_PREF_OXIMETRY_THRESHOLDS, app.getString(string.OXIMETRY_THRESHOLDS_DEFAULT))
        if (highlightValues && thresholdString != null) {
            lowerThreshold = try { thresholdString.toFloat() } catch (_: NumberFormatException) { 0f }
            upperThreshold = MAX_OXYGEN.toFloat() // Can't be more than 100%
        }
    }
    // Specific parameters, applicable to just this field
    override fun setupEntryFields() {
        value1.label = preferences.getString(SettingsActivity.KEY_PREF_OXIMETRY_CUSTOM_NAME, app.getString(string.oximetry)) ?: ""
        value1.unit = preferences.getString(SettingsActivity.KEY_PREF_OXIMETRY_CUSTOM_UNIT, "").toString()
        value1.template = preferences.getString(SettingsActivity.KEY_PREF_OXIMETRY_CUSTOM_TEMPLATE, app.getString(string.oximetryHint)) ?: ""

        // After custom was set once the above won't provide the default any longer
        if (value1.label.isEmpty()) value1.label = app.getString(string.oximetry)
        if (value1.unit.isEmpty()) value1.unit = preferences.getString(SettingsActivity.KEY_PREF_OXIMETRY_UNIT, app.getString(string.OXIMETRY_UNIT_DEFAULT)).toString()
        if (value1.template.isEmpty()) value1.template   = app.getString(string.oximetryHint)

        // No custom values supported, yet
        value2.label    = app.getString(string.pulse)
        value2.unit     = app.getString(string.pulseUnit)
        value2.template = app.getString(string.PULSE_HINT)

        comment.label = app.getString(string.comment)
    }
}
