/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

/*
 */

package com.zell_mbc.medilog.bloodpressure

import android.app.Activity
import android.content.Context
import android.content.SharedPreferences
import android.widget.Toast
import androidx.preference.PreferenceManager
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.preferences.SettingsActivity
import java.util.ArrayList
import java.util.Calendar
import androidx.core.content.edit
import com.zell_mbc.medilog.Tabs.BLOODPRESSURE
import com.zell_mbc.medilog.data.DataViewModel
import kotlin.Exception
import kotlin.math.max
import kotlin.math.min

class BloodPressureHelper(val context: Context) {
    var hyperGrade3Sys = 0
    var hyperGrade3Dia = 0
    var hyperGrade2Sys = 0
    var hyperGrade2Dia = 0
    var hyperGrade1Sys = 0
    var hyperGrade1Dia = 0
    private var hypotensionSys = 0
    private var hypotensionDia = 0

    val hyperGrade1 = 1
    val hyperGrade2 = 2
    val hyperGrade3 = 3
    val hypotension = 4

    private val gradeSettingsDivider = "/"
    val preferences: SharedPreferences = PreferenceManager.getDefaultSharedPreferences(context)

    init {
        // Check values
        //var sTmp = preferences.getString(SettingsActivity.KEY_PREF_HYPERGRADE1, context.getString(R.string.BLOODPRESSURE_GRADE1_VALUES)).toString()
        var sTmp = preferences.getString(SettingsActivity.KEY_PREF_HYPERGRADE1, context.getString(R.string.BLOODPRESSURE_GRADE1_VALUES)) ?: ""
        if (sTmp.contains(",")) {
            migrateThresholds() // Convert legacy (pre v3.0) settings
            sTmp = preferences.getString(SettingsActivity.KEY_PREF_HYPERGRADE1, context.getString(R.string.BLOODPRESSURE_GRADE1_VALUES)) ?: ""
        }

        try {
            val grade = sTmp.split(gradeSettingsDivider).toTypedArray()
            hyperGrade1Sys = grade[0].toInt()
            hyperGrade1Dia = grade[1].toInt()
        } catch (_: Exception) {
            Toast.makeText(context, context.getString(R.string.grade1Error) + " $sTmp", Toast.LENGTH_LONG).show()
        }
        sTmp = preferences.getString(SettingsActivity.KEY_PREF_HYPERGRADE2, context.getString(R.string.BLOODPRESSURE_GRADE2_VALUES)) ?: ""
        try {
            val grade = sTmp.split(gradeSettingsDivider).toTypedArray()
            hyperGrade2Sys = grade[0].toInt()
            hyperGrade2Dia = grade[1].toInt()
        } catch (_: Exception) {
            Toast.makeText(context, context.getString(R.string.grade2Error) + " $sTmp", Toast.LENGTH_LONG).show()
        }
        sTmp = preferences.getString(SettingsActivity.KEY_PREF_HYPERGRADE3, context.getString(R.string.BLOODPRESSURE_GRADE3_VALUES)) ?: ""
        try {
            val grade = sTmp.split(gradeSettingsDivider).toTypedArray()
            hyperGrade3Sys = grade[0].toInt()
            hyperGrade3Dia = grade[1].toInt()
        } catch (_: Exception) {
            Toast.makeText(context, context.getString(R.string.grade3Error) + " $sTmp", Toast.LENGTH_LONG).show()
        }
        sTmp = preferences.getString(SettingsActivity.KEY_PREF_HYPOTENSION, context.getString(R.string.BLOODPRESSURE_HYPOTENSION_VALUES)) ?: ""
        try {
            val grade = sTmp.split(gradeSettingsDivider).toTypedArray()
            hypotensionSys = grade[0].toInt()
            hypotensionDia = grade[1].toInt()
        } catch (_: Exception) {
            Toast.makeText(context, context.getString(R.string.hypotensionError) + " $sTmp", Toast.LENGTH_LONG).show()
        }
    }

    fun sysGrade(systolic: String): Int {
        val sys = systolic.toIntOrNull() ?: 0

        if (sys >= hyperGrade3Sys) return hyperGrade3
        if (sys >= hyperGrade2Sys) return hyperGrade2
        if (sys >= hyperGrade1Sys) return hyperGrade1
        if (sys <= hypotensionSys) return hypotension
        return 0
    }

    fun diaGrade(diastolic: String): Int {
        val dia = diastolic.toIntOrNull() ?: 0

        if (dia >= hyperGrade3Dia) return hyperGrade3
        if (dia >= hyperGrade2Dia) return hyperGrade2
        if (dia >= hyperGrade1Dia) return hyperGrade1
        if (dia <= hypotensionDia) return hypotension
        return 0
    }

    // Migrate , based settings of pre v3.0 settings to more logical | separator
    private fun migrateThresholds() {
        val sTmp1 = preferences.getString(SettingsActivity.KEY_PREF_HYPERGRADE1, context.getString(R.string.BLOODPRESSURE_GRADE1_VALUES)).toString()
        val sTmp2 = preferences.getString(SettingsActivity.KEY_PREF_HYPERGRADE2, context.getString(R.string.BLOODPRESSURE_GRADE2_VALUES)).toString()
        val sTmp3 = preferences.getString(SettingsActivity.KEY_PREF_HYPERGRADE3, context.getString(R.string.BLOODPRESSURE_GRADE3_VALUES)).toString()
        val sTmp4 = preferences.getString(SettingsActivity.KEY_PREF_HYPOTENSION, context.getString(R.string.BLOODPRESSURE_HYPOTENSION_VALUES)).toString()

        preferences.edit() {
            putString(SettingsActivity.KEY_PREF_HYPERGRADE1, sTmp1.replace(",", gradeSettingsDivider))
            putString(SettingsActivity.KEY_PREF_HYPERGRADE2, sTmp2.replace(",", gradeSettingsDivider))
            putString(SettingsActivity.KEY_PREF_HYPERGRADE3, sTmp3.replace(",", gradeSettingsDivider))
            putString(SettingsActivity.KEY_PREF_HYPOTENSION, sTmp4.replace(",", gradeSettingsDivider))
        }
    }
}

// Function to get a timestamp N months ago at 00:00
fun getPastMonthTimestamp(monthsAgo: Int): Long {
    val cal = Calendar.getInstance()
    cal.add(Calendar.MONTH, -monthsAgo)    // Move N months ago
    cal.set(Calendar.HOUR_OF_DAY, 0)       // Set time to 00:00:00
    cal.set(Calendar.MINUTE, 0)
    cal.set(Calendar.SECOND, 0)
    cal.set(Calendar.MILLISECOND, 0)
    return cal.timeInMillis
}

fun collectData(viewModel: DataViewModel, activity: Activity, arr: ArrayList<String>): ArrayList<String>  {
    if (viewModel.getSize(false) == 0) return arr

    val oneMonthAgo = getPastMonthTimestamp(1)
    val threeMonthsAgo = getPastMonthTimestamp(3)
    val twelfMonthsAgo = getPastMonthTimestamp(12)

    // Measurements
    var totalMeasurements = 0 //viewModel.getSize(false)
    var oneMonthMeasurements = 0
    var threeMonthMeasurements = 0
    var twelfMonthMeasurements = 0

    var sysMinY = 1000
    var sysAvgY = 0f
    var sysMaxY = 0
    var diaMinY = 1000
    var diaAvgY = 0f
    var diaMaxY = 0
    var pulseMinY = 1000
    var pulseAvgY = 0f
    var pulseMaxY = 0
    var mapMinY = 1000f
    var mapAvgY = 0f
    var mapMaxY = 0f

    var sysMin3M = 1000
    var sysAvg3M = 0f
    var sysMax3M = 0
    var diaMin3M = 1000
    var diaAvg3M = 0f
    var diaMax3M = 0
    var pulseMin3M = 1000
    var pulseAvg3M = 0f
    var pulseMax3M = 0
    var mapMin3M = 1000f
    var mapAvg3M = 0f
    var mapMax3M = 0f

    var sysMinM = 1000
    var sysAvgM = 0f
    var sysMaxM = 0
    var diaMinM = 1000
    var diaAvgM = 0f
    var diaMaxM = 0
    var pulseMinM = 1000
    var pulseAvgM = 0f
    var pulseMaxM = 0
    var mapMinM = 1000f
    var mapAvgM = 0f
    var mapMaxM = 0f

    // Need to iterate instead of SQL commands
    val items = viewModel.getItems("DESC", false)
    for (item in items) {
        if (item.type != BLOODPRESSURE) continue // Ignore Diary items potentially blended in
        totalMeasurements += 1
        val sysValue = try {
            item.value1.toInt()
        } catch (_: Exception) {
            0
        }
        val diaValue = try {
            item.value2.toInt()
        } catch (_: Exception) {
            0
        }
        val pulseValue = try {
            item.value3.toInt()
        } catch (_: Exception) {
            0
        }

        // Year
        if (item.timestamp >= twelfMonthsAgo) {
            twelfMonthMeasurements += 1
            sysMinY = min(sysMinY, sysValue)
            sysMaxY = max(sysMaxY, sysValue)
            diaMinY = min(diaMinY, diaValue)
            diaMaxY = max(diaMaxY, diaValue)
            pulseMinY = min(pulseMinY, pulseValue)
            pulseMaxY = max(pulseMaxY, pulseValue)

            var mapValue: Float = diaValue + (sysValue - diaValue).toFloat() / 3f
            mapMinY = min(mapMinY, mapValue)
            mapMaxY = max(mapMaxY, mapValue)

            sysAvgY += sysValue
            diaAvgY += diaValue
            pulseAvgY += pulseValue
            mapAvgY += mapValue

            if (item.timestamp >= threeMonthsAgo) {
                // 3 Months
                threeMonthMeasurements += 1
                sysMin3M = min(sysMin3M, sysValue)
                sysMax3M = max(sysMax3M, sysValue)
                diaMin3M = min(diaMin3M, diaValue)
                diaMax3M = max(diaMax3M, diaValue)
                pulseMin3M = min(pulseMin3M, pulseValue)
                pulseMax3M = max(pulseMax3M, pulseValue)

                mapValue = diaValue + (sysValue - diaValue).toFloat() / 3f
                mapMin3M = min(mapMin3M, mapValue)
                mapMax3M = max(mapMax3M, mapValue)

                sysAvg3M += sysValue
                diaAvg3M += diaValue
                pulseAvg3M += pulseValue
                mapAvg3M += mapValue
            }

            if (item.timestamp >= oneMonthAgo) {
                // Month
                oneMonthMeasurements += 1
                sysMinM = min(sysMinM, sysValue)
                sysMaxM = max(sysMaxM, sysValue)
                diaMinM = min(diaMinM, diaValue)
                diaMaxM = max(diaMaxM, diaValue)
                pulseMinM = min(pulseMinM, pulseValue)
                pulseMaxM = max(pulseMaxM, pulseValue)

                mapValue = diaValue + (sysValue - diaValue).toFloat() / 3f
                mapMinM = min(mapMinM, mapValue)
                mapMaxM = max(mapMaxM, mapValue)

                sysAvgM += sysValue
                diaAvgM += diaValue
                pulseAvgM += pulseValue
                mapAvgM += mapValue
            }
        }
    }
    // Average
    sysAvgY = sysAvgY/twelfMonthMeasurements
    diaAvgY = diaAvgY/twelfMonthMeasurements
    pulseAvgY = pulseAvgY/twelfMonthMeasurements
    mapAvgY = mapAvgY/twelfMonthMeasurements

    sysAvg3M = sysAvg3M/threeMonthMeasurements
    diaAvg3M = diaAvg3M/threeMonthMeasurements
    pulseAvg3M = pulseAvg3M/threeMonthMeasurements
    mapAvg3M = mapAvg3M/threeMonthMeasurements

    sysAvgM = sysAvgM/oneMonthMeasurements
    diaAvgM = diaAvgM/oneMonthMeasurements
    pulseAvgM = pulseAvgM/oneMonthMeasurements
    mapAvgM = mapAvgM/oneMonthMeasurements

    arr.add(activity.getString(R.string.measurementLabel) + "|" + oneMonthMeasurements.toString() + "|" + threeMonthMeasurements.toString() + "|" + twelfMonthMeasurements.toString())

    arr.add("||||") // Blank row
    arr.add("header|" + activity.getString( R.string.systolic) + "|||") // Header
    arr.add(activity.getString(R.string.min) + "|$sysMinM|$sysMin3M|$sysMinY")
    var s = "%.1f".format(sysAvgM) + "|" +
            "%.1f".format(sysAvg3M) + "|" +
            "%.1f".format(sysAvgY)
    arr.add(activity.getString(R.string.avg) + "|$s")
    arr.add(activity.getString(R.string.max) + "|$sysMaxM|$sysMax3M|$sysMaxY")

    arr.add("||||") // Blank row
    arr.add("header|" + activity.getString( R.string.diastolic) + "|||") // Header
    arr.add(activity.getString(R.string.min) + "|$diaMinM|$diaMin3M|$diaMinY")
    s = "%.1f".format(diaAvgM) + "|" +
        "%.1f".format(diaAvg3M) + "|" +
        "%.1f".format(diaAvgY)
    arr.add(activity.getString(R.string.avg) + "|$s")
    arr.add(activity.getString(R.string.max) + "|$diaMaxM|$diaMax3M|$diaMaxY")

    arr.add("||||") // Blank row
    arr.add("header|" + activity.getString( R.string.pulse) + "|||") // Header
    arr.add(activity.getString(R.string.min) + "|$pulseMinM|$pulseMin3M|$pulseMinY")
    s = "%.1f".format(pulseAvgM) + "|" +
            "%.1f".format(pulseAvg3M) + "|" +
            "%.1f".format(pulseAvgY)
    arr.add(activity.getString(R.string.avg) + "|$s")
    arr.add(activity.getString(R.string.max) + "|$pulseMaxM|$pulseMax3M|$pulseMaxY")

    // MAP mean arterial pressure, MAP = diastolischer Druck + (systolischer Druck – diastolischer Druck) / 3
    arr.add("||||") // Blank row
    arr.add("header|" + activity.getString( R.string.map) + "|||") // Header
    s = "%.1f".format(mapMinM) + "|" +
            "%.1f".format(mapMin3M) + "|" +
            "%.1f".format(mapMinY)
    arr.add(activity.getString(R.string.min) + "|$s")

    s = "%.1f".format(mapAvgM) + "|" +
        "%.1f".format(mapAvg3M) + "|" +
        "%.1f".format(mapAvgY)
    arr.add(activity.getString(R.string.avg) + "|$s")
    s = "%.1f".format(mapMaxM) + "|" +
            "%.1f".format(mapMax3M) + "|" +
            "%.1f".format(mapMaxY)
    arr.add(activity.getString(R.string.max) + "|$s")

    arr.add("footer|" + activity.getString(R.string.mapFormula) +"|||") // Footer

    return arr
}