// Copyright (c) 2017, the Dart project authors.  Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.
import 'dart:async';
import 'dart:convert';
import 'dart:io';

import 'package:build/build.dart';
import 'package:build/experiments.dart';
import 'package:build_config/build_config.dart';
import 'package:build_runner/src/asset/reader_writer.dart';
import 'package:build_runner/src/asset_graph/graph.dart';
import 'package:build_runner/src/build_script_generate/build_process_state.dart';
import 'package:build_runner/src/generate/build_definition.dart';
import 'package:build_runner/src/generate/build_phases.dart';
import 'package:build_runner/src/generate/exceptions.dart';
import 'package:build_runner/src/generate/phase.dart';
import 'package:build_runner/src/logging/build_log.dart';
import 'package:build_runner/src/options/testing_overrides.dart';
import 'package:build_runner/src/package_graph/package_graph.dart';
import 'package:build_runner/src/package_graph/target_graph.dart';
import 'package:build_runner/src/util/constants.dart';
import 'package:built_collection/built_collection.dart';
import 'package:crypto/crypto.dart';
import 'package:logging/logging.dart';
import 'package:package_config/package_config.dart';
import 'package:path/path.dart' as p;
import 'package:test/test.dart';
import 'package:test_descriptor/test_descriptor.dart' as d;
import 'package:watcher/watcher.dart';

import '../common/common.dart';

void main() {
  final languageVersion = LanguageVersion(2, 0);

  setUp(() {
    BuildLog.resetForTests(printOnFailure: printOnFailure);
  });

  group('BuildDefinition.prepareWorkspace', () {
    late String pkgARoot;
    late String pkgBRoot;
    late PackageGraph aPackageGraph;

    late PackageGraph packageGraph;
    late TargetGraph targetGraph;
    late ReaderWriter readerWriter;

    Future<File> createFile(String path, dynamic contents) async {
      final file = File(p.join(pkgARoot, path));
      expect(await file.exists(), isFalse);
      await file.create(recursive: true);
      if (contents is String) {
        await file.writeAsString(contents);
      } else {
        await file.writeAsBytes(contents as List<int>);
      }
      addTearDown(() async => await file.exists() ? await file.delete() : null);
      return file;
    }

    Future<void> deleteFile(String path) async {
      final file = File(p.join(pkgARoot, path));
      expect(await file.exists(), isTrue);
      await file.delete();
    }

    Future<void> modifyFile(String path, String contents) async {
      final file = File(p.join(pkgARoot, path));
      expect(await file.exists(), isTrue);
      await file.writeAsString(contents);
    }

    Future<String> readFile(String path) async {
      final file = File(p.join(pkgARoot, path));
      expect(await file.exists(), isTrue);
      return file.readAsString();
    }

    setUp(() async {
      pkgARoot = p.join(d.sandbox, 'pkg_a');
      pkgBRoot = p.join(d.sandbox, 'pkg_b');
      aPackageGraph = buildPackageGraph({
        rootPackage('a', languageVersion: languageVersion, path: pkgARoot): [
          'b',
        ],
        package('b', languageVersion: languageVersion, path: pkgBRoot): [],
      });
      await d.dir('pkg_a', [
        await pubspec('a'),
        d.file('pubspec.lock', 'packages: {}'),
        d.dir('.dart_tool', [
          d.dir('build', [
            d.dir('entrypoint', [d.file('build.dart', '// builds!')]),
          ]),
          d.file(
            'package_config.json',
            jsonEncode({
              'configVersion': 2,
              'packages': [
                {
                  'name': 'a',
                  'rootUri': p.toUri(pkgARoot).toString(),
                  'packageUri': 'lib/',
                  'languageVersion': languageVersion.toString(),
                },
                {
                  'name': 'b',
                  'rootUri': p.toUri(pkgBRoot).toString(),
                  'packageUri': 'lib/',
                  'languageVersion': languageVersion.toString(),
                },
              ],
            }),
          ),
        ]),
        d.file('build.yaml', '''
targets:
  \$default:
    sources:
      include:
        - lib/**
        - does_not_exist/**
      exclude:
        - lib/excluded/**
'''),
        d.dir('lib'),
      ]).create();
      await d.dir('pkg_b', [
        await pubspec('b'),
        d.file('build.yaml', '''
targets:
  \$default:
    sources:
      - lib/**
      - test/**
'''),
        d.dir('test', [d.file('some_test.dart')]),
        d.dir('lib', [d.file('some_lib.dart')]),
      ]).create();

      packageGraph = await PackageGraph.forPath(pkgARoot);
      readerWriter = ReaderWriter(packageGraph);
      targetGraph = await TargetGraph.forPackageGraph(
        packageGraph: packageGraph,
        readerWriter: readerWriter,
      );
    });

    group('reports updates', () {
      test('for deleted source and generated nodes', () async {
        await createFile(p.join('lib', 'a.txt'), 'a');
        await createFile(p.join('lib', 'b.txt'), 'b');
        final buildPhases = BuildPhases([
          InBuildPhase(TestBuilder(), 'a', hideOutput: false),
        ]);

        final originalAssetGraph = await AssetGraph.build(
          buildPhases,
          {makeAssetId('a|lib/a.txt'), makeAssetId('a|lib/b.txt')},
          <AssetId>{},
          aPackageGraph,
          readerWriter,
        );
        final generatedAId = makeAssetId('a|lib/a.txt.copy');
        originalAssetGraph.updateNode(generatedAId, (nodeBuilder) {
          nodeBuilder.digest = Digest([]);
          nodeBuilder.generatedNodeState.result = true;
        });

        await createFile(assetGraphPath, originalAssetGraph.serialize());

        await deleteFile(p.join('lib', 'b.txt'));
        final buildDefinition = await BuildDefinition.prepareWorkspace(
          packageGraph: packageGraph,
          targetGraph: targetGraph,
          readerWriter: readerWriter,
          buildPhases: buildPhases,
          skipBuildScriptCheck: true,
        );

        expect(buildDefinition.updates![generatedAId], ChangeType.REMOVE);
        expect(
          buildDefinition.updates![makeAssetId('a|lib/b.txt')],
          ChangeType.REMOVE,
        );
      });

      test('for new sources', () async {
        final buildPhases = BuildPhases([
          InBuildPhase(TestBuilder(), 'a', hideOutput: true),
        ]);

        final originalAssetGraph = await AssetGraph.build(
          buildPhases,
          <AssetId>{},
          <AssetId>{},
          aPackageGraph,
          readerWriter,
        );

        await createFile(assetGraphPath, originalAssetGraph.serialize());

        await createFile(p.join('lib', 'a.txt'), 'a');
        final buildDefinition = await BuildDefinition.prepareWorkspace(
          packageGraph: packageGraph,
          targetGraph: targetGraph,
          readerWriter: readerWriter,
          buildPhases: buildPhases,
          skipBuildScriptCheck: true,
        );

        expect(
          buildDefinition.updates![makeAssetId('a|lib/a.txt')],
          ChangeType.ADD,
        );
      });

      test('for changed sources', () async {
        final aTxt = AssetId('a', 'lib/a.txt');
        final aTxtCopy = AssetId('a', 'lib/a.txt.copy');
        await createFile(p.join('lib', 'a.txt'), 'a');
        final buildPhases = BuildPhases([
          InBuildPhase(TestBuilder(), 'a', hideOutput: true),
        ]);

        final originalAssetGraph = await AssetGraph.build(
          buildPhases,
          {aTxt},
          <AssetId>{},
          aPackageGraph,
          readerWriter,
        );

        // pretend a build happened
        originalAssetGraph.updateNode(aTxtCopy, (nodeBuilder) {
          nodeBuilder.generatedNodeState.inputs.add(aTxt);
        });
        await createFile(assetGraphPath, originalAssetGraph.serialize());

        await modifyFile(p.join('lib', 'a.txt'), 'b');
        final buildDefinition = await BuildDefinition.prepareWorkspace(
          packageGraph: packageGraph,
          targetGraph: targetGraph,
          readerWriter: readerWriter,
          buildPhases: buildPhases,
          skipBuildScriptCheck: true,
        );

        expect(
          buildDefinition.updates![makeAssetId('a|lib/a.txt')],
          ChangeType.MODIFY,
        );
      });

      test('retains non-output generated nodes', () async {
        await createFile(p.join('lib', 'test.txt'), 'a');
        final buildPhases = BuildPhases([
          InBuildPhase(TestBuilder(build: (_, _) {}), 'a', hideOutput: true),
        ]);

        final originalAssetGraph = await AssetGraph.build(
          buildPhases,
          {makeAssetId('a|lib/test.txt')},
          <AssetId>{},
          aPackageGraph,
          readerWriter,
        );
        final generatedSrcId = makeAssetId('a|lib/test.txt.copy');
        originalAssetGraph.updateNode(generatedSrcId, (nodeBuilder) {
          nodeBuilder.digest = null;
          nodeBuilder.generatedNodeState.result = true;
        });

        await createFile(assetGraphPath, originalAssetGraph.serialize());

        final buildDefinition = await BuildDefinition.prepareWorkspace(
          packageGraph: packageGraph,
          targetGraph: targetGraph,
          readerWriter: readerWriter,
          buildPhases: buildPhases,
          skipBuildScriptCheck: true,
        );
        expect(buildDefinition.assetGraph.contains(generatedSrcId), isTrue);
      });

      test('for changed BuilderOptions', () async {
        await createFile(p.join('lib', 'a.txt'), 'a');
        await createFile(p.join('lib', 'a.txt.copy'), 'a');
        await createFile(p.join('lib', 'a.txt.clone'), 'a');
        final inputSources = const InputSet(include: ['lib/a.txt']);
        final buildPhases = BuildPhases([
          InBuildPhase(
            TestBuilder(),
            'a',
            hideOutput: false,
            targetSources: inputSources,
          ),
          InBuildPhase(
            TestBuilder(buildExtensions: appendExtension('.clone')),
            'a',
            targetSources: inputSources,
            hideOutput: false,
          ),
        ]);

        final originalAssetGraph = await AssetGraph.build(
          buildPhases,
          {makeAssetId('a|lib/a.txt')},
          <AssetId>{},
          aPackageGraph,
          readerWriter,
        );
        final generatedACopyId = makeAssetId('a|lib/a.txt.copy');
        final generatedACloneId = makeAssetId('a|lib/a.txt.clone');
        for (final id in [generatedACopyId, generatedACloneId]) {
          originalAssetGraph.updateNode(id, (nodeBuilder) {
            nodeBuilder.digest = Digest([]);
            nodeBuilder.generatedNodeState.result = true;
          });
        }

        await createFile(assetGraphPath, originalAssetGraph.serialize());

        // Same as before, but change the `BuilderOptions` for the first phase.
        final newBuildPhases = BuildPhases([
          InBuildPhase(
            TestBuilder(),
            'a',
            builderOptions: const BuilderOptions({'test': 'option'}),
            targetSources: inputSources,
            hideOutput: false,
          ),
          InBuildPhase(
            TestBuilder(buildExtensions: appendExtension('.clone')),
            'a',
            targetSources: inputSources,
            hideOutput: false,
          ),
        ]);
        final buildDefinition = await BuildDefinition.prepareWorkspace(
          packageGraph: packageGraph,
          targetGraph: targetGraph,
          readerWriter: readerWriter,
          buildPhases: newBuildPhases,
          skipBuildScriptCheck: true,
        );

        final newAssetGraph = buildDefinition.assetGraph;
        expect(
          newAssetGraph.inBuildPhasesOptionsDigests[0],
          isNot(newAssetGraph.previousInBuildPhasesOptionsDigests![0]),
        );
        expect(
          newAssetGraph.inBuildPhasesOptionsDigests[1],
          newAssetGraph.previousInBuildPhasesOptionsDigests![1],
        );
      });
    });

    group('assetGraph', () {
      test('doesn\'t capture unrecognized cacheDir files as inputs', () async {
        final generatedId = AssetId(
          'a',
          p.url.join(generatedOutputDirectory, 'a', 'lib', 'test.txt'),
        );
        await createFile(generatedId.path, 'a');
        final buildPhases = BuildPhases([
          InBuildPhase(
            TestBuilder(
              buildExtensions: appendExtension('.copy', from: '.txt'),
            ),
            'a',
            hideOutput: true,
          ),
        ]);

        final assetGraph = await AssetGraph.build(
          buildPhases,
          <AssetId>{},
          <AssetId>{},
          aPackageGraph,
          readerWriter,
        );
        final expectedIds = placeholderIdsFor(aPackageGraph);
        expect(
          assetGraph.allNodes.map((node) => node.id),
          unorderedEquals(expectedIds),
        );

        await createFile(assetGraphPath, assetGraph.serialize());

        final buildDefinition = await BuildDefinition.prepareWorkspace(
          packageGraph: packageGraph,
          targetGraph: targetGraph,
          readerWriter: readerWriter,
          buildPhases: buildPhases,
          skipBuildScriptCheck: true,
        );

        expect(buildDefinition.assetGraph.contains(generatedId), isFalse);
      });

      test('includes generated entrypoint', () async {
        final entryPoint = AssetId(
          'a',
          p.url.join(entryPointDir, 'build.dart'),
        );
        final buildDefinition = await BuildDefinition.prepareWorkspace(
          packageGraph: packageGraph,
          targetGraph: targetGraph,
          readerWriter: readerWriter,
          buildPhases: BuildPhases([]),
          skipBuildScriptCheck: true,
        );
        expect(buildDefinition.assetGraph.contains(entryPoint), isTrue);
      });

      test('does not run Builders on generated entrypoint', () async {
        final entryPoint = AssetId(
          'a',
          p.url.join(entryPointDir, 'build.dart'),
        );
        final buildPhases = BuildPhases([
          InBuildPhase(TestBuilder(), 'a', hideOutput: true),
        ]);
        final buildDefinition = await BuildDefinition.prepareWorkspace(
          packageGraph: packageGraph,
          targetGraph: targetGraph,
          readerWriter: readerWriter,
          buildPhases: buildPhases,
          skipBuildScriptCheck: true,
        );
        expect(
          buildDefinition.assetGraph.contains(entryPoint.addExtension('.copy')),
          isFalse,
        );
      });

      test('does\'nt include sources not matching the target glob', () async {
        await createFile(p.join('lib', 'a.txt'), 'a');
        await createFile(p.join('lib', 'excluded', 'b.txt'), 'b');

        final buildPhases = BuildPhases([InBuildPhase(TestBuilder(), 'a')]);
        final buildDefinition = await BuildDefinition.prepareWorkspace(
          packageGraph: packageGraph,
          targetGraph: targetGraph,
          readerWriter: readerWriter,
          buildPhases: buildPhases,
          skipBuildScriptCheck: true,
        );
        final assetGraph = buildDefinition.assetGraph;
        expect(assetGraph.contains(AssetId('a', 'lib/a.txt')), isTrue);
        expect(
          assetGraph.contains(AssetId('a', 'lib/excluded/b.txt')),
          isFalse,
        );
      });

      test('does\'nt include non-lib sources in targets in deps', () async {
        final buildDefinition = await BuildDefinition.prepareWorkspace(
          packageGraph: packageGraph,
          targetGraph: targetGraph,
          readerWriter: readerWriter,
          buildPhases: BuildPhases([]),
          skipBuildScriptCheck: true,
        );
        final assetGraph = buildDefinition.assetGraph;
        expect(assetGraph.contains(AssetId('b', 'lib/some_lib.dart')), isTrue);
        expect(
          assetGraph.contains(AssetId('b', 'test/some_test.dart')),
          isFalse,
        );
      });
    });

    group('invalidation', () {
      final logs = <LogRecord>[];
      setUp(() async {
        logs.clear();
        buildLog.configuration = buildLog.configuration.rebuild((b) {
          b.onLog = logs.add;
        });
      });

      test('invalidates the graph when adding a build phase', () async {
        var buildPhases = BuildPhases([
          InBuildPhase(TestBuilder(), 'a', hideOutput: true),
        ]);

        final originalAssetGraph = await AssetGraph.build(
          buildPhases,
          <AssetId>{},
          <AssetId>{},
          aPackageGraph,
          readerWriter,
        );

        await createFile(assetGraphPath, originalAssetGraph.serialize());

        buildPhases = BuildPhases([
          ...buildPhases.inBuildPhases,
          InBuildPhase(
            TestBuilder(),
            'a',
            targetSources: const InputSet(include: ['.copy']),
            hideOutput: true,
          ),
        ]);

        await expectLater(
          () => BuildDefinition.prepareWorkspace(
            packageGraph: packageGraph,
            targetGraph: targetGraph,
            readerWriter: readerWriter,
            buildPhases: buildPhases,
            skipBuildScriptCheck: true,
          ),
          throwsA(const TypeMatcher<BuildScriptChangedException>()),
        );
        expect(
          buildProcessState.fullBuildReason,
          FullBuildReason.incompatibleBuild,
        );
        expect(File(assetGraphPath).existsSync(), isFalse);
      });

      test(
        'invalidates the graph if a phase has different build extension',
        () async {
          var buildPhases = BuildPhases([
            InBuildPhase(TestBuilder(), 'a', hideOutput: true),
          ]);

          final originalAssetGraph = await AssetGraph.build(
            buildPhases,
            <AssetId>{},
            <AssetId>{},
            aPackageGraph,
            readerWriter,
          );

          await createFile(assetGraphPath, originalAssetGraph.serialize());

          buildPhases = BuildPhases([
            InBuildPhase(
              TestBuilder(buildExtensions: appendExtension('different')),
              'a',
              hideOutput: true,
            ),
          ]);

          await expectLater(
            () => BuildDefinition.prepareWorkspace(
              packageGraph: packageGraph,
              targetGraph: targetGraph,
              readerWriter: readerWriter,
              buildPhases: buildPhases,
              skipBuildScriptCheck: true,
            ),
            throwsA(const TypeMatcher<BuildScriptChangedException>()),
          );
          expect(
            buildProcessState.fullBuildReason,
            FullBuildReason.incompatibleBuild,
          );
          expect(File(assetGraphPath).existsSync(), isFalse);
        },
      );

      test('invalidates the graph if the dart sdk version changes', () async {
        final buildPhases = BuildPhases([
          InBuildPhase(TestBuilder(), 'a', hideOutput: true),
        ]);

        final originalAssetGraph = await AssetGraph.build(
          buildPhases,
          <AssetId>{},
          <AssetId>{},
          aPackageGraph,
          readerWriter,
        );

        final bytes = originalAssetGraph.serialize();
        final serialized =
            json.decode(utf8.decode(bytes)) as Map<String, dynamic>;
        serialized['dart_version'] = 'some_fake_version';
        final encoded = utf8.encode(json.encode(serialized));
        await createFile(assetGraphPath, encoded);

        await expectLater(
          () => BuildDefinition.prepareWorkspace(
            packageGraph: packageGraph,
            targetGraph: targetGraph,
            readerWriter: readerWriter,
            buildPhases: buildPhases,
            skipBuildScriptCheck: true,
          ),
          throwsA(const TypeMatcher<BuildScriptChangedException>()),
        );

        expect(
          buildProcessState.fullBuildReason,
          FullBuildReason.incompatibleBuild,
        );
      });

      test(
        'does not invalidate if a different Builder has the same extensions',
        () async {
          var buildPhases = BuildPhases([
            InBuildPhase(
              TestBuilder(),
              'a',
              builderKey: 'testbuilder',
              hideOutput: true,
              builderOptions: const BuilderOptions({'foo': 'bar'}),
            ),
          ]);

          final originalAssetGraph = await AssetGraph.build(
            buildPhases,
            <AssetId>{},
            <AssetId>{},
            aPackageGraph,
            readerWriter,
          );

          await createFile(assetGraphPath, originalAssetGraph.serialize());

          buildPhases = BuildPhases([
            InBuildPhase(
              DelegatingBuilder(TestBuilder()),
              'a',
              builderKey: 'testbuilder',
              hideOutput: true,
              builderOptions: const BuilderOptions({'baz': 'zap'}),
            ),
          ]);
          logs.clear();

          final buildDefinition = await BuildDefinition.prepareWorkspace(
            packageGraph: packageGraph,
            targetGraph: targetGraph,
            readerWriter: readerWriter,
            buildPhases: buildPhases,
            skipBuildScriptCheck: true,
          );
          expect(
            logs.any(
              (log) =>
                  log.level == Level.WARNING &&
                  log.message.contains('build phases have changed'),
            ),
            isFalse,
          );

          final newAssetGraph = buildDefinition.assetGraph;
          expect(
            originalAssetGraph.buildPhasesDigest,
            equals(newAssetGraph.buildPhasesDigest),
          );
        },
      );
      test(
        'does not invalidate the graph if the BuilderOptions change',
        () async {
          var buildPhases = BuildPhases([
            InBuildPhase(
              TestBuilder(),
              'a',
              hideOutput: true,
              builderOptions: const BuilderOptions({'foo': 'bar'}),
            ),
          ]);

          final originalAssetGraph = await AssetGraph.build(
            buildPhases,
            <AssetId>{},
            <AssetId>{},
            aPackageGraph,
            readerWriter,
          );

          await createFile(assetGraphPath, originalAssetGraph.serialize());

          buildPhases = BuildPhases([
            InBuildPhase(
              TestBuilder(),
              'a',
              hideOutput: true,
              builderOptions: const BuilderOptions({'baz': 'zap'}),
            ),
          ]);
          logs.clear();

          final buildDefinition = await BuildDefinition.prepareWorkspace(
            packageGraph: packageGraph,
            targetGraph: targetGraph,
            readerWriter: readerWriter,
            buildPhases: buildPhases,
            skipBuildScriptCheck: true,
          );
          expect(
            logs.any(
              (log) =>
                  log.level == Level.WARNING &&
                  log.message.contains('build phases have changed'),
            ),
            isFalse,
          );

          final newAssetGraph = buildDefinition.assetGraph;
          expect(
            originalAssetGraph.buildPhasesDigest,
            equals(newAssetGraph.buildPhasesDigest),
          );
        },
      );

      test('deletes old source outputs if the build phases change', () async {
        var buildPhases = BuildPhases([
          InBuildPhase(TestBuilder(), 'a', hideOutput: false),
        ]);
        final aTxt = AssetId('a', 'lib/a.txt');
        await createFile(aTxt.path, 'hello');

        final originalAssetGraph = await AssetGraph.build(
          buildPhases,
          <AssetId>{aTxt},
          <AssetId>{},
          aPackageGraph,
          readerWriter,
        );

        final aTxtCopy = AssetId('a', 'lib/a.txt.copy');
        // Pretend we already output this without actually running a build.
        originalAssetGraph.updateNode(aTxtCopy, (nodeBuilder) {
          nodeBuilder.digest = Digest([]);
        });
        await createFile(aTxtCopy.path, 'hello');

        await createFile(assetGraphPath, originalAssetGraph.serialize());

        buildPhases = BuildPhases([
          ...buildPhases.inBuildPhases,
          InBuildPhase(
            TestBuilder(),
            'a',
            targetSources: const InputSet(include: ['.copy']),
            hideOutput: true,
          ),
        ]);

        expect(await readerWriter.canRead(aTxtCopy), true);
        await expectLater(
          () => BuildDefinition.prepareWorkspace(
            packageGraph: packageGraph,
            targetGraph: targetGraph,
            readerWriter: readerWriter,
            buildPhases: buildPhases,
            skipBuildScriptCheck: true,
          ),
          throwsA(const TypeMatcher<BuildScriptChangedException>()),
        );
        expect(await readerWriter.canRead(aTxtCopy), false);
      });

      test('invalidates the graph if the root package name changes', () async {
        var buildPhases = BuildPhases([
          InBuildPhase(TestBuilder(), 'a', hideOutput: false),
        ]);
        final aTxt = AssetId('a', 'lib/a.txt');
        await createFile(aTxt.path, 'hello');

        final originalAssetGraph = await AssetGraph.build(
          buildPhases,
          <AssetId>{aTxt},
          <AssetId>{},
          aPackageGraph,
          readerWriter,
        );

        final aTxtCopy = AssetId('a', 'lib/a.txt.copy');
        // Pretend we already output this without actually running a build.
        originalAssetGraph.updateNode(aTxtCopy, (nodeBuilder) {
          nodeBuilder.digest = Digest([]);
        });
        await createFile(aTxtCopy.path, 'hello');

        await createFile(assetGraphPath, originalAssetGraph.serialize());

        await modifyFile(
          'pubspec.yaml',
          (await readFile('pubspec.yaml')).replaceFirst('name: a', 'name: c'),
        );
        await modifyFile(
          '.dart_tool/package_config.json',
          (await readFile(
            '.dart_tool/package_config.json',
          )).replaceFirst('"name":"a"', '"name":"c"'),
        );

        packageGraph = await PackageGraph.forPath(pkgARoot);
        readerWriter = ReaderWriter(packageGraph);
        buildPhases = BuildPhases([
          InBuildPhase(TestBuilder(), 'c', hideOutput: false),
        ]);

        expect(await readerWriter.canRead(AssetId('c', aTxtCopy.path)), true);
        await expectLater(
          () => BuildDefinition.prepareWorkspace(
            packageGraph: packageGraph,
            targetGraph: targetGraph,
            readerWriter: readerWriter,
            buildPhases: buildPhases,
            skipBuildScriptCheck: true,
          ),
          throwsA(const TypeMatcher<BuildScriptChangedException>()),
        );
        expect(await readerWriter.canRead(AssetId('c', aTxtCopy.path)), false);
      });

      test(
        'invalidates the graph if the language version of a package changes',
        () async {
          final assetGraph = await AssetGraph.build(
            BuildPhases([]),
            <AssetId>{},
            {AssetId('a', '.dart_tool/package_config.json')},
            aPackageGraph,
            readerWriter,
          );

          final graph = await createFile(
            assetGraphPath,
            assetGraph.serialize(),
          );

          await modifyFile(
            '.dart_tool/package_config.json',
            jsonEncode({
              'configVersion': 2,
              'packages': [
                {
                  'name': 'a',
                  'rootUri': p.toUri(pkgARoot).toString(),
                  'packageUri': 'lib/',
                  'languageVersion': languageVersion.toString(),
                },
                {
                  'name': 'b',
                  'rootUri': p.toUri(pkgBRoot).toString(),
                  'packageUri': 'lib/',
                  'languageVersion':
                      LanguageVersion(
                        languageVersion.major,
                        languageVersion.minor + 1,
                      ).toString(),
                },
              ],
            }),
          );

          packageGraph = await PackageGraph.forPath(pkgARoot);

          await expectLater(
            () => BuildDefinition.prepareWorkspace(
              packageGraph: packageGraph,
              targetGraph: targetGraph,
              readerWriter: readerWriter,
              buildPhases: BuildPhases([]),
              skipBuildScriptCheck: true,
            ),
            throwsA(const TypeMatcher<BuildScriptChangedException>()),
          );

          expect(graph.existsSync(), isFalse);
        },
      );

      test('invalidates the graph if the enabled experiments change', () async {
        AssetGraph assetGraph;
        assetGraph = await withEnabledExperiments(
          () => AssetGraph.build(
            BuildPhases([]),
            <AssetId>{},
            <AssetId>{},
            aPackageGraph,
            readerWriter,
          ),
          ['a'],
        );

        final graph = await createFile(assetGraphPath, assetGraph.serialize());
        packageGraph = aPackageGraph;

        await expectLater(
          () => BuildDefinition.prepareWorkspace(
            packageGraph: packageGraph,
            targetGraph: targetGraph,
            readerWriter: readerWriter,
            buildPhases: BuildPhases([]),
            skipBuildScriptCheck: true,
          ),
          throwsA(const TypeMatcher<BuildScriptChangedException>()),
        );

        expect(graph.existsSync(), isFalse);
      });
    });

    group('regression tests', () {
      test('load can skip files under the generated dir', () async {
        await createFile(
          p.join('.dart_tool', 'build', 'generated', '.foo'),
          'a',
        );
        expect(
          BuildDefinition.prepareWorkspace(
            packageGraph: packageGraph,
            targetGraph: targetGraph,
            readerWriter: readerWriter,
            buildPhases: BuildPhases([]),
            skipBuildScriptCheck: true,
          ),
          completes,
        );
      });

      // https://github.com/dart-lang/build/issues/1042
      test('a missing sources/include does not cause an error', () async {
        final rootPkg = packageGraph.root.name;
        final targetGraph = await TargetGraph.forPackageGraph(
          packageGraph: packageGraph,
          testingOverrides: TestingOverrides(
            buildConfig:
                {
                  rootPkg: BuildConfig.fromMap(rootPkg, [], {
                    'targets': {
                      'another': <String, dynamic>{},
                      '\$default': {
                        'sources': {
                          'exclude': ['lib/src/**'],
                        },
                      },
                    },
                  }),
                }.build(),
          ),
        );

        expect(
          targetGraph.allModules['$rootPkg:another']!.sourceIncludes,
          isNotEmpty,
        );
        expect(
          targetGraph.allModules['$rootPkg:$rootPkg']!.sourceIncludes,
          isNotEmpty,
        );
      });

      test(
        'a missing sources/include results in the default sources',
        () async {
          final rootPkg = packageGraph.root.name;
          final targetGraph = await TargetGraph.forPackageGraph(
            packageGraph: packageGraph,
            testingOverrides: TestingOverrides(
              buildConfig:
                  {
                    rootPkg: BuildConfig.fromMap(rootPkg, [], {
                      'targets': {
                        'another': <String, dynamic>{},
                        '\$default': {
                          'sources': {
                            'exclude': ['lib/src/**'],
                          },
                        },
                      },
                    }),
                  }.build(),
            ),
          );
          expect(
            targetGraph.allModules['$rootPkg:another']!.sourceIncludes.map(
              (glob) => glob.pattern,
            ),
            defaultRootPackageSources,
          );
          expect(
            targetGraph.allModules['$rootPkg:$rootPkg']!.sourceIncludes.map(
              (glob) => glob.pattern,
            ),
            defaultRootPackageSources,
          );
        },
      );

      test('allows a target config with empty sources list', () async {
        final rootPkg = packageGraph.root.name;
        final targetGraph = await TargetGraph.forPackageGraph(
          packageGraph: packageGraph,
          testingOverrides: TestingOverrides(
            buildConfig:
                {
                  rootPkg: BuildConfig.fromMap(rootPkg, [], {
                    'targets': {
                      'another': <String, dynamic>{},
                      '\$default': {
                        'sources': {'include': <String>[]},
                      },
                    },
                  }),
                }.build(),
          ),
        );
        expect(
          BuildDefinition.prepareWorkspace(
            packageGraph: packageGraph,
            targetGraph: targetGraph,
            readerWriter: readerWriter,
            buildPhases: BuildPhases([]),
            skipBuildScriptCheck: true,
          ),
          completes,
        );
      });
    });
  });
}
