import 'dart:io';
import 'dart:math';
import 'dart:typed_data';

import 'package:analyzer/dart/ast/ast.dart';
import 'package:analyzer/dart/ast/visitor.dart';
import 'package:analyzer/file_system/file_system.dart';
import 'package:analyzer/file_system/overlay_file_system.dart';
import 'package:analyzer/file_system/physical_file_system.dart';
import 'package:analyzer/src/dart/analysis/analysis_context_collection.dart';
import 'package:analyzer/src/dart/analysis/byte_store.dart';
import 'package:analyzer/src/dart/analysis/results.dart';
import 'package:analyzer/src/dart/element/element.dart';
import 'package:analyzer/src/util/file_paths.dart' as file_paths;
import 'package:analyzer/src/util/performance/operation_performance.dart';
import 'package:collection/collection.dart';
import 'package:test/test.dart';

import '../../test/src/summary/element_text.dart';
import '../../test/util/diff.dart';

Future<void> main(List<String> args) async {
  const pkgPath = '/Users/scheglov/Source/Dart/sdk.git/sdk/pkg';
  // args = ['$pkgPath/analyzer'];
  // args = ['$pkgPath/analyzer'];
  args = ['$pkgPath/analyzer_cli'];

  var baseInitByteStore = MemoryByteStore();
  var fineInitByteStore = MemoryByteStore();

  var resourceProvider = OverlayResourceProvider(
    PhysicalResourceProvider.INSTANCE,
  );
  var pathContext = resourceProvider.pathContext;

  var folders = <Folder>[];
  for (var relPath in args) {
    var folder = resourceProvider.getFolder(relPath);
    if (!folder.exists) {
      stderr.writeln('Error: The directory "$relPath" does not exist.');
      exit(1);
    }
    folders.add(folder);
  }

  var dartPaths =
      AnalysisContextCollectionImpl(
            resourceProvider: resourceProvider,
            includedPaths: [...folders.map((e) => e.path)],
          ).contexts
          .expand((context) => context.contextRoot.analyzedFiles())
          .where((path) => file_paths.isDart(pathContext, path))
          .sorted();
  // print(dartFiles.join('\n'));

  var libPaths =
      dartPaths
          .where((path) => path.startsWith('$pkgPath/analyzer_cli/lib'))
          .toList();

  print(dartPaths.join('\n'));

  test('Initial libraries', () async {
    var timer = Stopwatch()..start();
    await _initialAnalysis(
      baseByteStore: baseInitByteStore,
      fineByteStore: fineInitByteStore,
      resourceProvider: resourceProvider,
      folders: folders,
    );
    print('[files: ${dartPaths.length}]');
    print('[time: ${timer.elapsedMilliseconds} ms]');
  });

  var random = Random(0);
  for (var i = 0; i < 1; i++) {
    var path = libPaths[random.nextInt(libPaths.length)];
    print('\n\n$path');
    var baseContext = AnalysisContextCollectionImpl(
      resourceProvider: resourceProvider,
      includedPaths: [...folders.map((e) => e.path)],
    ).contextFor(path);
    var parsedResult = baseContext.currentSession.getParsedUnit(path);
    parsedResult as ParsedUnitResultImpl;
    var mutationsBuilder = _MutationsBuilder();
    parsedResult.unit.accept(mutationsBuilder);
    print('[mutations: ${mutationsBuilder.mutations.length}]');

    var content = resourceProvider.getFile(path).readAsStringSync();
    for (var mutation in mutationsBuilder.mutations) {
      test('Mutation: $mutation', () async {
        var resourceProvider = OverlayResourceProvider(
          PhysicalResourceProvider.INSTANCE,
        );

        var baseCollection = AnalysisContextCollectionImpl(
          byteStore: _CascadingByteStore(baseInitByteStore, MemoryByteStore()),
          resourceProvider: resourceProvider,
          includedPaths: [...folders.map((e) => e.path)],
        );
        var baseContext = baseCollection.contextFor(path);

        var fineCollection = AnalysisContextCollectionImpl(
          byteStore: _CascadingByteStore(fineInitByteStore, MemoryByteStore()),
          resourceProvider: resourceProvider,
          includedPaths: [...folders.map((e) => e.path)],
          withFineDependencies: true,
        );
        var fineContext = fineCollection.contextFor(path);

        await _assertBaseFineLibraryTextAll(
          baseCollection: baseCollection,
          fineCollection: fineCollection,
          dartPaths: dartPaths,
        );

        var mutatedContent = mutation.getUpdatedContent(content);
        var shortDiff = generateFocusedDiff(content, mutatedContent);
        print('\n${'-' * 32} diff ${'-' * 32}');
        print(shortDiff.join('\n'));
        print('-' * 32);

        resourceProvider.setOverlay(
          path,
          content: mutatedContent,
          modificationStamp: 0,
        );
        baseContext.changeFile(path);
        await baseContext.applyPendingFileChanges();

        print('--- fineContext.changeFile');
        fineContext.changeFile(path);
        await fineContext.applyPendingFileChanges();

        await _assertBaseFineLibraryTextAll(
          baseCollection: baseCollection,
          fineCollection: fineCollection,
          dartPaths: dartPaths,
        );

        resourceProvider.setOverlay(
          path,
          content: content,
          modificationStamp: 0,
        );
        baseContext.changeFile(path);
        await baseContext.applyPendingFileChanges();

        fineContext.changeFile(path);
        await fineContext.applyPendingFileChanges();

        print('\n${'-' * 24} baseCollection');
        baseCollection.printPerformance();

        print('\n${'-' * 24} fineCollection');
        fineCollection.printPerformance();
      });
    }
  }
}

Future<void> _assertBaseFineLibraryTextAll({
  required AnalysisContextCollectionImpl baseCollection,
  required AnalysisContextCollectionImpl fineCollection,
  required List<String> dartPaths,
}) async {
  for (var path in dartPaths) {
    var baseElement = await baseCollection.getLibraryElement(path);
    var baseTextTimer = Stopwatch()..start();
    var baseText = getLibraryText(
      library: baseElement,
      configuration: ElementTextConfiguration(),
    );
    baseTextTimer.stop();
    await baseCollection.contextFor(path).currentSession.getErrors(path);

    var fineElement = await fineCollection.getLibraryElement(path);
    var fineTextTimer = Stopwatch()..start();
    var fineText = getLibraryText(
      library: fineElement,
      configuration: ElementTextConfiguration(),
    );
    fineTextTimer.stop();
    if (fineText != baseText) {
      print('[path: $path]');
      var timer = Stopwatch()..start();
      var short = generateFocusedDiff(baseText, fineText);
      print('-------- Short diff --------');
      print(short.join('\n'));
      print('----------------------------');
      print('[baseTextTimer: ${baseTextTimer.elapsedMilliseconds} ms]');
      print('[fineTextTimer: ${fineTextTimer.elapsedMilliseconds} ms]');
      print('[diffTime: ${timer.elapsedMilliseconds} ms]');
      print(
        '[baseText: ${baseText.length} bytes]'
        '[${baseText.split('\n').length} lines]',
      );
      print(
        '[fineText: ${fineText.length} bytes]'
        '[${fineText.split('\n').length} lines]',
      );
      throw StateError('See the diff above');
    }
    await fineCollection.contextFor(path).currentSession.getErrors(path);
  }
}

Future<void> _initialAnalysis({
  required ByteStore baseByteStore,
  required ByteStore fineByteStore,
  required ResourceProvider resourceProvider,
  required List<Folder> folders,
}) async {
  var pathContext = resourceProvider.pathContext;

  var baseCollection = AnalysisContextCollectionImpl(
    byteStore: baseByteStore,
    resourceProvider: resourceProvider,
    includedPaths: [...folders.map((e) => e.path)],
  );

  var fineCollection = AnalysisContextCollectionImpl(
    byteStore: fineByteStore,
    resourceProvider: resourceProvider,
    includedPaths: [...folders.map((e) => e.path)],
    withFineDependencies: true,
  );

  var dartPaths =
      baseCollection.contexts
          .expand((context) => context.contextRoot.analyzedFiles())
          .where((path) => file_paths.isDart(pathContext, path))
          .sorted();

  await _assertBaseFineLibraryTextAll(
    baseCollection: baseCollection,
    fineCollection: fineCollection,
    dartPaths: dartPaths,
  );
}

class Mutation {
  final _Replacement replacement;

  Mutation({required this.replacement});

  String getUpdatedContent(String content) {
    return content.replaceRange(
      replacement.offset,
      replacement.end,
      replacement.text,
    );
  }
}

class ReplaceReturnTypeWithVoid extends Mutation {
  ReplaceReturnTypeWithVoid({required super.replacement});
}

class _CascadingByteStore implements ByteStore {
  final ByteStore parent;
  final ByteStore local;

  _CascadingByteStore(this.parent, this.local);

  @override
  Uint8List? get(String key) {
    return local.get(key) ?? parent.get(key);
  }

  @override
  Uint8List putGet(String key, Uint8List bytes) {
    return local.putGet(key, bytes);
  }

  @override
  void release(Iterable<String> keys) {}
}

class _MutationsBuilder extends RecursiveAstVisitor<void> {
  final mutations = <Mutation>[];

  @override
  void visitMethodDeclaration(MethodDeclaration node) {
    var returnType = node.returnType;
    if (returnType != null) {
      if ('$returnType' != 'void') {
        mutations.add(
          ReplaceReturnTypeWithVoid(
            replacement: _Replacement(
              returnType.offset,
              returnType.end,
              'void',
            ),
          ),
        );
      }
    }

    super.visitMethodDeclaration(node);
  }
}

class _Replacement {
  final int offset;
  final int end;
  final String text;

  _Replacement(this.offset, this.end, this.text);
}

extension on AnalysisContextCollectionImpl {
  Future<LibraryElementImpl> getLibraryElement(String path) async {
    var analysisSession = contextFor(path).currentSession;
    var unitResult = await analysisSession.getUnitElement(path);
    unitResult as UnitElementResultImpl;
    return unitResult.fragment.element;
  }

  void printPerformance() {
    var buffer = StringBuffer();
    scheduler.accumulatedPerformance.write(buffer: buffer);
    scheduler.accumulatedPerformance = OperationPerformanceImpl('<scheduler>');
    print(buffer);
  }
}
