/*
 * Copyright (C) 2025 Yubico.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.yubico.authenticator.piv

object TestData {
    object RSA2048 {
        val PKCS12 = PKCS12_HEX.hexToByteArray()
        private const val PKCS12_HEX = "30820a5f02010330820a1506092a864886f70d010701a0820a060482" +
                "0a02308209fe3082046a06092a864886f70d010706a082045b308204570201003082045006092a8" +
                "64886f70d010701305f06092a864886f70d01050d3052303106092a864886f70d01050c30240410" +
                "94ecdd1cbea1477b713901994804bb4702020800300c06082a864886f70d02090500301d0609608" +
                "64801650304012a0410caec11f260bdc96ba0a59bad277418a4808203e043fc1d246460b3edb92b" +
                "9449f13f82c196a3837d15647b3d8172746063cbf8da5ff7d0baff37e835219003c2f9213ebe084" +
                "9f6fc8a495d5d5acc4a2f519f6b893ced799945662060d079b002967ff8d331f60672d73c08c43f" +
                "b5f57ff0e59c0e6a30bb3f53a159cc7f01ebd01a8aa3fd13a2639c85e3c3462e2c30411d2774092" +
                "548ab8fe0b43a436994d82d0218e8dd36e93767997f336f17bac8eba394adb31c004d1fd37f1b16" +
                "25dc9a48b48df844698da311f5c5740c6a1755277104e7fcbfd4d7318acaecab59028ec617e46e2" +
                "1199821c8a415662ac496746e40c42ff78a21d9f896f5b5c26b970e9e06919a28f6c2578a09b2cf" +
                "f2e9e2037bb883c61bdef99271067da22de5a15d5e03d40c4f7645f8913ed95c1b45711f75174e7" +
                "424739f20755adf66f43b41ffba609ed74896b5edc5695a5959f30db454ed358087ec84dabef5eb" +
                "80d9861e5286100d91c60cf833d767afe0356b5400523a1741607d3b4bd6e84608012c7763be94a" +
                "978450998ef31b3f78d83741911664b1922993af3a551d03de24249880c1f09e10017c30c020a5a" +
                "b55dd3290df29397446c087444fb7532106e5a9ee3762a45458a44f4295c9d8aee231a4332c396f" +
                "1c781f8c5b8b3a9cad39acec78863f5fb181513a77f55ad305d9a2865f1370543ed058997c3954c" +
                "f37255392d27c4c8d277b5446a017311e3a04aece860394e3272ccf0a25b39c7b558d948d1e87cd" +
                "c1174c9df15b10a02cb9644b58fc02eda753940f1467f47cc3e2ca72df69c60d675ac644fe739bb" +
                "8fb05b2b39de75302712d0cf0830d2290827172080c23a811c81ca11325759f491001d9efa15dd4" +
                "2e97f4082f0f9c4fd311c4d083fab58c7a6282c00d32dcdc7995274817dfcbcad345f7d2a464501" +
                "ba4473f288a6b14cde3c4851740166d62fac9b58f93d898a373f061731e917ec261595768a6ccbb" +
                "dc77af9a18b1e88a43259729070983ea1757d52c1f071feb0cbefe9158afdd4f42e6893aa9dec80" +
                "b7f32fa11e8e899cdd2697edcbc5494bf48f96153624f38b5868e3847db39ccf6fe98f1108f4dad" +
                "c9548a8304c5b5a55e2d6196801d7ae4cb5f4db88927af67ecf7cfdd2d75e21cda5b1cdba78fd14" +
                "9214dcaa06361f4472cc62ab69397251bee9841134c9c01c462fa8ddd5d443e0e055f5cb4a53e41" +
                "d77edc1111298ab757b5f88a2a3bc1590ebcd553b6ad5cb1c869081a5be73af53d5bd58bdc6c01e" +
                "7bd0d2a3dce6b29f1e231b918671c5f03ba49c3af26eb4f38fdc50a3e5b4d28fc8cea17a7ef45d7" +
                "ef631f9a8008d56b2b24bfd71949ea21a6dddf8de5d5153a991be8de9843c101f58c5d221b1fd6d" +
                "0440de3b5cbf925f0ae92b6dd9ee8b2d7c6cb1fed2439f6838277e6239dbf8c062fe3082058c060" +
                "92a864886f70d010701a082057d048205793082057530820571060b2a864886f70d010c0a0102a0" +
                "82053930820535305f06092a864886f70d01050d3052303106092a864886f70d01050c302404104" +
                "93a0c381e8350da4fab27a465b387d802020800300c06082a864886f70d02090500301d06096086" +
                "4801650304012a0410011db7eccf0d6574a9e0844d7e989530048204d098a23193e1876327bc8e3" +
                "b86261268e046699c86759b3b665e5215519e30ef366dc5c629b9c07716d626d720d09525f2f60c" +
                "86534fb9f8aab74959a4dd9392cb84e2c3985372e0fc2f39fd9271713a9a31db94a8c160ff429f1" +
                "901b5eff7d18bbc30e1fb66ea228d363c41c73e1563fb4a2627446d392de29c07e06f82e0a47316" +
                "ee93ffd58f807a344fe4d46f99ae6ba20dbce8f38f7db67de3b58898ec5aef5bef69f91dc2a127c" +
                "172494e1c03b0488ee8a4d825d83f91c9216addfb0be8eada6f04d34460882180ab772ce3c65354" +
                "41ec8734cfff1f38975b70249f2418965e8f7de9e56fab0084fc1aaeac530a9d110468fa1db9544" +
                "cc35297668460c1b42f129b014bff7aef5b888d86d72de4286883184bb35c945c92c074f22aafdc" +
                "9d7dd50879107fd848baf6838f882cf432403ed191b64114ba816267dec077de4be6e76f93a263b" +
                "135a85227b3ee7cdef078f21bbe9f80a2b709e8a17211cb5cd018160723165662ddc628138bb529" +
                "05c202f35085146942d68e094867a1dc847f21edcf341e8bc700947f3682b59fbdb57e4b9a3b939" +
                "3b1148ff46ef48e9a04e30e122a44c5fbbe6a7bffd622ba5623f049c1036e484b666b959eb58a43" +
                "27afc224ffb57a970d3d7e5b88cf2a65eaca1ab8e9f04d9030574df9c134b4e446177a6f4463a77" +
                "214e1a7ee762f93893bcc2a76e89a874d380d8efc7e3fc8466e42061b731dd455486f5d07bf61ef" +
                "29a0db635cb8ec9139024b0f768b95b57bd88a767fc77eb79b6fee557bb084a188eb8b0b2ddf9e2" +
                "b7c249768ac233b0f50da0424d0d41c85587785699ba32b0869da21aa156b2bcc5a16e86ef61339" +
                "a584f7da7db0b538077097b89b0059d1b198cd4a659a5082c61f8ba1ca3b8b030188feed100cfaa" +
                "81d5dce43a40543d9e76943ace557f3573c40b1e5e577bb371763639d530df9da7fe8e747b290f4" +
                "7f0b4b15439aee379bd2a481d779b451c3569a0ce48b648dac6e8b63bdbc2e1bbe22509d227a817" +
                "4448a3b60ea675f7d8f804f8a9aea1618922bae70021bca3fe83b957808468f8a9a2a0e1abb7d95" +
                "882b4c629373b3aa3a3d9fa1bd99abfa0094099051b82e1dac5a3fed03713a9647c00dc1a7e006f" +
                "aaca7832ee79e334d41babbb8f0ec6c2c1e27feb675d5bc5d7bba6ab1419d6a547880a61f1a3bf1" +
                "613cd2c46a77ba9ed6cf5da67ea4c039ef4a1912c1e1d50c388601deb4748656ca39d704988b60c" +
                "2458aee81651273c10ae62e157fdc563017816e462239c86bca04543343f493b4275727afd77288" +
                "5ab06b01ff92798f9134a5d7fd3dca5d2f08e1b1f5b29b55ac1e30f66b5846ba63aac1a52ce9b49" +
                "ce06f36875988be13a10b726fb2995e388dd721ae01e71ff076a7b252a1710a6b01c41ca416ba68" +
                "7ececf515ea5fc92aad3f95b4e6ee704706a84d3cf10645669622a4ec9c6cb90ec97baf41950692" +
                "cfbd6f1a8ed3769b43a1a97938f466786b0046b4c824d34fbd4b40ace1b1eff96dd57eecb7e63d4" +
                "c7a681f05c23e51223049d75d220db0b62a912869d71f6d708018e6a2f15de3e579a0230b15bd07" +
                "2a9db48cb61983e9359876ccbaaf918fbd2a675037735fd1e86c75652ea7fdc41519182432f1f37" +
                "638b3b27604b22611a712182d3e4fa4664c86c0c665326487fd8fa780422a952cc8aaae62116eda" +
                "985625ed2b4e9047e9d82c5b317e7580350d89a4e133150fa0b0061cb8138456f92cc3c0f312530" +
                "2306092a864886f70d01091531160414d19884b1536e383e7ff1181b0694183339cfc3ad3041303" +
                "1300d060960864801650304020105000420cba0cb53e5da0c7066745ad7344832e6c1de59aa43ab" +
                "4924f00f5f667bf8349e04089594cb0c096e9e5202020800"

        val PEM = PEM_HEX.hexToByteArray()
        private const val PEM_HEX = "2d2d2d2d2d424547494e2050524956415445204b45592d2d2d2d2d0a4d4" +
                "94945764149424144414e42676b71686b6947397730424151454641415343424b59776767536941" +
                "674541416f494241514332767a744d46633054525836410a38652f4539574c52705475565169346" +
                "37237424d7971396a586871324356397a583754312b586567677655506c6375396e795758566173" +
                "594c354f656e6c52790a4d6d457049672f46563162674a624b4a32706e4966622b726f712f6b343" +
                "1514e55435433626b7a4545466a576c6f3134687a3056726c4b346c2b76457232354e0a53713254" +
                "73444f552b3054487561705a2f6e4c643837695044354c4d4b4d787553617a346433516b6561587" +
                "3312b637a4f4a564169683371706e4d2f4f43356d0a672b59565044457978593835727245366a67" +
                "4a5945644a556f67566d32317561736e55663874496e45456f564f6b77384a576445395a4f424f4" +
                "95964374348630a454752364a7159576b4a494b7a6e79615774537a6e6c76627633453933523633" +
                "476f627933455679345452594b4f556c497971462f62436c43304f6c726350440a31613845626c4" +
                "65a41674d4241414543676639626c6855494a6b745545475a766e35326b5a50413749556475624e" +
                "5956765a4333556842684355694d387238770a59424f642f37563062324356496a44674a474a6c7" +
                "7714d57596c744d63454237783850703877364e3865726c424d786346786a494658437446436835" +
                "644547530a42426943746173707977654874314b56794753482b534a372b6e52494f416e3247506" +
                "3594a756d7a445733724554746f65476f5a42597866762b35534c30616b0a7642782f7639746d56" +
                "566634477372626e576c7974682f70572f336e545a6c48387939657a564b6e684d4847323669667" +
                "36b394c416a6a4b2b55785a6a4b31570a73414b42396b566c497635567455676f6f55316c6b6270" +
                "382f544a4e72672f6932384c4172375731414c39737855744c314b4a594b376d6c62736a6839502" +
                "b380a3377534646534d38394b6677716e3930644466367773356a59386b417a4f58656566397778" +
                "566b436759454137414c784f642b4650636f772f4938527039724d0a42336f337756506f3676563" +
                "4746c6f414b6c643461746f6a5376776776614e30566d73696a436a456557724271636c39362b42" +
                "785168545a353277586e5479540a51736d616473756a6473696a3967384865423879464e3667374" +
                "a3952594856386a4c6c3974622b4849726434634667412f6d456e436d344a747941576b494a750a" +
                "5a77576f4e61416759576e43506f7142737562684766304367594541786a6c773466684e466c525" +
                "24a36704d614f784454356775356a686b7539647444726f310a794b756945554942425a69312f4a" +
                "72374534447a62305764485a696a55586438654838326c4c3739384d6a626e6565615a71784d532" +
                "b492b4743436e42492b4a0a756b416676626f306b7a543678774a2f313332634255445266474c2b" +
                "4b4837696a53306b7034587a315a374a624f2f53574f4a59545731362f38647342474d550a7a4a6" +
                "5365659304367594541704656554a436a34504e4d4a6b7749486d436f6f514e6966382f52365451" +
                "674a4f6d656d6a3955434774797a45475a4959654e720a6754547542457671716d526b587458573" +
                "45351526d4b4b754b34362f54704f384d686677667546436539597648496a51655a713349344c30" +
                "364733796d7369540a7065554f6e447531573734476c61736858663746305752585458534259625" +
                "1414c616e7075785535467273754c714141755649373050304367594275416954650a6b36476365" +
                "64444d4e5a76515177616348725977654332586664663961325137364e793576537136413256454" +
                "f6e4b4c41573659573062317a4f6365614453410a337744596a694a3652633530464c7641725842" +
                "75554c4872532f76453854416257456b46543169734e61505458465338722b34746b437079594c4" +
                "8796f58427a0a7278442f6b377964326c437044334e71647274517853695532515465626d70316e" +
                "4f4d425a514b42675144593467514c2b756f384c575a5367664156532b6f460a416c74634468666" +
                "f327663485758302b615558416c743551573535734871335a6e63333958634e43382b6b504b426a" +
                "6b7a7872324b43304d473856672f706b4c0a526e2b2b71472b67664b584e4d7438487a676e62485" +
                "06465336c4552763662486a5a39394a784f4f494d365150464a747a78625773416e5936676c6a65" +
                "5758750a794d6567584367397145416d704348346b78794c71413d3d0a2d2d2d2d2d454e4420505" +
                "24956415445204b45592d2d2d2d2d0a2d2d2d2d2d424547494e2043455254494649434154452d2d" +
                "2d2d2d0a4d49494465444343416d4367417749424167495241505a4951445741395865657643474" +
                "74450586d673873774451594a4b6f5a496876634e4151454c425141770a5454454c4d416b474131" +
                "554542684d4356564d784454414c42674e564241674d4246526c633351784454414c42674e56424" +
                "1634d4246526c633351784454414c0a42674e5642416f4d4246526c633351784554415042674e56" +
                "42414d4d4346526c633352445a584a304d423458445449314d4467774e7a41314d5441304e6c6f5" +
                "80a445449324d4467774e7a41314d5441304e6c6f775454454c4d416b474131554542684d435656" +
                "4d784454414c42674e564241674d4246526c633351784454414c0a42674e564241634d4246526c6" +
                "33351784454414c42674e5642416f4d4246526c633351784554415042674e5642414d4d4346526c" +
                "633352445a584a304d4949420a496a414e42676b71686b6947397730424151454641414f4341513" +
                "8414d49494243674b4341514541747238375442584e4530562b6750487678505669306155370a6c" +
                "554975484b2b77544d71765931346174676c6663312b3039666c336f494c314435584c765a386c6" +
                "c31577247432b546e703555636a4a684b534950785664570a344357796964715a7948322f71364b" +
                "76354f4e554456416b3932354d784242593170614e6549633946613553754a6672784b397554557" +
                "1746b37417a6c5074450a78376d715766357933664f346a772b537a436a4d626b6d732b4864304a" +
                "486d6c374e666e4d7a695651496f6436715a7a507a67755a6f506d465477784d7357500a4f61367" +
                "84f6f344357424853564b49465a7474626d724a31482f4c534a78424b4654704d5043566e525057" +
                "5467546947486577683342426b6569616d467043530a437335386d6c72557335356232373978506" +
                "4306574787147387478466375453057436a6c4a534d716866327770517444706133447739577642" +
                "473552575149440a415141426f314d775554416442674e564851344546675155363458334b30487" +
                "535344e574a455731586d3678616d4652304a3077487759445652306a424267770a466f41553634" +
                "58334b30487535344e574a455731586d3678616d4652304a307744775944565230544151482f424" +
                "15577417745422f7a414e42676b71686b69470a397730424151734641414f43415145416559786d" +
                "7a4530716e2b7a5578736f3056675173322b656b4b437a4868346937312b53346c4679326956454" +
                "e6459792f0a36743866317967644c55364a777755505a38386d75457a6c4a7a42584b4342506636" +
                "45666b7241732b3168542b3669556631704c57547337486779506b6a38310a446c714f624571454" +
                "e43556a4453776152475578416474425137745272355678777453626d5334474e65444e31766c68" +
                "6d38596830364f6c4233727047424b6f0a484a572f556d3233504a644e7a414f5a4259786d796f4" +
                "f77777042683035694b736f30712b76536d365844497a7052723853443770704f46496c6e6b6371" +
                "726f0a6755385a394b6e734a44464f716f6c682b694b3353706c654c3132696965744d6d32316c3" +
                "84b6c6d4830687651326d48454d786358664d757377335349597a470a666c775078303975672f63" +
                "4a35596b70432b4568436d666472675357324272594f5a414336673d3d0a2d2d2d2d2d454e44204" +
                "3455254494649434154452d2d2d2d2d0a"

        val PEM_ENC = PEM_ENC_HEX.hexToByteArray()
        private const val PEM_ENC_HEX = "2d2d2d2d2d424547494e20454e43525950544544205052495641544" +
                "5204b45592d2d2d2d2d0a4d4949464e54426642676b71686b69473977304242513077556a417842" +
                "676b71686b694739773042425177774a4151514b51577933794e7855373679612f494d0a2f567a7" +
                "77277494343414177444159494b6f5a496876634e41676b464144416442676c67686b67425a514d" +
                "4541536f4545424a36784d37666a4d76744544584d0a6e657a4c6e5049456767545177486f7a2b7" +
                "7794644356f444142702f2f393164644f30396f755532362b32444543487050677a4e49636e5764" +
                "6467324e425a6c0a52776c4364394a5251475338503149784d546562492f6877486c37554c44674" +
                "9356e6f61777a444d554b474149704277777330504737596e56716b6f6973456b0a71306f733038" +
                "623355595a764a55766a6f44374f386542736153762b784b5856364e72494b30644d454a4f61523" +
                "149642f646a6270433374554e5643555355340a39487a714f6b4e4f773531564f7a7a7235445a71" +
                "797046356e426d4f73714f34547979394944774a576835316e527a6b4d5575753279767458566c4" +
                "86f6b50370a4334727764706861716768766664527a4b79384f6b566c55514c4830376851775775" +
                "4d51354c7071574a742f77377a452b614132415776335638724c397045650a79704a6456542b6f7" +
                "9414b746e2b3136342b71393349772f695479454c396f7269587a592f326c735633667a706f6547" +
                "2f745557624a6843662f555851716b6e0a566257684a734c31504a69706f4f324239557249306d5" +
                "867664e697979727679436d575a73345637524f4b7a2f394e2f69634e38756e714d695831583458" +
                "4d4b0a75767166757a486568756c78793748594f4751774743334e7871394b304f6768537946514" +
                "e355567505a374475526449305948314a37775575466251312b61460a32766b5343737851756e78" +
                "616342524b79585943734f734c524239612f476c764b7841654c5435766e52444e424f575265666" +
                "433685a47675a6a6e58634b706d0a71325572635252624f2f395676472b3047616873324b6a4873" +
                "643146346a626d4d7634364b4e525a704f6456456a326e5878547871347a584734672f72584e570" +
                "a41423141454f786d756a72346f784e7266534d386c5a414834646d54684b71626e656b4e585655" +
                "76507361637a3879716239783738624f6b7950486938796e2b0a4a3564314d586a6f4f34474d747" +
                "15068424759704f764e684c52695866584d6a324978544d497564652f715167485634692f556872" +
                "55627a3846362b325a72440a4f2f6768456d503932553765706542456d6b4e74564a2f563458424" +
                "c4a5337766647342f45614d4936392f485a30796a5a3146517761726b484c675832504b540a744b" +
                "4159746c61464e4e717175487a6368424b6535364e7a66366f2f37726347764c326737556277764" +
                "f67496841552f4c56764f4355424e2f7153464466566f0a55784a694f717271544f2b5338774a6c" +
                "31702f312f5157644c50697a536830444341653243644a70774c7344335578376d77617173704f4" +
                "331376d774b6b31320a5166724239324935525a447761566f4b32466a7048626872627857613936" +
                "787a616a7363686e6c7058594d64494c617557687636794c4162422b4669716b72370a412f316f5" +
                "475662b776f6751756957395a4d7339374645596b3375376756774932515831504f72766259344e" +
                "6c496757364e62687355537774794438723830740a5a7262795265724730704d2b43437441586e4" +
                "c396e3557666a78706c2b756f3370374779624f6d737175393637675a5551333233437456316c32" +
                "51376e424a430a614c3452317444713466495832642b666a3572646b73456161645042754854627" +
                "26b304b525056427736617a6648653542326d6a4258455562446745623148580a415a6851354671" +
                "55574b6e6a6c48434377472f43593872646f77345476524276322b4661397669337a645a3342554" +
                "931646954524c74423369664561395a6d550a526e624a7a67696c6534644755514678663577707a" +
                "48367042517576322b4f507332716e685671313053504e7334446a516d2b747a5a3474784142783" +
                "24a74680a394f5571314b6b33683257345251584237446132383252454e466e30697a4f7471362b" +
                "4871664a76307257504631456f4d4a49684a677650754c5977447a42740a2f3642344e59782b417" +
                "42f31736f563178493132545049654745323932442b6741357536617a7566433747744233733277" +
                "4e2b4f78735679797169596b547a630a4b7767363967374159343339646b596c6857744d5839467" +
                "6753031625a5644304c6a6259753564566374342f70536946636758524b6d4571756b5139416d48" +
                "450a51726757624443656e55637a7331317a66486244534f596330507679536e5862547a594e624" +
                "362745753647a31456e5770654a662f5a31326567636f54622b6c0a357639635643534a48344449" +
                "543362694c754f676237427346684254767266375849734151706b3733376444484e44346c567a5" +
                "12b556b3d0a2d2d2d2d2d454e4420454e435259505445442050524956415445204b45592d2d2d2d" +
                "2d0a2d2d2d2d2d424547494e2043455254494649434154452d2d2d2d2d0a4d49494465444343416" +
                "d4367417749424167495241505a495144574139586565764347474450586d673873774451594a4b" +
                "6f5a496876634e4151454c425141770a5454454c4d416b474131554542684d4356564d784454414" +
                "c42674e564241674d4246526c633351784454414c42674e564241634d4246526c63335178445441" +
                "4c0a42674e5642416f4d4246526c633351784554415042674e5642414d4d4346526c633352445a5" +
                "84a304d423458445449314d4467774e7a41314d5441304e6c6f580a445449324d4467774e7a4131" +
                "4d5441304e6c6f775454454c4d416b474131554542684d4356564d784454414c42674e564241674" +
                "d4246526c633351784454414c0a42674e564241634d4246526c633351784454414c42674e564241" +
                "6f4d4246526c633351784554415042674e5642414d4d4346526c633352445a584a304d4949420a4" +
                "96a414e42676b71686b6947397730424151454641414f43415138414d49494243674b4341514541" +
                "747238375442584e4530562b6750487678505669306155370a6c554975484b2b77544d717659313" +
                "46174676c6663312b3039666c336f494c314435584c765a386c6c31577247432b546e703555636a" +
                "4a684b534950785664570a344357796964715a7948322f71364b76354f4e554456416b3932354d7" +
                "84242593170614e6549633946613553754a6672784b3975545571746b37417a6c5074450a78376d" +
                "715766357933664f346a772b537a436a4d626b6d732b4864304a486d6c374e666e4d7a695651496" +
                "f6436715a7a507a67755a6f506d465477784d7357500a4f6136784f6f344357424853564b49465a" +
                "7474626d724a31482f4c534a78424b4654704d5043566e5250575467546947486577683342426b6" +
                "569616d467043530a437335386d6c72557335356232373978506430657478714738747846637545" +
                "3057436a6c4a534d716866327770517444706133447739577642473552575149440a415141426f3" +
                "14d775554416442674e564851344546675155363458334b30487535344e574a455731586d367861" +
                "6d4652304a3077487759445652306a424267770a466f4155363458334b30487535344e574a45573" +
                "1586d3678616d4652304a307744775944565230544151482f42415577417745422f7a414e42676b" +
                "71686b69470a397730424151734641414f43415145416559786d7a4530716e2b7a5578736f30566" +
                "75173322b656b4b437a4868346937312b53346c4679326956454e6459792f0a3674386631796764" +
                "4c55364a777755505a38386d75457a6c4a7a42584b434250663645666b7241732b3168542b36695" +
                "56631704c57547337486779506b6a38310a446c714f624571454e43556a44537761524755784164" +
                "74425137745272355678777453626d5334474e65444e31766c686d38596830364f6c42337270474" +
                "24b6f0a484a572f556d3233504a644e7a414f5a4259786d796f4f77777042683035694b736f3071" +
                "2b76536d365844497a7052723853443770704f46496c6e6b6371726f0a6755385a394b6e734a444" +
                "64f716f6c682b694b3353706c654c3132696965744d6d32316c384b6c6d4830687651326d48454d" +
                "786358664d757377335349597a470a666c775078303975672f634a35596b70432b4568436d66647" +
                "2675357324272594f5a414336673d3d0a2d2d2d2d2d454e442043455254494649434154452d2d2d" +
                "2d2d0a"

        val DER_CERT = DER_CERT_HEX.hexToByteArray()
        private const val DER_CERT_HEX = "3082037830820260a003020102021100f648403580f5779ebc2186" +
                "0cf5e683cb300d06092a864886f70d01010b0500304d310b3009060355040613025553310d300b0" +
                "6035504080c0454657374310d300b06035504070c0454657374310d300b060355040a0c04546573" +
                "743111300f06035504030c085465737443657274301e170d3235303830373035313034365a170d3" +
                "236303830373035313034365a304d310b3009060355040613025553310d300b06035504080c0454" +
                "657374310d300b06035504070c0454657374310d300b060355040a0c04546573743111300f06035" +
                "504030c08546573744365727430820122300d06092a864886f70d01010105000382010f00308201" +
                "0a0282010100b6bf3b4c15cd13457e80f1efc4f562d1a53b95422e1cafb04ccaaf635e1ab6095f7" +
                "35fb4f5f977a082f50f95cbbd9f259755ab182f939e9e5472326129220fc55756e025b289da99c8" +
                "7dbfaba2afe4e3540d5024f76e4cc41058d6968d78873d15ae52b897ebc4af6e4d4aad93b03394f" +
                "b44c7b9aa59fe72ddf3b88f0f92cc28cc6e49acf877742479a5ecd7e7333895408a1deaa6733f38" +
                "2e6683e6153c3132c58f39aeb13a8e025811d254a20566db5b9ab2751ff2d227104a153a4c3c256" +
                "744f5938138861dec21dc10647a26a61690920ace7c9a5ad4b39e5bdbbf713ddd1eb71a86f2dc45" +
                "72e1345828e525232a85fdb0a50b43a5adc3c3d5af046e51590203010001a3533051301d0603551" +
                "d0e04160414eb85f72b41eee783562445b55e6eb16a6151d09d301f0603551d23041830168014eb" +
                "85f72b41eee783562445b55e6eb16a6151d09d300f0603551d130101ff040530030101ff300d060" +
                "92a864886f70d01010b05000382010100798c66cc4d2a9fecd4c6ca3456042cdbe7a4282cc78788" +
                "bbd7e4b8945cb689510d758cbfeadf1fd7281d2d4e89c3050f67cf26b84ce527305728204f7fa11" +
                "f92b02cfb5853fba8947f5a4b593b3b1e0c8f923f350e5a8e6c4a843425230d2c1a44653101db41" +
                "43bb51af9571c2d49b992e0635e0cdd6f9619bc621d3a3a5077ae91812a81c95bf526db73c974dc" +
                "c0399058c66ca83b0c29061d3988ab28d2afaf4a6e970c8ce946bf120fba693852259e472aae881" +
                "4f19f4a9ec24314eaa8961fa22b74a995e2f5da289eb4c9b6d65f0a9661f486f43698710cc5c5df" +
                "32eb30dd2218cc67e5c0fc74f6e83f709e589290be1210a67ddae0496d81ad8399002ea"

        val DER_KEY = DER_KEY_HEX.hexToByteArray()
        private const val DER_KEY_HEX = "308204a20201000282010100b6bf3b4c15cd13457e80f1efc4f562d" +
                "1a53b95422e1cafb04ccaaf635e1ab6095f735fb4f5f977a082f50f95cbbd9f259755ab182f939e" +
                "9e5472326129220fc55756e025b289da99c87dbfaba2afe4e3540d5024f76e4cc41058d6968d788" +
                "73d15ae52b897ebc4af6e4d4aad93b03394fb44c7b9aa59fe72ddf3b88f0f92cc28cc6e49acf877" +
                "742479a5ecd7e7333895408a1deaa6733f382e6683e6153c3132c58f39aeb13a8e025811d254a20" +
                "566db5b9ab2751ff2d227104a153a4c3c256744f5938138861dec21dc10647a26a61690920ace7c" +
                "9a5ad4b39e5bdbbf713ddd1eb71a86f2dc4572e1345828e525232a85fdb0a50b43a5adc3c3d5af0" +
                "46e515902030100010281ff5b961508264b5410666f9f9da464f03b21476e6cd615bd90b7521061" +
                "09488cf2bf3060139dffb5746f60952230e0246265c2a316625b4c70407bc7c3e9f30e8df1eae50" +
                "4cc5c1718c81570ad142879744192041882b5ab29cb0787b75295c86487f9227bfa74483809f618" +
                "f71826e9b30d6deb113b68786a19058c5fbfee522f46a4bc1c7fbfdb665557f81acadb9d6972b61" +
                "fe95bfde74d9947f32f5ecd52a784c1c6dba89fb24f4b0238caf94c598cad56b00281f6456522fe" +
                "55b54828a14d6591ba7cfd324dae0fe2dbc2c0afb5b500bf6cc54b4bd4a2582bb9a56ec8e1f4ffb" +
                "cdf048515233cf4a7f0aa7f747437fac2ce6363c900cce5de79ff70c55902818100ec02f139df85" +
                "3dca30fc8f11a7dacc077a37c153e8eaf578b65a002a57786ada234afc20bda374566b228c28c47" +
                "96ac1a9c97debe0714214d9e76c179d3c9342c99a76cba376c8a3f60f07781f3214dea0ec9f5160" +
                "757c8cb97db5bf8722b778705800fe61270a6e09b7201690826e6705a835a0206169c23e8a81b2e" +
                "6e119fd02818100c63970e1f84d16545127aa4c68ec434f982ee63864bbd76d0eba35c8aba21142" +
                "010598b5fc9afb1380f36f459d1d98a351777c787f3694befdf0c8db9de79a66ac4c4be23e1820a" +
                "7048f89ba401fbdba349334fac7027fd77d9c0540d17c62fe287ee28d2d24a785f3d59ec96cefd2" +
                "58e2584d6d7affc76c046314cc97ba558d02818100a455542428f83cd309930207982a2840d89ff" +
                "3f47a4d08093a67a68fd5021adcb310664861e36b8134ee044beaaa64645ed5d6e1241198a2ae2b" +
                "8ebf4e93bc3217f07ee1427bd62f1c88d0799ab72382f4e86df29ac893a5e50e9c3bb55bbe0695a" +
                "b215dfec5d164574d748161b4002da9e9bb153916bb2e2ea000b9523bd0fd0281806e0224de93a1" +
                "9c79d0cc359bd043069c1eb630782d977dd7fd6b643be8dcb9bd2aba0365443a728b016e985b46f" +
                "5cce71e683480df00d88e227a45ce7414bbc0ad706e50b1eb4bfbc4f1301b5849054f58ac35a3d3" +
                "5c54bcafee2d902a7260b1f2a17073af10ff93bc9dda50a90f736a76bb50c52894d904de6e6a759" +
                "ce3016502818100d8e2040bfaea3c2d665281f0154bea05025b5c0e17e8daf707597d3e6945c096" +
                "de505b9e6c1eadd99dcdfd5dc342f3e90f2818e4cf1af6282d0c1bc560fe990b467fbea86fa07ca" +
                "5cd32df07ce09db1cf75ede5111bfa6c78d9f7d27138e20ce903c526dcf16d6b009d8ea09637965" +
                "eec8c7a05c283da84026a421f8931c8ba8"
    }

    object X25519 {
        val PEM = PEM_HEX.hexToByteArray()
        private const val PEM_HEX = "2d2d2d2d2d424547494e2050524956415445204b45592d2d2d2d2d0a4d4" +
                "3344341514177425159444b32567542434945494a6952677a734949666d4a4b6b57656b34586e35" +
                "414a715867567368636b4a65356c6c4a59704d5a694a340a2d2d2d2d2d454e44205052495641544" +
                "5204b45592d2d2d2d2d0a"

        val PEM_ENC = PEM_ENC_HEX.hexToByteArray()
        private const val PEM_ENC_HEX = "2d2d2d2d2d424547494e20454e43525950544544205052495641544" +
                "5204b45592d2d2d2d2d0a4d49476a4d463847435371475349623344514546445442534d44454743" +
                "53714753496233445145464444416b4242416c6b35526b673748496a762f79573179360a47576a7" +
                "7416749494144414d42676771686b694739773043435155414d423047435743475341466c417751" +
                "424b675151305a2f534235545a377157487374386c0a484544712f77524177506a7273786f6c747" +
                "14b346955584a6a7343545354397959477573397350454b586e55512b6d6957386d6f6e6a73714b" +
                "66764a724a784a0a34526f56334f52756972425858546234727772465072734e364a324c6d773d3" +
                "d0a2d2d2d2d2d454e4420454e435259505445442050524956415445204b45592d2d2d2d2d0a"

        val DER = DER_HEX.hexToByteArray()
        private const val DER_HEX = "302e020100300506032b656e042204209891833b0821f9892a459e9385e" +
                "7e4026a5e056c85c9097b9965258a4c662278"
    }

    object ED25519 {
        val PKCS12 = PKCS12_HEX.hexToByteArray()
        private const val PKCS12_HEX = "308203f6020103308203ac06092a864886f70d010701a082039d0482" +
                "0399308203953082029a06092a864886f70d010706a082028b308202870201003082028006092a8" +
                "64886f70d010701305f06092a864886f70d01050d3052303106092a864886f70d01050c30240410" +
                "861e6cc05174c447b5f3ac29ce75669e02020800300c06082a864886f70d02090500301d0609608" +
                "64801650304012a0410b7a67bc377cfd82e127310f2f3db12e680820210cd641daead0ff576ac3d" +
                "e6bce8d8c66a85a79a5d032f29e143bf75ed431f1dd95257f250c9815c40b291c932bf1099c2d25" +
                "8c201bc74031e17c348b148361083aa8802d3f2ca894b95421d9c9c427a09dab8836e3619b3b32b" +
                "833965de4cce329da85520d46b0fd006dfadab9a0fc30583e14f32871ce39d814d641f0bab8af3d" +
                "e9b4c906408b1304058414e4b3e094c82bf089b23e6cda6177d01914c408d5d358e68fd78bf6187" +
                "e84c218ae16d5ccffeadf8dbfc9b568ad1c604a7c57706b493055b9c4b3f81d310a2138f5323903" +
                "0816ef36fc1d24c4a28cb348ddf4d1eff757fca0e09bf74e01a56321a6c744516589a02eeba7819" +
                "6493d3a9df04e1ef380a605ffa924ba7beadec29bcfe6f051dfe734d227c6cbaa9a2c4c241de09b" +
                "90c6f4e99592d739538fb27f3d507c407cefb94712381a0abe1259c6c27de47cd954fdd4d129321" +
                "50fde3970e917f62d8cf12fe668e7b3b423a4507ea02b20cc80d10d6b4a0a97deafe9f20296858e" +
                "e39601fbc5dcaf149d1f60ef2a1cadc30d63aeb51a8f72a7fd82eaad3b2e6beae00b7c5f65660d4" +
                "9a9d77ad0d20ba9fb3bba6a53ce1d0ac1756f740370bc76745bf70e150862f84ba2cccbaddd413b" +
                "175373843a770d9a8fc01ec0bb6df4187b52e78cc1198e50c5f6fde5b67449cd251d571f79b30f1" +
                "616acead46d7073cb4b37f22d87f45d75b89e0b9edb600e09d19794d8d9ff3183500ba6f47c01b5" +
                "2c542dd583081f406092a864886f70d010701a081e60481e33081e03081dd060b2a864886f70d01" +
                "0c0a0102a081a63081a3305f06092a864886f70d01050d3052303106092a864886f70d01050c302" +
                "404108a916e880e3475b0cfad0c747ee3023502020800300c06082a864886f70d02090500301d06" +
                "0960864801650304012a0410dc78b40243750cb9e25cec90d59fb48f0440bc47404f9f0f3169947" +
                "03dfd9980d6f79a70bea698e1c08c4b03229d9b98fdf9dd0859369154a4537d6cec2c1f36c5077b" +
                "e8bc3262d7dfa4f1a238505b792b743125302306092a864886f70d010915311604141c458990fa7" +
                "fee1e592d05731049891d2507aa2130413031300d06096086480165030402010500042051cc48e0" +
                "5aca3856d0f096629ac02a6cfe397ec7ba8fde7ad6bd56220a6e1ac004088550aae32410466e020" +
                "20800"

        val PEM = PEM_HEX.hexToByteArray()
        private const val PEM_HEX = "2d2d2d2d2d424547494e2050524956415445204b45592d2d2d2d2d0a4d4" +
                "3344341514177425159444b325677424349454948664d58546f6c455152756d3639522b35747138" +
                "4e5279444a36434f4332442f344b6558476151756e6a540a2d2d2d2d2d454e44205052495641544" +
                "5204b45592d2d2d2d2d0a2d2d2d2d2d424547494e2043455254494649434154452d2d2d2d2d0a4d" +
                "494942717a4343415632674177494241674951496243646570516d4f70364163576e316a4f2b4b6" +
                "444414642674d725a5841775454454c4d416b47413155450a42684d4356564d784454414c42674e" +
                "564241674d4246526c633351784454414c42674e564241634d4246526c633351784454414c42674" +
                "e5642416f4d4246526c0a633351784554415042674e5642414d4d4346526c633352445a584a304d" +
                "423458445449314d4467774e7a41314d5441304e316f58445449324d4467774e7a41310a4d54413" +
                "04e316f775454454c4d416b474131554542684d4356564d784454414c42674e564241674d424652" +
                "6c633351784454414c42674e564241634d4246526c0a633351784454414c42674e5642416f4d424" +
                "6526c633351784554415042674e5642414d4d4346526c633352445a584a304d436f77425159444b" +
                "325677417945410a4f4c6473624d557062694a614a49485455434373506132492f4531794f30662" +
                "f6e2f7476773071394a5a326a557a42524d4230474131556444675157424251710a2b56314d4435" +
                "492f7356376a537666434a554a656b6a7250376a416642674e5648534d4547444157674251712b5" +
                "6314d4435492f7356376a537666434a554a650a6b6a7250376a415042674e5648524d4241663845" +
                "425441444151482f4d4155474179746c63414e424145334e766464386e4636724f3977504a30646" +
                "f4764662b0a754c504f647733684a42397852436a366c6b356473624663476279314764424c555a" +
                "3376324871754532314f7a43654b323562534d376451625251752f41673d0a2d2d2d2d2d454e442" +
                "043455254494649434154452d2d2d2d2d0a"

        val PEM_ENC = PEM_ENC_HEX.hexToByteArray()
        private const val PEM_ENC_HEX = "2d2d2d2d2d424547494e20454e43525950544544205052495641544" +
                "5204b45592d2d2d2d2d0a4d49476a4d463847435371475349623344514546445442534d44454743" +
                "53714753496233445145464444416b4242425232576d675269586c516d4c4e714f474c0a6154384" +
                "b416749494144414d42676771686b694739773043435155414d423047435743475341466c417751" +
                "424b6751517941666a42446450675136414d7070370a323659674e415241376379474b743955386" +
                "d536164386a7556666157625a3339786a54704c563151353737785a7434445067702f6164746668" +
                "4e75456a426a610a664f6d4b324675436c785459654934416f4c57596c6c3766366b457475673d3" +
                "d0a2d2d2d2d2d454e4420454e435259505445442050524956415445204b45592d2d2d2d2d0a2d2d" +
                "2d2d2d424547494e2043455254494649434154452d2d2d2d2d0a4d494942717a434341563267417" +
                "7494241674951496243646570516d4f70364163576e316a4f2b4b6444414642674d725a58417754" +
                "54454c4d416b47413155450a42684d4356564d784454414c42674e564241674d4246526c6333517" +
                "84454414c42674e564241634d4246526c633351784454414c42674e5642416f4d4246526c0a6333" +
                "51784554415042674e5642414d4d4346526c633352445a584a304d423458445449314d4467774e7" +
                "a41314d5441304e316f58445449324d4467774e7a41310a4d5441304e316f775454454c4d416b47" +
                "4131554542684d4356564d784454414c42674e564241674d4246526c633351784454414c42674e5" +
                "64241634d4246526c0a633351784454414c42674e5642416f4d4246526c63335178455441504267" +
                "4e5642414d4d4346526c633352445a584a304d436f77425159444b325677417945410a4f4c64736" +
                "24d557062694a614a49485455434373506132492f4531794f30662f6e2f7476773071394a5a326a" +
                "557a42524d4230474131556444675157424251710a2b56314d4435492f7356376a537666434a554" +
                "a656b6a7250376a416642674e5648534d4547444157674251712b56314d4435492f7356376a5376" +
                "66434a554a650a6b6a7250376a415042674e5648524d4241663845425441444151482f4d4155474" +
                "179746c63414e424145334e766464386e4636724f3977504a30646f4764662b0a754c504f647733" +
                "684a42397852436a366c6b356473624663476279314764424c555a3376324871754532314f7a436" +
                "54b323562534d376451625251752f41673d0a2d2d2d2d2d454e442043455254494649434154452d" +
                "2d2d2d2d0a"

        val DER_CERT = DER_CERT_HEX.hexToByteArray()
        private const val DER_CERT_HEX = "308201ab3082015da003020102021021b09d7a94263a9e807169f5" +
                "8cef8a74300506032b6570304d310b3009060355040613025553310d300b06035504080c0454657" +
                "374310d300b06035504070c0454657374310d300b060355040a0c04546573743111300f06035504" +
                "030c085465737443657274301e170d3235303830373035313034375a170d3236303830373035313" +
                "034375a304d310b3009060355040613025553310d300b06035504080c0454657374310d300b0603" +
                "5504070c0454657374310d300b060355040a0c04546573743111300f06035504030c08546573744" +
                "3657274302a300506032b657003210038b76c6cc5296e225a2481d35020ac3dad88fc4d723b47ff" +
                "9ffb6fc34abd259da3533051301d0603551d0e041604142af95d4c0f923fb15ee34af7c225425e9" +
                "23acfee301f0603551d230418301680142af95d4c0f923fb15ee34af7c225425e923acfee300f06" +
                "03551d130101ff040530030101ff300506032b65700341004dcdbdd77c9c5eab3bdc0f27476819d" +
                "7feb8b3ce770de1241f714428fa964e5db1b15c19bcb519d04b519defd87aae136d4ecc278adb96" +
                "d233b7506d142efc08"

        val DER_KEY = DER_KEY_HEX.hexToByteArray()
        private const val DER_KEY_HEX = "302e020100300506032b65700422042077cc5d3a2511046e9baf51f" +
                "b9b6af0d4720c9e82382d83ff829e5c6690ba78d3"

    }

    object ECCP384 {
        val PKCS12 = PKCS12_HEX.hexToByteArray()
        private const val PKCS12_HEX = "308204fe020103308204b406092a864886f70d010701a08204a50482" +
                "04a13082049d3082031a06092a864886f70d010706a082030b308203070201003082030006092a8" +
                "64886f70d010701305f06092a864886f70d01050d3052303106092a864886f70d01050c30240410" +
                "70f6cc65af8629eb8e685594bd62984202020800300c06082a864886f70d02090500301d0609608" +
                "64801650304012a041033590b727eb0e2e47da43f13c18c1ff880820290f076742de4eb6b811e0a" +
                "80a1defeb7f12db4de3f5bea5ab52b59489b12cd226244af13af2b2adcaa32dcfa2d88ccc4b6a0d" +
                "8982975242613df5789b8144700b01bff8cdb98bc29d53d6bc7f001551f932c88fb59dee3b057e5" +
                "e9bc0ae21a250514faa34dd9f1b4c6413066ddd43f481a5e8f0b7b9b514573c261978fddb622a90" +
                "ed4d560ebcfdd73fab54ef678e8faf4892d341ea58e27b881843e84bc9e058818aae0e441862ada" +
                "b276fc7a1a42f42951785635f156171acd022555a281015f45d2c0a271745128401f12e4e124c49" +
                "5a621622a8ee6a13396456e9a61195d96e4e8829e1855fa409639a5c0ec813571539b4a1970fc57" +
                "f00bb5b5ceb1336f1aab73db34056390440d66a97671bc3ac8a4b46295f6450114d17a8f6b59d1f" +
                "ac6c3216704ab9760b9015fbdcb5d63a65a55cf07bfaa50ba71b14881bbdc2b607ecc50bf7c95e9" +
                "76e68a0bee8613b3c10ddf1abf653448b734dca650641fb5a9a6e9feecaa6d9061bc1e6aa9faa87" +
                "36b4f8ec52ec146f1060a7ccbfcf4cca23d5ed8bafda2dcafbb8592b14c77bba5a28619fedbf2ee" +
                "d2b391f84ba0be9bd63c287c204097babfdc599b649edcdb49dda62a13a84d88df6eeb6d0ab0260" +
                "b6585cdf5914db8ec3e9a70166c0442d6e18d0a6d32ac3160b17158c8ea3fc7b7dd4c6b32448d07" +
                "e9c6bd4bc12531772f8d7f9e3996699c1eddb2192612caa8f8a9b937275f5c3e857bc72f3d21d35" +
                "182ca216aaa5772d0bb5193894560177bbce86ee70299009596a09b24e64f98c55aead699171eed" +
                "e3aca339c3937c8e90a90b5133ac2e4781f038a239a8bbb571c8b17461ffcb2a5323828bf6006dc" +
                "898cfaf54a9e3f7f6b72f189d06cf244c3796ee7073a469c5fea048455c66ad7542a80dba8ba81e" +
                "2d5f8d9c9ec41e0e9ca5f41f49eb3082017b06092a864886f70d010701a082016c0482016830820" +
                "16430820160060b2a864886f70d010c0a0102a082012830820124305f06092a864886f70d01050d" +
                "3052303106092a864886f70d01050c302404105176b836b18b21bd2e30a5699e48f0c0020208003" +
                "00c06082a864886f70d02090500301d060960864801650304012a0410cbf2e9303c033280aecfe5" +
                "f9fe89dfa50481c044bf20e2d1a8e87d00151d8e1e277e99b1ac9262920a5b6c3b4f02f2cb7329b" +
                "61b4755c4c13b9fdbc0d060cc4394e71234643a19ff22ae636d5f6c59ff04dbeb23c8932cc67387" +
                "272276c331055126ca129e9d2cbcb7fcbb0f3b70bb183e722b4d261865b1278103ee0b4b09b5892" +
                "3109f3a436cbe5059299850bc47cd4869b3c57ef8aae94394761c7caae9e0982db951501ff11938" +
                "6a935465ae5a1bd10b7bd434f67b82781494cc88f846cc72f31b35e56f69fa3f694b34a04056665" +
                "7bba53125302306092a864886f70d01091531160414436855b1c9581d517c38c9a130670cf216e8" +
                "12ae30413031300d06096086480165030402010500042036b2c64f4b8f95ed92087fb6f7d8fab9a" +
                "dcf0dffe1a4a443884a8e094b5ca62604088fa6e47c144a951402020800"

        val PEM = PEM_HEX.hexToByteArray()
        private const val PEM_HEX = "2d2d2d2d2d424547494e2045432050524956415445204b45592d2d2d2d2" +
                "d0a4d49476b416745424244415a4378444b707159362b2b3746645149627434586e354d70327156" +
                "39556a592f39445978684a79686d3831736163392b4d32534c730a667955786b2b623169764f674" +
                "27759464b34454541434b685a414e69414151425a494f4c73636c3478586459442b473030567767" +
                "6f4635714a5955474b5663540a6c412f486e524763367a6c54444b315158314754764c746244526" +
                "239465a5046716d5a71443373534b713575626445553337494f4b35707a396a3856394566780a63" +
                "645a4c3879686d68744353634d505735316c7270727a516755414271644d3d0a2d2d2d2d2d454e4" +
                "42045432050524956415445204b45592d2d2d2d2d0a2d2d2d2d2d424547494e2043455254494649" +
                "434154452d2d2d2d2d0a4d4949434b44434341612b674177494241674952414f5a72494a6a57615" +
                "877627571785a74525a444a324177436759494b6f5a497a6a3045417749775454454c0a4d416b47" +
                "4131554542684d4356564d784454414c42674e564241674d4246526c633351784454414c42674e5" +
                "64241634d4246526c633351784454414c42674e560a42416f4d4246526c63335178455441504267" +
                "4e5642414d4d4346526c633352445a584a304d423458445449314d4467774e7a41314d5441304e3" +
                "16f58445449320a4d4467774e7a41314d5441304e316f775454454c4d416b474131554542684d43" +
                "56564d784454414c42674e564241674d4246526c633351784454414c42674e560a4241634d42465" +
                "26c633351784454414c42674e5642416f4d4246526c633351784554415042674e5642414d4d4346" +
                "526c633352445a584a304d485977454159480a4b6f5a497a6a3043415159464b344545414349445" +
                "9674145415753446937484a654d563357412f68744e4663494b42656169574642696c5845355150" +
                "783530520a6e4f733555777974554639526b377937577730572f5257547861706d6167393745697" +
                "175626d3352464e2b7944697561632f592f4666524838584857532f4d6f0a5a6f62516b6e444431" +
                "75645a613661383049464141616e546f314d775554416442674e5648513445466751555035794b7" +
                "6392f41726a684d52396d446b3248540a5a35734b35703077487759445652306a42426777466f41" +
                "555035794b76392f41726a684d52396d446b3248545a35734b35703077447759445652305441514" +
                "82f0a42415577417745422f7a414b42676771686b6a4f5051514441674e6e4144426b416a414634" +
                "735a547433367358364157307476436877334e49613444752b56670a6433727841774f3237796f5" +
                "74d423450586953764c3967766264584a756847526b436f434d41544d314a714e39535a65577537" +
                "5a2b31554a577241566e414e670a4e747735624562416c44546a6876315771756c33765835464e7" +
                "63463354442742f74635438413d3d0a2d2d2d2d2d454e442043455254494649434154452d2d2d2d" +
                "2d0a"

        val PEM_ENC = PEM_ENC_HEX.hexToByteArray()
        private const val PEM_ENC_HEX = "2d2d2d2d2d424547494e2045432050524956415445204b45592d2d2" +
                "d2d2d0a50726f632d547970653a20342c454e435259505445440a44454b2d496e666f3a20414553" +
                "2d3235362d4342432c4341453545303036354330384536453642333046363239433834463130453" +
                "9340a0a354847727674385844534a38676c49335a3951767a7145324874666d634d6a61376b5058" +
                "52716e724a4d4558617a6b5a4f4735784b6c313357414a66596356320a31774a776834466f56565" +
                "172364e3135506c5364434b744963322f2f70503950433732584c49376157714234644f51495441" +
                "493555452f6961536d4939526e2f0a41644b4f464456373472466558497251692f4e2f7153624b6" +
                "14c477575374a596d594a394746552b5741516758703131757345794d7758486259386351715266" +
                "0a4c6f377a752b624739665142414b72716449676335564135464f7972303546424478494a4d623" +
                "5464455553d0a2d2d2d2d2d454e442045432050524956415445204b45592d2d2d2d2d0a2d2d2d2d" +
                "2d424547494e2043455254494649434154452d2d2d2d2d0a4d4949434b44434341612b674177494" +
                "241674952414f5a72494a6a57615877627571785a74525a444a324177436759494b6f5a497a6a30" +
                "45417749775454454c0a4d416b474131554542684d4356564d784454414c42674e564241674d424" +
                "6526c633351784454414c42674e564241634d4246526c633351784454414c42674e560a42416f4d" +
                "4246526c633351784554415042674e5642414d4d4346526c633352445a584a304d4234584454493" +
                "14d4467774e7a41314d5441304e316f58445449320a4d4467774e7a41314d5441304e316f775454" +
                "454c4d416b474131554542684d4356564d784454414c42674e564241674d4246526c63335178445" +
                "4414c42674e560a4241634d4246526c633351784454414c42674e5642416f4d4246526c63335178" +
                "4554415042674e5642414d4d4346526c633352445a584a304d485977454159480a4b6f5a497a6a3" +
                "043415159464b3445454143494459674145415753446937484a654d563357412f68744e4663494b" +
                "42656169574642696c5845355150783530520a6e4f733555777974554639526b377937577730572" +
                "f5257547861706d6167393745697175626d3352464e2b7944697561632f592f4666524838584857" +
                "532f4d6f0a5a6f62516b6e44443175645a613661383049464141616e546f314d775554416442674" +
                "e5648513445466751555035794b76392f41726a684d52396d446b3248540a5a35734b3570307748" +
                "7759445652306a42426777466f41555035794b76392f41726a684d52396d446b3248545a35734b3" +
                "570307744775944565230544151482f0a42415577417745422f7a414b42676771686b6a4f505151" +
                "4441674e6e4144426b416a414634735a547433367358364157307476436877334e49613444752b5" +
                "6670a6433727841774f3237796f574d423450586953764c3967766264584a756847526b436f434d" +
                "41544d314a714e39535a655775375a2b31554a577241566e414e670a4e747735624562416c44546" +
                "a6876315771756c33765835464e763463354442742f74635438413d3d0a2d2d2d2d2d454e442043" +
                "455254494649434154452d2d2d2d2d0a"

        val DER_CERT = DER_CERT_HEX.hexToByteArray()
        private const val DER_CERT_HEX = "30820228308201afa003020102021100e66b2098d6697c1bbaac59" +
                "b516432760300a06082a8648ce3d040302304d310b3009060355040613025553310d300b0603550" +
                "4080c0454657374310d300b06035504070c0454657374310d300b060355040a0c04546573743111" +
                "300f06035504030c085465737443657274301e170d3235303830373035313034375a170d3236303" +
                "830373035313034375a304d310b3009060355040613025553310d300b06035504080c0454657374" +
                "310d300b06035504070c0454657374310d300b060355040a0c04546573743111300f06035504030" +
                "c0854657374436572743076301006072a8648ce3d020106052b81040022036200040164838bb1c9" +
                "78c577580fe1b4d15c20a05e6a258506295713940fc79d119ceb39530cad505f5193bcbb5b0d16f" +
                "d1593c5aa666a0f7b122aae6e6dd114dfb20e2b9a73f63f15f447f171d64bf3286686d09270c3d6" +
                "e7596ba6bcd0814001a9d3a3533051301d0603551d0e041604143f9c8abfdfc0ae384c47d983936" +
                "1d3679b0ae69d301f0603551d230418301680143f9c8abfdfc0ae384c47d9839361d3679b0ae69d" +
                "300f0603551d130101ff040530030101ff300a06082a8648ce3d0403020367003064023005e2c65" +
                "3b77eac5fa016d2dbc2870dcd21ae03bbe560777af10303b6ef2a16301e0f5e24af2fd82f6dd5c9" +
                "ba1191902a023004ccd49a8df5265e5aeed9fb55095ab0159c036036dc396c46c09434e386fd56a" +
                "ae977bd7e4536fe1ce4306dfed713f0"

        val DER_KEY = DER_KEY_HEX.hexToByteArray()
        private const val DER_KEY_HEX = "3081a40201010430190b10caa6a63afbeec575021bb785e7e4ca76a" +
                "95f548d8ffd0d8c61272866f35b1a73df8cd922ec7f253193e6f58af3a00706052b81040022a164" +
                "036200040164838bb1c978c577580fe1b4d15c20a05e6a258506295713940fc79d119ceb39530ca" +
                "d505f5193bcbb5b0d16fd1593c5aa666a0f7b122aae6e6dd114dfb20e2b9a73f63f15f447f171d6" +
                "4bf3286686d09270c3d6e7596ba6bcd0814001a9d3"

    }

}