import 'dart:io';

import 'package:args/args.dart';

void main(List<String> arguments) {
  var parser =
      ArgParser()..addFlag(
        'help',
        abbr: 'h',
        negatable: false,
        help: 'Shows this usage information.',
      );

  // Command: generate-conservative
  // This command now expects named options. 'mandatory: true' ensures they are provided.
  var generateCommand =
      ArgParser()
        ..addFlag(
          'help',
          abbr: 'h',
          negatable: false,
          help: 'Shows usage for this command.',
        )
        ..addOption(
          'mutations-directory',
          abbr: 'm',
          help: 'The main directory with mutation data (required).',
          valueHelp: 'dir',
          mandatory: true,
        )
        ..addMultiOption(
          'analysis-directories',
          abbr: 'a',
          help: 'A list of existing directories to use for analysis.',
          valueHelp: 'dir1,dir2,...',
          defaultsTo: [],
        );
  parser.addCommand('generate-conservative', generateCommand);
  commandParsers['generate-conservative'] = generateCommand;

  // Command: verify-fine
  var verifyCommand =
      ArgParser()..addFlag(
        'help',
        abbr: 'h',
        negatable: false,
        help: 'Shows usage for this command.',
      );
  parser.addCommand('verify-fine', verifyCommand);
  commandParsers['verify-fine'] = verifyCommand;

  ArgResults argResults;
  try {
    // The parser will throw a FormatException if a mandatory option is missing
    // or if unexpected positional arguments are provided for a command.
    argResults = parser.parse(arguments);
  } on FormatException catch (e) {
    print('Error: ${e.message}\n');
    // If the error is about a command, show that command's usage.
    if (arguments.isNotEmpty && commandParsers.containsKey(arguments.first)) {
      printCommandUsage(arguments.first);
    } else {
      printUsage(parser);
    }
    exit(1);
  }

  // Handle command-specific logic first.
  if (argResults.command != null) {
    var commandName = argResults.command!.name!;
    var commandArgs = argResults.command!;

    // Handle command-specific --help flag.
    if (commandArgs['help'] as bool) {
      printCommandUsage(commandName);
      exit(0);
    }

    // NEW: Check for unexpected positional arguments for commands that don't expect them.
    if (commandName == 'generate-conservative' && commandArgs.rest.isNotEmpty) {
      print(
        'Error: The "generate-conservative" command received unexpected positional arguments.',
      );
      print(
        'Please use named options like --mutations-directory and --analysis-directories.\n',
      );
      printCommandUsage(commandName);
      exit(1);
    }

    // Execute the command.
    switch (commandName) {
      case 'generate-conservative':
        handleGenerateConservative(commandArgs);
      case 'verify-fine':
        handleVerifyFine(commandArgs);
    }
  } else {
    // If no command was used, handle top-level flags.
    if (argResults['help'] as bool) {
      printUsage(parser);
      exit(0);
    }

    // If we reach here, no command was given and it wasn't a request for help.
    print('Error: No command specified.\n');
    printUsage(parser);
    exit(1);
  }
}

// A map to hold the parsers for each command.
// This makes it easier to retrieve the correct parser for printing help messages.
final commandParsers = <String, ArgParser>{};

/// Handles the logic for the "generate-conservative" command.
void handleGenerateConservative(ArgResults args) {
  // The 'mandatory: true' flag in the parser ensures this value exists.
  var mainDirPath = args['mutations-directory'] as String;
  var mainDir = Directory(mainDirPath);

  if (!mainDir.existsSync()) {
    print(
      'Error: The specified mutations directory does not exist: $mainDirPath',
    );
    exit(1);
  }

  print('Running "generate-conservative"...');
  print('Mutations directory: ${mainDir.path}');

  var analysisDirsPaths = args['analysis-directories'] as List<String>;
  if (analysisDirsPaths.isNotEmpty) {
    print('Processing analysis directories:');
    for (var dirPath in analysisDirsPaths) {
      var dir = Directory(dirPath);
      if (dir.existsSync()) {
        print('  - Found: ${dir.path}');
      } else {
        print('  - Warning: Directory not found, skipping: $dirPath');
      }
    }
  } else {
    print('No analysis directories provided.');
  }

  print('\nGeneration logic would run now.');
}

/// Handles the logic for the "verify-fine" command.
void handleVerifyFine(ArgResults args) {
  print('Running "verify-fine"...');
  print('\nVerification logic would run now.');
}

/// Prints the usage information for a specific command.
void printCommandUsage(String commandName) {
  var parser = commandParsers[commandName];
  if (parser == null) {
    print('Error: Unknown command "$commandName"');
    exit(1);
  }

  print('''
Usage: dart your_script_name.dart $commandName [options]

Options for $commandName:
${parser.usage}
''');
}

/// Prints the general usage information for the entire tool.
void printUsage(ArgParser parser) {
  print('''
Usage: dart your_script_name.dart <command> [options]

A tool for managing fine-grained testing.

Available commands:
  generate-conservative   Generates files based on directories.
  verify-fine             Verifies the generated files.

Use "dart your_script_name.dart <command> --help" for more information about a command.

Global options:
${parser.usage}
''');
}
