/*
 * Copyright (C) 2022 Yubico.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import 'package:freezed_annotation/freezed_annotation.dart';

part 'models.freezed.dart';
part 'models.g.dart';

enum InteractionEvent { remove, insert, touch }

@freezed
abstract class FidoState with _$FidoState {
  const FidoState._();

  factory FidoState({
    required Map<String, dynamic> info,
    required bool unlocked,
    int? pinRetries,
  }) = _FidoState;

  factory FidoState.fromJson(Map<String, dynamic> json) =>
      _$FidoStateFromJson(json);

  bool get hasPin => info['options']['clientPin'] == true;

  int get minPinLength => info['min_pin_length'] as int;

  bool get credMgmt =>
      info['options']['credMgmt'] == true ||
      info['options']['credentialMgmtPreview'] == true;

  int? get remainingCreds => info['remaining_disc_creds'];

  bool? get bioEnroll => info['options']['bioEnroll'];

  bool get alwaysUv => info['options']['alwaysUv'] == true;

  bool get forcePinChange => info['force_pin_change'] == true;

  bool get pinBlocked => pinRetries == 0;

  bool? get enterpriseAttestation => info['options']['ep'];
}

@freezed
sealed class PinResult with _$PinResult {
  factory PinResult.success() = PinResultSuccess;
  factory PinResult.failed(FidoPinFailureReason reason) = PinResultFailure;
}

@freezed
sealed class FidoPinFailureReason with _$FidoPinFailureReason {
  factory FidoPinFailureReason.invalidPin(int retries, bool authBlocked) =
      FidoInvalidPin;
  const factory FidoPinFailureReason.weakPin() = FidoWeakPin;
}

@freezed
abstract class Fingerprint with _$Fingerprint {
  const Fingerprint._();
  factory Fingerprint(String templateId, String? name) = _Fingerprint;

  factory Fingerprint.fromJson(Map<String, dynamic> json) =>
      _$FingerprintFromJson(json);

  String get label => name ?? 'Unnamed (ID: $templateId)';
}

@freezed
sealed class FingerprintEvent with _$FingerprintEvent {
  factory FingerprintEvent.capture(int remaining) = FingerprintEventCapture;
  factory FingerprintEvent.complete(Fingerprint fingerprint) =
      FingerprintEventComplete;
  factory FingerprintEvent.error(int code) = FingerprintEventError;
}

@freezed
abstract class FidoCredential with _$FidoCredential {
  factory FidoCredential({
    required String rpId,
    required String credentialId,
    required String userId,
    required String userName,
    String? displayName,
  }) = _FidoCredential;

  factory FidoCredential.fromJson(Map<String, dynamic> json) =>
      _$FidoCredentialFromJson(json);
}
