/*
 * Wire
 * Copyright (C) 2024 Wire Swiss GmbH
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see http://www.gnu.org/licenses/.
 */

package com.wire.kalium.persistence.dao.conversation

import app.cash.sqldelight.coroutines.asFlow
import com.wire.kalium.persistence.ConversationDetailsQueries
import com.wire.kalium.persistence.ConversationDetailsWithEventsQueries
import com.wire.kalium.persistence.ConversationsQueries
import com.wire.kalium.persistence.MembersQueries
import com.wire.kalium.persistence.UnreadEventsQueries
import com.wire.kalium.persistence.cache.FlowCache
import com.wire.kalium.persistence.dao.ConversationIDEntity
import com.wire.kalium.persistence.dao.QualifiedIDEntity
import com.wire.kalium.persistence.dao.UserIDEntity
import com.wire.kalium.persistence.db.ReadDispatcher
import com.wire.kalium.persistence.db.WriteDispatcher
import com.wire.kalium.persistence.util.mapToList
import com.wire.kalium.persistence.util.mapToOne
import com.wire.kalium.persistence.util.mapToOneOrDefault
import com.wire.kalium.persistence.util.mapToOneOrNull
import com.wire.kalium.util.DateTimeUtil
import com.wire.kalium.util.DateTimeUtil.toIsoDateTimeString
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.flowOn
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.withContext
import kotlinx.datetime.Instant
import kotlinx.datetime.toInstant
import kotlin.time.Duration

internal const val MLS_DEFAULT_EPOCH = 0L
internal const val MLS_DEFAULT_LAST_KEY_MATERIAL_UPDATE_MILLI = 0L
internal val MLS_DEFAULT_CIPHER_SUITE = ConversationEntity.CipherSuite.MLS_128_DHKEMX25519_AES128GCM_SHA256_Ed25519

// TODO: Refactor. We can split this into smaller DAOs.
//       For example, one for Members, one for Protocol/MLS-related things, etc.
//       Even if they operate on the same table underneath, these DAOs can represent/do different things.
@Suppress("TooManyFunctions", "LongParameterList")
internal class ConversationDAOImpl internal constructor(
    private val conversationDetailsCache: FlowCache<ConversationIDEntity, ConversationViewEntity?>,
    private val conversationCache: FlowCache<ConversationIDEntity, ConversationEntity?>,
    private val conversationQueries: ConversationsQueries,
    private val conversationDetailsQueries: ConversationDetailsQueries,
    private val conversationDetailsWithEventsQueries: ConversationDetailsWithEventsQueries,
    private val memberQueries: MembersQueries,
    private val unreadEventsQueries: UnreadEventsQueries,
    private val readDispatcher: ReadDispatcher,
    private val writeDispatcher: WriteDispatcher,
) : ConversationDAO {
    private val conversationMapper = ConversationMapper
    private val conversationDetailsWithEventsMapper = ConversationDetailsWithEventsMapper
    override val platformExtensions: ConversationExtensions =
        ConversationExtensionsImpl(conversationDetailsWithEventsQueries, conversationDetailsWithEventsMapper, readDispatcher)

    // region Get/Observe by ID

    override suspend fun observeConversationById(
        qualifiedID: QualifiedIDEntity
    ): Flow<ConversationEntity?> =
        conversationCache.get(qualifiedID) {
            conversationQueries.selectConversationByQualifiedId(qualifiedID, conversationMapper::fromViewToModel)
                .asFlow()
                .flowOn(readDispatcher.value)
                .mapToOneOrNull()
        }

    override suspend fun getConversationById(
        qualifiedID: QualifiedIDEntity
    ): ConversationEntity? = observeConversationById(qualifiedID).first()

    override suspend fun observeConversationDetailsById(
        conversationId: QualifiedIDEntity
    ): Flow<ConversationViewEntity?> = conversationDetailsCache.get(conversationId) {
        conversationDetailsQueries.selectConversationDetailsByQualifiedId(conversationId, conversationMapper::fromViewToModel)
            .asFlow()
            .flowOn(readDispatcher.value)
            .mapToOneOrNull()
    }

    override suspend fun getConversationDetailsById(
        qualifiedID: QualifiedIDEntity
    ): ConversationViewEntity? =
        observeConversationDetailsById(qualifiedID).first()

    // endregion

    override suspend fun getSelfConversationId(protocol: ConversationEntity.Protocol) = withContext(readDispatcher.value) {
        conversationQueries.selfConversationId(protocol).executeAsOneOrNull()
    }

    override suspend fun getE2EIConversationClientInfoByClientId(clientId: String): E2EIConversationClientInfoEntity? =
        withContext(readDispatcher.value) {
            conversationQueries.getMLSGroupIdAndUserIdByClientId(clientId, conversationMapper::toE2EIConversationClient)
                .executeAsOneOrNull()
        }

    override suspend fun getMLSGroupIdByUserId(userId: UserIDEntity): String? =
        withContext(readDispatcher.value) {
            conversationQueries.getMLSGroupIdByUserId(userId)
                .executeAsOneOrNull()
        }

    override suspend fun getMLSGroupIdByConversationId(conversationId: QualifiedIDEntity): String? =
        withContext(readDispatcher.value) {
            conversationQueries.getMLSGroupIdByConversationId(conversationId)
                .executeAsOneOrNull()
                ?.mls_group_id
        }

    override suspend fun insertConversation(conversationEntity: ConversationEntity) = withContext(writeDispatcher.value) {
        nonSuspendingInsertConversation(conversationEntity)
    }

    override suspend fun insertConversations(conversationEntities: List<ConversationEntity>) = withContext(writeDispatcher.value) {
        conversationQueries.transaction {
            for (conversationEntity: ConversationEntity in conversationEntities) {
                nonSuspendingInsertConversation(conversationEntity)
            }
        }
    }

    override suspend fun insertOrUpdateLastModified(conversationEntities: List<ConversationEntity>) = withContext(writeDispatcher.value) {
        conversationQueries.transaction {
            for (conversationEntity: ConversationEntity in conversationEntities) {
                with(conversationEntity) {
                    conversationQueries.insertConversationOrUpdateLastModifiedDate(
                        qualified_id = id,
                        name = name,
                        type = type,
                        team_id = teamId,
                        mls_group_id = if (protocolInfo is ConversationEntity.ProtocolInfo.MLSCapable) protocolInfo.groupId
                        else null,
                        mls_group_state = if (protocolInfo is ConversationEntity.ProtocolInfo.MLSCapable) protocolInfo.groupState
                        else ConversationEntity.GroupState.ESTABLISHED,
                        mls_epoch = if (protocolInfo is ConversationEntity.ProtocolInfo.MLSCapable) protocolInfo.epoch.toLong()
                        else MLS_DEFAULT_EPOCH,
                        protocol = when (protocolInfo) {
                            is ConversationEntity.ProtocolInfo.MLS -> ConversationEntity.Protocol.MLS
                            is ConversationEntity.ProtocolInfo.Mixed -> ConversationEntity.Protocol.MIXED
                            is ConversationEntity.ProtocolInfo.Proteus -> ConversationEntity.Protocol.PROTEUS
                        },
                        muted_status = mutedStatus,
                        muted_time = mutedTime,
                        creator_id = creatorId,
                        last_modified_date = lastModifiedDate,
                        last_notified_date = lastNotificationDate,
                        access_list = access,
                        access_role_list = accessRole,
                        last_read_date = lastReadDate,
                        mls_last_keying_material_update_date = if (protocolInfo is ConversationEntity.ProtocolInfo.MLSCapable)
                            protocolInfo.keyingMaterialLastUpdate
                        else Instant.fromEpochMilliseconds(MLS_DEFAULT_LAST_KEY_MATERIAL_UPDATE_MILLI),
                        mls_cipher_suite = if (protocolInfo is ConversationEntity.ProtocolInfo.MLSCapable) protocolInfo.cipherSuite
                        else MLS_DEFAULT_CIPHER_SUITE,
                        receipt_mode = receiptMode,
                        message_timer = messageTimer,
                        user_message_timer = userMessageTimer,
                        incomplete_metadata = hasIncompleteMetadata,
                        archived = archived,
                        archived_date_time = archivedInstant,
                        is_channel = isChannel,
                        channel_access = channelAccess,
                        channel_add_permission = channelAddPermission,
                        wire_cell = wireCell,
                    )
                }
            }
        }
    }

    private fun nonSuspendingInsertConversation(conversationEntity: ConversationEntity) {
        with(conversationEntity) {
            conversationQueries.insertConversation(
                qualified_id = id,
                name = name,
                type = type,
                team_id = teamId,
                mls_group_id = if (protocolInfo is ConversationEntity.ProtocolInfo.MLSCapable) protocolInfo.groupId
                else null,
                mls_group_state = if (protocolInfo is ConversationEntity.ProtocolInfo.MLSCapable) protocolInfo.groupState
                else ConversationEntity.GroupState.ESTABLISHED,
                mls_epoch = if (protocolInfo is ConversationEntity.ProtocolInfo.MLSCapable) protocolInfo.epoch.toLong()
                else MLS_DEFAULT_EPOCH,
                protocol = when (protocolInfo) {
                    is ConversationEntity.ProtocolInfo.MLS -> ConversationEntity.Protocol.MLS
                    is ConversationEntity.ProtocolInfo.Mixed -> ConversationEntity.Protocol.MIXED
                    is ConversationEntity.ProtocolInfo.Proteus -> ConversationEntity.Protocol.PROTEUS
                },
                muted_status = mutedStatus,
                muted_time = mutedTime,
                creator_id = creatorId,
                last_modified_date = lastModifiedDate,
                last_notified_date = lastNotificationDate,
                access_list = access,
                access_role_list = accessRole,
                last_read_date = lastReadDate,
                mls_last_keying_material_update_date = if (protocolInfo is ConversationEntity.ProtocolInfo.MLSCapable)
                    protocolInfo.keyingMaterialLastUpdate
                else Instant.fromEpochMilliseconds(MLS_DEFAULT_LAST_KEY_MATERIAL_UPDATE_MILLI),
                mls_cipher_suite = if (protocolInfo is ConversationEntity.ProtocolInfo.MLSCapable) protocolInfo.cipherSuite
                else MLS_DEFAULT_CIPHER_SUITE,
                receipt_mode = receiptMode,
                message_timer = messageTimer,
                user_message_timer = userMessageTimer,
                incomplete_metadata = hasIncompleteMetadata,
                archived = archived,
                archived_date_time = archivedInstant,
                is_channel = isChannel,
                channel_access = channelAccess,
                channel_add_permission = channelAddPermission,
                wire_cell = wireCell,
                history_sharing_retention_seconds = historySharingRetentionSeconds,
            )
        }
    }

    override suspend fun updateConversation(conversationEntity: ConversationEntity) = withContext(writeDispatcher.value) {
        conversationQueries.updateConversation(
            conversationEntity.name,
            conversationEntity.type,
            conversationEntity.teamId,
            conversationEntity.id
        )
    }

    override suspend fun updateConversationGroupState(groupState: ConversationEntity.GroupState, groupId: String) =
        withContext(writeDispatcher.value) {
            conversationQueries.updateConversationGroupState(groupState, groupId)
        }

    override suspend fun updateMlsGroupStateAndCipherSuite(
        groupState: ConversationEntity.GroupState,
        cipherSuite: ConversationEntity.CipherSuite,
        groupId: String
    ) = withContext(writeDispatcher.value) {
        conversationQueries.updateMlsGroupStateAndCipherSuite(groupState, cipherSuite, groupId)
    }

    override suspend fun updateMLSGroupIdAndState(
        conversationId: QualifiedIDEntity,
        newGroupId: String,
        newEpoch: Long,
        groupState: ConversationEntity.GroupState
    ) = withContext(writeDispatcher.value) {
        conversationQueries.updateMLSGroupIdAndState(newGroupId, groupState, newEpoch, conversationId)
    }

    override suspend fun updateConversationModifiedDate(qualifiedID: QualifiedIDEntity, date: Instant) =
        withContext(writeDispatcher.value) {
            conversationQueries.updateConversationModifiedDate(date, qualifiedID)
        }

    override suspend fun updateConversationNotificationDate(qualifiedID: QualifiedIDEntity) = withContext(writeDispatcher.value) {
        conversationQueries.updateConversationNotificationsDateWithTheLastMessage(qualifiedID)
    }

    override suspend fun updateAllConversationsNotificationDate() = withContext(writeDispatcher.value) {
        conversationQueries.updateAllNotifiedConversationsNotificationsDate()
    }

    override suspend fun getAllConversations(): Flow<List<ConversationEntity>> {
        return conversationQueries.selectAllConversations(conversationMapper::fromViewToModel)
            .asFlow()
            .mapToList()
            .flowOn(readDispatcher.value)
    }

    override suspend fun getAllConversationDetails(
        fromArchive: Boolean,
        filter: ConversationFilterEntity
    ): Flow<List<ConversationViewEntity>> {
        return conversationDetailsQueries.selectAllConversationDetails(fromArchive, filter.toString(), conversationMapper::fromViewToModel)
            .asFlow()
            .mapToList()
            .flowOn(readDispatcher.value)
    }

    override suspend fun getAllConversationDetailsWithEvents(
        fromArchive: Boolean,
        onlyInteractionEnabled: Boolean,
        newActivitiesOnTop: Boolean,
        strictMLSFilter: Boolean,
    ): Flow<List<ConversationDetailsWithEventsEntity>> {
        return conversationDetailsWithEventsQueries.selectAllConversationDetailsWithEvents(
            fromArchive = fromArchive,
            onlyInteractionsEnabled = onlyInteractionEnabled,
            newActivitiesOnTop = newActivitiesOnTop,
            strict_mls = if (strictMLSFilter) 1 else 0,
            mapper = conversationDetailsWithEventsMapper::fromViewToModel
        ).asFlow()
            .mapToList()
            .flowOn(readDispatcher.value)
    }

    override suspend fun getCellName(conversationId: QualifiedIDEntity): String? = withContext(readDispatcher.value) {
        conversationQueries.getCellName(conversationId).executeAsOneOrNull()?.wire_cell
    }

    override suspend fun getConversationIds(
        type: ConversationEntity.Type,
        protocol: ConversationEntity.Protocol,
        teamId: String?
    ): List<QualifiedIDEntity> {
        return withContext(readDispatcher.value) {
            conversationQueries.selectConversationIds(protocol, type, teamId).executeAsList()
        }
    }

    override suspend fun getTeamConversationIdsReadyToCompleteMigration(teamId: String): List<QualifiedIDEntity> {
        return withContext(readDispatcher.value) {
            conversationQueries.selectAllTeamProteusConversationsReadyForMigration(teamId)
                .executeAsList()
                .map { it.qualified_id }
        }
    }

    override suspend fun getOneOnOneConversationIdsWithOtherUser(
        userId: UserIDEntity,
        protocol: ConversationEntity.Protocol
    ): List<QualifiedIDEntity> =
        withContext(readDispatcher.value) {
            conversationQueries.selectOneOnOneConversationIdsByProtocol(protocol, userId).executeAsList()
        }

    override suspend fun observeOneOnOneConversationWithOtherUser(userId: UserIDEntity): Flow<ConversationEntity?> {
        return conversationQueries.selectActiveOneOnOneConversation(userId, conversationMapper::fromViewToModel)
            .asFlow()
            .mapToOneOrNull()
            .flowOn(readDispatcher.value)
    }

    override suspend fun observeOneOnOneConversationDetailsWithOtherUser(userId: UserIDEntity): Flow<ConversationViewEntity?> {
        return conversationDetailsQueries.selectActiveOneOnOneConversationDetails(userId, conversationMapper::fromViewToModel)
            .asFlow()
            .mapToOneOrNull()
            .flowOn(readDispatcher.value)
    }

    override suspend fun getConversationProtocolInfo(qualifiedID: QualifiedIDEntity): ConversationEntity.ProtocolInfo? =
        withContext(readDispatcher.value) {
            conversationQueries.selectProtocolInfoByQualifiedId(qualifiedID, conversationMapper::mapProtocolInfo).executeAsOneOrNull()
        }

    override suspend fun getConversationByGroupID(groupID: String): ConversationEntity? = withContext(readDispatcher.value) {
        conversationQueries.selectByGroupId(groupID, mapper = conversationMapper::toConversationEntity)
            .executeAsOneOrNull()
    }

    override suspend fun getConversationIdByGroupID(groupID: String) = withContext(readDispatcher.value) {
        conversationQueries.getConversationIdByGroupId(groupID).executeAsOneOrNull()
    }

    override suspend fun getConversationsByGroupState(groupState: ConversationEntity.GroupState): List<ConversationEntity> =
        withContext(readDispatcher.value) {
            conversationQueries.selectByGroupState(groupState, conversationMapper::fromViewToModel)
                .executeAsList()
        }

    override suspend fun deleteConversationByQualifiedID(qualifiedID: QualifiedIDEntity) = withContext(writeDispatcher.value) {
        conversationQueries.transactionWithResult {
            conversationQueries.deleteConversation(qualifiedID)
            conversationQueries.selectChanges().executeAsOne() > 0
        }
    }

    override suspend fun markConversationAsDeletedLocally(qualifiedID: QualifiedIDEntity) = withContext(writeDispatcher.value) {
        conversationQueries.transactionWithResult {
            conversationQueries.markAsDeletedLocally(qualifiedID)
            conversationQueries.selectChanges().executeAsOne() > 0
        }
    }

    override suspend fun updateConversationMutedStatus(
        conversationId: QualifiedIDEntity,
        mutedStatus: ConversationEntity.MutedStatus,
        mutedStatusTimestamp: Long
    ) = withContext(writeDispatcher.value) {
        conversationQueries.updateConversationMutingStatus(
            mutedStatus,
            mutedStatusTimestamp,
            conversationId
        )
    }

    override suspend fun updateConversationArchivedStatus(
        conversationId: QualifiedIDEntity,
        isArchived: Boolean,
        archivedStatusTimestamp: Long
    ) = withContext(writeDispatcher.value) {
        conversationQueries.updateConversationArchivingStatus(
            isArchived,
            archivedStatusTimestamp.toIsoDateTimeString().toInstant(),
            conversationId
        )
    }

    override suspend fun updateAccess(
        conversationID: QualifiedIDEntity,
        accessList: List<ConversationEntity.Access>,
        accessRoleList: List<ConversationEntity.AccessRole>
    ) = withContext(writeDispatcher.value) {
        conversationQueries.updateAccess(accessList, accessRoleList, conversationID)
    }

    override suspend fun updateConversationReadDate(conversationID: QualifiedIDEntity, date: Instant) = withContext(writeDispatcher.value) {
        unreadEventsQueries.deleteUnreadEvents(date, conversationID)
        conversationQueries.updateConversationReadDate(date, conversationID)
    }

    override suspend fun updateKeyingMaterial(groupId: String, timestamp: Instant) = withContext(writeDispatcher.value) {
        conversationQueries.updateKeyingMaterialDate(timestamp, groupId)
    }

    override suspend fun getConversationsByKeyingMaterialUpdate(threshold: Duration): List<String> = withContext(readDispatcher.value) {
        conversationQueries.selectByKeyingMaterialUpdate(
            ConversationEntity.GroupState.ESTABLISHED,
            DateTimeUtil.currentInstant().minus(threshold)
        ).executeAsList()
    }

    override suspend fun setProposalTimer(proposalTimer: ProposalTimerEntity) = withContext(writeDispatcher.value) {
        conversationQueries.updateProposalTimer(proposalTimer.firingDate.toString(), proposalTimer.groupID)
    }

    override suspend fun clearProposalTimer(groupID: String) = withContext(writeDispatcher.value) {
        conversationQueries.clearProposalTimer(groupID)
    }

    override suspend fun getProposalTimers(): Flow<List<ProposalTimerEntity>> {
        return conversationQueries.selectProposalTimers()
            .asFlow()
            .flowOn(readDispatcher.value)
            .mapToList()
            .map { list -> list.map { ProposalTimerEntity(it.mls_group_id, it.mls_proposal_timer.toInstant()) } }
    }

    override suspend fun whoDeletedMeInConversation(conversationId: QualifiedIDEntity, selfUserIdString: String): UserIDEntity? =
        withContext(readDispatcher.value) {
            conversationQueries.whoDeletedMeInConversation(conversationId, selfUserIdString).executeAsOneOrNull()
        }

    override suspend fun updateConversationName(conversationId: QualifiedIDEntity, conversationName: String, dateTime: Instant) =
        withContext(writeDispatcher.value) {
            conversationQueries.updateConversationName(conversationName, dateTime, conversationId)
        }

    override suspend fun updateConversationType(conversationID: QualifiedIDEntity, type: ConversationEntity.Type) =
        withContext(writeDispatcher.value) {
            conversationQueries.updateConversationType(type, conversationID)
        }

    override suspend fun updateConversationProtocolAndCipherSuite(
        conversationId: QualifiedIDEntity,
        groupID: String?,
        protocol: ConversationEntity.Protocol,
        cipherSuite: ConversationEntity.CipherSuite
    ): Boolean {
        return withContext(writeDispatcher.value) {
            conversationQueries.updateConversationGroupIdAndProtocolInfo(
                groupID,
                protocol,
                cipherSuite,
                conversationId
            ).executeAsOne() > 0
        }
    }

    override suspend fun getConversationsByUserId(userId: UserIDEntity): List<ConversationEntity> = withContext(readDispatcher.value) {
        memberQueries.selectConversationsByMember(userId, conversationMapper::fromViewToModel).executeAsList()
    }

    override suspend fun updateConversationReceiptMode(conversationID: QualifiedIDEntity, receiptMode: ConversationEntity.ReceiptMode) =
        withContext(writeDispatcher.value) {
            conversationQueries.updateConversationReceiptMode(receiptMode, conversationID)
        }

    override suspend fun updateGuestRoomLink(
        conversationId: QualifiedIDEntity,
        link: String,
        isPasswordProtected: Boolean
    ) = withContext(writeDispatcher.value) {
        conversationQueries.updateGuestRoomLink(link, isPasswordProtected, conversationId)
    }

    override suspend fun deleteGuestRoomLink(conversationId: QualifiedIDEntity) = withContext(writeDispatcher.value) {
        conversationQueries.updateGuestRoomLink(null, false, conversationId)
    }

    override suspend fun observeGuestRoomLinkByConversationId(conversationId: QualifiedIDEntity): Flow<ConversationGuestLinkEntity?> =
        conversationQueries.getGuestRoomLinkByConversationId(conversationId).asFlow().mapToOneOrNull().map {
            it?.guest_room_link?.let { link -> ConversationGuestLinkEntity(link, it.is_guest_password_protected) }
        }.flowOn(readDispatcher.value)

    override suspend fun updateMessageTimer(conversationId: QualifiedIDEntity, messageTimer: Long?) = withContext(writeDispatcher.value) {
        conversationQueries.updateMessageTimer(messageTimer, conversationId)
    }

    override suspend fun updateUserMessageTimer(
        conversationId: QualifiedIDEntity,
        messageTimer: Long?
    ) = withContext(writeDispatcher.value) {
        conversationQueries.updateUserMessageTimer(messageTimer, conversationId)
    }

    override suspend fun getConversationsWithoutMetadata(): List<QualifiedIDEntity> = withContext(readDispatcher.value) {
        conversationQueries.selectConversationIdsWithoutMetadata().executeAsList()
    }

    override suspend fun updateDegradedConversationNotifiedFlag(conversationId: QualifiedIDEntity, updateFlag: Boolean) =
        withContext(writeDispatcher.value) {
            conversationQueries.updateDegradedConversationNotifiedFlag(updateFlag, conversationId)
        }

    override suspend fun observeDegradedConversationNotified(conversationId: QualifiedIDEntity): Flow<Boolean> =
        conversationQueries.selectDegradedConversationNotified(conversationId)
            .asFlow()
            .mapToOneOrDefault(true)
            .flowOn(readDispatcher.value)

    override suspend fun clearContent(conversationId: QualifiedIDEntity) = withContext(writeDispatcher.value) {
        conversationQueries.clearContent(conversationId)
    }

    override suspend fun updateMlsVerificationStatus(
        verificationStatus: ConversationEntity.VerificationStatus,
        conversationId: QualifiedIDEntity
    ) = withContext(writeDispatcher.value) {
        conversationQueries.updateMlsVerificationStatus(verificationStatus, conversationId)
    }

    override suspend fun observeUnreadArchivedConversationsCount(): Flow<Long> =
        unreadEventsQueries.getUnreadArchivedConversationsCount().asFlow()
            .flowOn(readDispatcher.value)
            .mapToOne()

    override suspend fun updateLegalHoldStatus(
        conversationId: QualifiedIDEntity,
        legalHoldStatus: ConversationEntity.LegalHoldStatus
    ) = withContext(writeDispatcher.value) {
        conversationQueries.transactionWithResult {
            conversationQueries.updateLegalHoldStatus(legalHoldStatus, conversationId)
            conversationQueries.selectChanges().executeAsOne() > 0
        }

    }

    override suspend fun updateLegalHoldStatusChangeNotified(conversationId: QualifiedIDEntity, notified: Boolean) =
        withContext(writeDispatcher.value) {
            conversationQueries.transactionWithResult {
                conversationQueries.upsertLegalHoldStatusChangeNotified(conversationId, notified)
                conversationQueries.selectChanges().executeAsOne() > 0
            }
        }

    override suspend fun observeLegalHoldStatus(conversationId: QualifiedIDEntity) =
        conversationQueries.selectLegalHoldStatus(conversationId)
            .asFlow()
            .mapToOneOrDefault(ConversationEntity.LegalHoldStatus.DISABLED)
            .flowOn(readDispatcher.value)

    override suspend fun observeLegalHoldStatusChangeNotified(conversationId: QualifiedIDEntity) =
        conversationQueries.selectLegalHoldStatusChangeNotified(conversationId)
            .asFlow()
            .mapToOneOrDefault(true)
            .flowOn(readDispatcher.value)

    override suspend fun getEstablishedSelfMLSGroupId(): String? =
        withContext(readDispatcher.value) {
            conversationQueries
                .getEstablishedSelfMLSGroupId()
                .executeAsOneOrNull()
                ?.mls_group_id
        }

    override suspend fun selectGroupStatusMembersNamesAndHandles(groupID: String): EpochChangesDataEntity? =
        withContext(readDispatcher.value) {
            conversationQueries.transactionWithResult {
                val (conversationId, mlsVerificationStatus) = conversationQueries.conversationIDByGroupId(groupID).executeAsOneOrNull()
                    ?: return@transactionWithResult null
                memberQueries.selectMembersNamesAndHandle(conversationId).executeAsList()
                    .let { members ->
                        val membersMap = members.associate { it.user to NameAndHandleEntity(it.name, it.handle) }
                        EpochChangesDataEntity(
                            conversationId,
                            mlsVerificationStatus,
                            membersMap
                        )
                    }
            }
        }

    override suspend fun isAChannel(conversationId: QualifiedIDEntity): Boolean = withContext(readDispatcher.value) {
        conversationQueries.selectIsChannel(conversationId).executeAsOneOrNull() ?: false
    }

    override suspend fun updateChannelAddPermission(
        conversationId: QualifiedIDEntity,
        channelAddPermission: ConversationEntity.ChannelAddPermission
    ) = withContext(writeDispatcher.value) {
        conversationQueries.updateChannelAddPermission(channelAddPermission, conversationId)
    }

    override suspend fun hasConversationWithCell() = withContext(readDispatcher.value) {
        conversationQueries.hasConversationWithCell().executeAsOne()
    }
}
