/*
 * Wire
 * Copyright (C) 2024 Wire Swiss GmbH
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see http://www.gnu.org/licenses/.
 */
package com.wire.kalium.logic.util.arrangement.repository

import com.wire.kalium.common.error.CoreFailure
import com.wire.kalium.common.functional.Either
import com.wire.kalium.logic.data.conversation.EpochChangesObserver
import com.wire.kalium.logic.data.conversation.GroupWithEpoch
import com.wire.kalium.logic.data.conversation.MLSConversationRepository
import com.wire.kalium.logic.data.id.GroupID
import io.mockative.any
import io.mockative.coEvery
import io.mockative.eq
import io.mockative.every
import io.mockative.mock
import kotlinx.coroutines.flow.Flow

internal interface MLSConversationRepositoryArrangement {
    val epochChangesObserver: EpochChangesObserver
    val mlsConversationRepository: MLSConversationRepository

    fun withObserveEpochChanges(result: Flow<GroupWithEpoch>)
    suspend fun withSuccessfulLeaveGroup(groupId: GroupID = GroupID("mls_group_id"))
    suspend fun withFailedLeaveGroup(groupId: GroupID = GroupID("mls_group_id"))
}

internal open class MLSConversationRepositoryArrangementImpl :
    MLSConversationRepositoryArrangement {

    override val epochChangesObserver: EpochChangesObserver = mock(EpochChangesObserver::class)
    override val mlsConversationRepository: MLSConversationRepository = mock(MLSConversationRepository::class)


    override fun withObserveEpochChanges(result: Flow<GroupWithEpoch>) {
        every {
            epochChangesObserver.observe()
        }.returns(result)
    }

    override suspend fun withSuccessfulLeaveGroup(groupId: GroupID) {
        coEvery { mlsConversationRepository.leaveGroup(any(), eq(groupId)) }.returns(Either.Right(Unit))
    }

    override suspend fun withFailedLeaveGroup(groupId: GroupID) {
        coEvery { mlsConversationRepository.leaveGroup(any(), eq(groupId)) }.returns(Either.Left(CoreFailure.Unknown(null)))
    }

}
