/*
 * Wire
 * Copyright (C) 2024 Wire Swiss GmbH
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see http://www.gnu.org/licenses/.
 */
package com.wire.kalium.logic.sync.receiver.conversation.message

import com.wire.kalium.common.error.CoreFailure
import com.wire.kalium.common.error.E2EIFailure
import com.wire.kalium.common.error.MLSFailure
import com.wire.kalium.common.error.NetworkFailure
import com.wire.kalium.common.error.ProteusFailure
import com.wire.kalium.common.error.StorageFailure
import com.wire.kalium.common.error.wrapNetworkMlsFailureIfApplicable

sealed class MLSMessageFailureResolution {
    data object Ignore : MLSMessageFailureResolution()
    data object InformUser : MLSMessageFailureResolution()
    data object OutOfSync : MLSMessageFailureResolution()
    data object ResetConversation : MLSMessageFailureResolution()
}

internal object MLSMessageFailureHandler {
    fun handleFailure(failure: CoreFailure): MLSMessageFailureResolution {
        fun handleRejected(rejectedFailure: NetworkFailure.MlsMessageRejectedFailure) = when (rejectedFailure) {
            is NetworkFailure.MlsMessageRejectedFailure.GroupOutOfSync,
            NetworkFailure.MlsMessageRejectedFailure.InvalidLeafNodeIndex,
            NetworkFailure.MlsMessageRejectedFailure.InvalidLeafNodeSignature -> MLSMessageFailureResolution.ResetConversation

            NetworkFailure.MlsMessageRejectedFailure.ClientMismatch,
            NetworkFailure.MlsMessageRejectedFailure.CommitMissingReferences,
            NetworkFailure.MlsMessageRejectedFailure.MissingGroupInfo,
            is NetworkFailure.MlsMessageRejectedFailure.Other,
            NetworkFailure.MlsMessageRejectedFailure.StaleMessage -> MLSMessageFailureResolution.InformUser
        }
        return when (val failure = failure.wrapNetworkMlsFailureIfApplicable()) {
            // Received messages targeting a future epoch (outside epoch bounds), we might have lost messages.
            is MLSFailure.WrongEpoch,
            is MLSFailure.InvalidGroupId -> MLSMessageFailureResolution.OutOfSync

            is NetworkFailure.MlsMessageRejectedFailure -> handleRejected(failure)
            is MLSFailure.MessageRejected -> handleRejected(failure.cause)

            // Received already sent or received message, can safely be ignored.
            is MLSFailure.DuplicateMessage,
                // Received message was targeting a future epoch and been buffered, can safely be ignored.
            is MLSFailure.BufferedFutureMessage,
                // Received self commit, any unmerged group has know been when merged by CoreCrypto.
            is MLSFailure.SelfCommitIgnored,
                // Message arrive in an unmerged group, it has been buffered and will be consumed later.
            is MLSFailure.UnmergedPendingGroup,
            is MLSFailure.StaleProposal,
            is MLSFailure.StaleCommit,
            is MLSFailure.MessageEpochTooOld,
            is MLSFailure.InternalErrors,
            is MLSFailure.Disabled,
            MLSFailure.CommitForMissingProposal,
            MLSFailure.ConversationNotFound,
            MLSFailure.BufferedCommit,
            MLSFailure.OrphanWelcome,
            is CoreFailure.DevelopmentAPINotAllowedOnProduction -> MLSMessageFailureResolution.Ignore

            MLSFailure.ConversationAlreadyExists,
            MLSFailure.ConversationDoesNotSupportMLS,
            is MLSFailure.Generic,
            is MLSFailure.Other,
            is E2EIFailure,
            is CoreFailure.FeatureFailure,
            CoreFailure.MissingClientRegistration,
            is CoreFailure.MissingKeyPackages,
            NetworkFailure.FeatureNotSupported,
            is NetworkFailure.FederatedBackendFailure.ConflictingBackends,
            is NetworkFailure.FederatedBackendFailure.FailedDomains,
            is NetworkFailure.FederatedBackendFailure.FederationDenied,
            is NetworkFailure.FederatedBackendFailure.FederationNotEnabled,
            is NetworkFailure.FederatedBackendFailure.General,
            is NetworkFailure.NoNetworkConnection,
            is NetworkFailure.ProxyError,
            is NetworkFailure.ServerMiscommunication,
            is ProteusFailure,
            StorageFailure.DataNotFound,
            is StorageFailure.Generic,
            is MLSFailure.InvalidGroupId,
            is CoreFailure.Unknown -> MLSMessageFailureResolution.InformUser
        }
    }
}
