/*
 * Wire
 * Copyright (C) 2024 Wire Swiss GmbH
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see http://www.gnu.org/licenses/.
 */
package com.wire.android.ui.edit

import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.defaultMinSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.shape.CornerSize
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.Button
import androidx.compose.material3.ButtonDefaults
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.LocalContentColor
import androidx.compose.material3.LocalMinimumInteractiveComponentSize
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.TextUnit
import androidx.compose.ui.unit.sp
import com.wire.android.R
import com.wire.android.ui.common.bottomsheet.rememberWireModalSheetState
import com.wire.android.ui.common.dimensions
import com.wire.android.ui.common.spacers.VerticalSpace
import com.wire.android.ui.emoji.EmojiPickerBottomSheet
import com.wire.android.ui.theme.WireTheme
import com.wire.android.ui.theme.wireColorScheme
import com.wire.android.ui.theme.wireTypography
import com.wire.android.util.ui.PreviewMultipleThemes

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun ReactionOption(
    ownReactions: Set<String>,
    onReactionClick: (emoji: String) -> Unit,
    modifier: Modifier = Modifier,
    emojiFontSize: TextUnit = 28.sp
) {
    val emojiPickerState = rememberWireModalSheetState<Unit>(skipPartiallyExpanded = false)
    CompositionLocalProvider(LocalContentColor provides MaterialTheme.colorScheme.onSurface) {
        Column(modifier = modifier) {
            VerticalSpace.x8()
            Row {
                Spacer(modifier = Modifier.width(dimensions().spacing8x))
                Text(
                    stringResource(R.string.label_reactions).uppercase(),
                    style = MaterialTheme.wireTypography.label01
                )
            }
            Row(
                modifier = Modifier.fillMaxWidth(),
                verticalAlignment = Alignment.CenterVertically,
                horizontalArrangement = Arrangement.SpaceEvenly
            ) {
                ownReactions.plus(listOf("👍", "🙂", "❤️", "☹️", "👎")).take(5).forEach { emoji ->
                    CompositionLocalProvider(
                        LocalMinimumInteractiveComponentSize provides Dp.Unspecified,
                    ) {
                        val containerColor = if (ownReactions.contains(emoji)) {
                            MaterialTheme.wireColorScheme.primaryVariant
                        } else {
                            MaterialTheme.wireColorScheme.surface
                        }

                        Button(
                            onClick = {
                                onReactionClick(emoji)
                            },
                            shape = RoundedCornerShape(corner = CornerSize(dimensions().spacing12x)),
                            modifier = Modifier.defaultMinSize(minWidth = dimensions().spacing1x, minHeight = dimensions().spacing1x),
                            contentPadding = PaddingValues(dimensions().spacing6x),
                            colors = ButtonDefaults.buttonColors(
                                containerColor = containerColor,
                                contentColor = MaterialTheme.wireColorScheme.secondaryButtonSelectedOutline
                            )
                        ) {
                            Text(emoji, style = TextStyle(fontSize = emojiFontSize))
                        }
                    }
                }
                IconButton(
                    onClick = {
                        emojiPickerState.show(Unit)
                    },
                ) {
                    Icon(
                        painter = painterResource(id = R.drawable.ic_more_emojis),
                        contentDescription = stringResource(R.string.content_description_more_emojis)
                    )
                }
            }
            VerticalSpace.x8()
        }
    }
    EmojiPickerBottomSheet(
        sheetState = emojiPickerState,
        onEmojiSelected = { emoji, _ ->
            emojiPickerState.hide()
            onReactionClick(emoji)
        }
    )
}

@PreviewMultipleThemes
@Composable
private fun PreviewReactionOption() = WireTheme {
    Box(modifier = Modifier.background(MaterialTheme.wireColorScheme.surface)) {
        ReactionOption(
            setOf("😆"),
            onReactionClick = {}
        )
    }
}
