/*
 * Wire
 * Copyright (C) 2024 Wire Swiss GmbH
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see http://www.gnu.org/licenses/.
 */

package com.wire.kalium.logic.util

import com.wire.kalium.logic.data.user.UserId
import com.wire.kalium.persistence.dbPassphrase.PassphraseStorage
import io.ktor.util.decodeBase64Bytes
import io.ktor.util.encodeBase64
import io.mockative.any
import io.mockative.doesNothing
import io.mockative.eq
import io.mockative.every
import io.mockative.mock
import io.mockative.once
import io.mockative.twice
import io.mockative.verify
import kotlinx.coroutines.test.runTest
import kotlin.test.BeforeTest
import kotlin.test.Test
import kotlin.test.assertEquals
import kotlin.test.assertTrue

class SecurityHelperTest {

        private val passphraseStorage: PassphraseStorage = mock(PassphraseStorage::class)

    private lateinit var securityHelper: SecurityHelper

    private val userId = UserId("df8703fb-bbab-4b10-a369-0ef781a17cf5", "wire.com")
    private val mlsV1Alias = "mls_db_secret_alias_$userId"
    private val mlsV2Alias = "mls_db_secret_alias_v2_$userId"
    private val proteusV1Alias = "proteus_db_secret_alias_$userId"
    private val proteusV2Alias = "proteus_db_secret_alias_v2_$userId"
    private val rootPath = "/root/path"

    @BeforeTest
    fun setup() {
        // Use a no-op migrator by default for tests to avoid file system dependencies
        val noOpMigrator: DatabaseMigrator = { _, _, _ -> }
        securityHelper = SecurityHelperImpl(passphraseStorage, noOpMigrator)
    }

    @Test
    fun whenCallingGlobalDBSecretTwice_thenTheSameValueIsReturned() = runTest {
        every {
            passphraseStorage.getPassphrase(any())
        }.returns(null)
        every { passphraseStorage.setPassphrase(any(), any()) }.doesNothing()
        val secret1 = securityHelper.globalDBSecret()
        every {
            passphraseStorage.getPassphrase(any())
        }.returns(secret1.value.encodeBase64())

        val secret2 = securityHelper.globalDBSecret()
        assertTrue(secret1.value.contentEquals(secret2.value))

        verify {
            passphraseStorage.getPassphrase(any())
        }.wasInvoked(exactly = twice)
        verify {
            passphraseStorage.setPassphrase(any(), any())
        }.wasInvoked(exactly = once)
    }

    @Test
    fun whenCallingUserDBSecretTwice_thenTheSameValueIsReturned() = runTest {
        every {
            passphraseStorage.getPassphrase(any())
        }.returns(null)
        every { passphraseStorage.setPassphrase(any(), any()) }.doesNothing()
        val userId = UserId("df8703fb-bbab-4b10-a369-0ef781a17cf5", "wire.com")
        val secret1 = securityHelper.userDBSecret(userId)
        every {
            passphraseStorage.getPassphrase(any())
        }.returns(secret1.value.encodeBase64())

        val secret2 = securityHelper.userDBSecret(userId)
        assertTrue(secret1.value.contentEquals(secret2.value))
        verify {
            passphraseStorage.getPassphrase(any())
        }.wasInvoked(exactly = twice)
        verify {
            passphraseStorage.setPassphrase(any(), any())
        }.wasInvoked(exactly = once)
    }

    @Test
    fun whenCallingMlsDBSecretTwiceForTheSameUser_thenTheSameValueIsReturned() = runTest {
        val userId = UserId("df8703fb-bbab-4b10-a369-0ef781a17cf5", "wire.com")

        val oldKeyBytes = ByteArray(32) { 0 }
        every { passphraseStorage.setPassphrase(any(), any()) }.doesNothing()
        setupSequencedGetPassphrase(mlsV2Alias, listOf(oldKeyBytes.encodeBase64(), oldKeyBytes.encodeBase64()))
        setupSequencedGetPassphrase(mlsV1Alias, listOf(null, "oldKeyBase64"))

        val secret1 = securityHelper.mlsDBSecret(userId, rootPath)
        val secret2 = securityHelper.mlsDBSecret(userId, rootPath)

        assertTrue(secret1.passphrase.contentEquals(secret2.passphrase))

        verify { passphraseStorage.setPassphrase(any(), any()) }.wasNotInvoked()
    }

    @Test
    fun givenV2Exists_whenCallingMlsDBSecret_thenReturnsV2WithoutMigration() = runTest {
        val v2b64 = "newBase64"

        setupSequencedGetPassphrase(mlsV2Alias, listOf(v2b64))
        every { passphraseStorage.setPassphrase(any(), any()) }.doesNothing()

        val secret = securityHelper.mlsDBSecret(userId, rootPath)

        assertTrue(secret.passphrase.contentEquals(v2b64.decodeBase64Bytes()))
        verify { passphraseStorage.setPassphrase(any(), any()) }.wasNotInvoked()
    }

    @Test
    fun givenOnlyV1Exists_whenCallingMlsDBSecret_thenMigratesToV2() = runTest {
        val v1b64 = "oldBase64"
        val captured = mutableListOf<String>()
        
        val tempDir = "/tmp/test"
        setupSequencedGetPassphrase(mlsV2Alias, listOf(null))
        setupSequencedGetPassphrase(mlsV1Alias, listOf(v1b64))
        setupSequencedSetPassphrase(mlsV2Alias, captured)

        val secret = securityHelper.mlsDBSecret(userId, tempDir)
        
        assertEquals(32, secret.passphrase.size)
        assertEquals(1, captured.size)
        assertTrue(captured[0].isNotEmpty())
    }

    private fun setupSequencedSetPassphrase(key: String, capturedValues: MutableList<String>) = apply {
        every { passphraseStorage.setPassphrase(any(), any()) }
            .invokes { args ->
                val receivedKey = args[0] as String
                val receivedValue = args[1] as String
                if (receivedKey == key) {
                    capturedValues.add(receivedValue)
                }
            }
    }

    private fun setupSequencedGetPassphrase(key: String, returnList: List<String?>) = apply {
        var invocationCounter = 0
        every { passphraseStorage.getPassphrase(key) }
            .invokes { _ ->
                if (invocationCounter < returnList.size) {
                    val returnValue = returnList[invocationCounter]
                    invocationCounter++
                    returnValue
                } else {
                    null
                }
            }
    }

    @Test
    fun givenNeitherV1NorV2Exist_whenCallingMlsDBSecret_thenGeneratesNewV2Secret() = runTest {
        setupSequencedGetPassphrase(mlsV2Alias, listOf(null, null))
        setupSequencedGetPassphrase(mlsV1Alias, listOf(null))
        every { passphraseStorage.setPassphrase(any(), any()) }.doesNothing()

        val secret = securityHelper.mlsDBSecret(userId, rootPath)

        assertEquals(32, secret.passphrase.size)
        verify { passphraseStorage.setPassphrase(eq(mlsV2Alias), any()) }.wasInvoked(exactly = once)
    }

    // PROTEUS DB SECRET TESTS

    @Test
    fun givenV2Exists_whenCallingProteusDBSecret_thenReturnsV2WithoutMigration() = runTest {
        val v2b64 = "proteusV2Secret"

        setupSequencedGetPassphrase(proteusV2Alias, listOf(v2b64))
        every { passphraseStorage.setPassphrase(any(), any()) }.doesNothing()

        val secret = securityHelper.proteusDBSecret(userId, rootPath)

        assertTrue(secret.passphrase.contentEquals(v2b64.decodeBase64Bytes()))
        verify { passphraseStorage.setPassphrase(any(), any()) }.wasNotInvoked()
    }

    @Test
    fun givenOnlyV1Exists_whenCallingProteusDBSecret_thenMigratesToV2() = runTest {
        val v1b64 = "proteusV1Secret"
        val captured = mutableListOf<String>()
        
        val tempDir = "/tmp/test"
        setupSequencedGetPassphrase(proteusV2Alias, listOf(null))
        setupSequencedGetPassphrase(proteusV1Alias, listOf(v1b64))
        setupSequencedSetPassphrase(proteusV2Alias, captured)

        val secret = securityHelper.proteusDBSecret(userId, tempDir)
        
        assertEquals(32, secret.passphrase.size)
        assertEquals(1, captured.size)
        assertTrue(captured[0].isNotEmpty())
    }

    @Test
    fun givenNeitherV1NorV2Exist_whenCallingProteusDBSecret_thenGeneratesNewV2Secret() = runTest {
        setupSequencedGetPassphrase(proteusV2Alias, listOf(null, null))
        setupSequencedGetPassphrase(proteusV1Alias, listOf(null))
        every { passphraseStorage.setPassphrase(any(), any()) }.doesNothing()

        val secret = securityHelper.proteusDBSecret(userId, rootPath)

        assertEquals(32, secret.passphrase.size)
        verify { passphraseStorage.setPassphrase(eq(proteusV2Alias), any()) }.wasInvoked(exactly = once)
    }

    @Test
    fun whenCallingProteusDBSecretTwiceForSameUser_thenReturnsSameValue() = runTest {
        val secretB64 = "proteusSecretValue"
        setupSequencedGetPassphrase(proteusV2Alias, listOf(secretB64, secretB64))
        every { passphraseStorage.setPassphrase(any(), any()) }.doesNothing()

        val secret1 = securityHelper.proteusDBSecret(userId, rootPath)
        val secret2 = securityHelper.proteusDBSecret(userId, rootPath)

        assertTrue(secret1.passphrase.contentEquals(secret2.passphrase))
    }

    @Test
    fun givenV1ExistsForMls_whenCallingMlsDBSecret_thenGeneratesNewKeyAndStoresV2() = runTest {
        val v1Secret = "oldMlsSecret"
        val capturedV2Keys = mutableListOf<String>()
        
        val tempDir = "/tmp/test"
        setupSequencedGetPassphrase(mlsV2Alias, listOf(null))
        setupSequencedGetPassphrase(mlsV1Alias, listOf(v1Secret))
        setupSequencedSetPassphrase(mlsV2Alias, capturedV2Keys)

        val secret = securityHelper.mlsDBSecret(userId, tempDir)
        
        // Verify new key was generated (32 bytes)
        assertEquals(32, secret.passphrase.size)
        // Verify v2 key was stored
        assertEquals(1, capturedV2Keys.size)
        // Verify stored key is base64 encoded
        assertTrue(capturedV2Keys[0].isNotEmpty())
        // Verify the returned secret matches what was stored
        assertTrue(secret.passphrase.contentEquals(capturedV2Keys[0].decodeBase64Bytes()))
    }

    @Test
    fun givenV1ExistsForProteus_whenCallingProteusDBSecret_thenGeneratesNewKeyAndStoresV2() = runTest {
        val v1Secret = "oldProteusSecret"
        val capturedV2Keys = mutableListOf<String>()
        
        val tempDir = "/tmp/test"
        setupSequencedGetPassphrase(proteusV2Alias, listOf(null))
        setupSequencedGetPassphrase(proteusV1Alias, listOf(v1Secret))
        setupSequencedSetPassphrase(proteusV2Alias, capturedV2Keys)

        val secret = securityHelper.proteusDBSecret(userId, tempDir)
        
        // Verify new key was generated (32 bytes)
        assertEquals(32, secret.passphrase.size)
        // Verify v2 key was stored
        assertEquals(1, capturedV2Keys.size)
        // Verify stored key is base64 encoded
        assertTrue(capturedV2Keys[0].isNotEmpty())
        // Verify the returned secret matches what was stored
        assertTrue(secret.passphrase.contentEquals(capturedV2Keys[0].decodeBase64Bytes()))
    }
}
