/*
 * Wire
 * Copyright (C) 2024 Wire Swiss GmbH
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see http://www.gnu.org/licenses/.
 */

package com.wire.kalium.logic.data.message.receipt

import com.wire.kalium.logic.data.id.ConversationId
import com.wire.kalium.logic.data.id.IdMapper
import com.wire.kalium.logic.data.id.toDao
import com.wire.kalium.logic.data.user.UserId
import com.wire.kalium.logic.di.MapperProvider
import com.wire.kalium.persistence.dao.receipt.ReceiptDAO
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.map
import kotlinx.datetime.Instant

interface ReceiptRepository {
    suspend fun persistReceipts(
        userId: UserId,
        conversationId: ConversationId,
        date: Instant,
        type: ReceiptType,
        messageIds: List<String>
    )

    suspend fun observeMessageReceipts(
        conversationId: ConversationId,
        messageId: String,
        type: ReceiptType
    ): Flow<List<DetailedReceipt>>
}

class ReceiptRepositoryImpl(
    private val receiptDAO: ReceiptDAO,
    private val receiptsMapper: ReceiptsMapper = MapperProvider.receiptsMapper(),
    private val idMapper: IdMapper = MapperProvider.idMapper()
) : ReceiptRepository {

    override suspend fun persistReceipts(
        userId: UserId,
        conversationId: ConversationId,
        date: Instant,
        type: ReceiptType,
        messageIds: List<String>
    ) {
        receiptDAO.insertReceipts(
            userId = userId.toDao(),
            conversationId = conversationId.toDao(),
            date = date,
            type = receiptsMapper.toTypeEntity(type),
            messageIds = messageIds
        )
    }

    override suspend fun observeMessageReceipts(
        conversationId: ConversationId,
        messageId: String,
        type: ReceiptType
    ): Flow<List<DetailedReceipt>> =
        receiptDAO.observeDetailedReceiptsForMessage(
            conversationId = conversationId.toDao(),
            messageId = messageId,
            type = receiptsMapper.toTypeEntity(type = type)
        ).map {
            it.map { detailedReceipt ->
                receiptsMapper.fromEntityToModel(
                    detailedReceiptEntity = detailedReceipt
                )
            }
        }
}
