/*
 * Wire
 * Copyright (C) 2024 Wire Swiss GmbH
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see http://www.gnu.org/licenses/.
 */
package com.wire.kalium.logic.data.conversation

import com.wire.kalium.cryptography.MLSEpochObserver
import com.wire.kalium.cryptography.MLSGroupId
import com.wire.kalium.logic.data.id.GroupID
import io.mockative.Mockable
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.MutableSharedFlow

@Mockable
interface EpochChangesObserver : MLSEpochObserver {
    fun observe(): Flow<GroupWithEpoch>
    suspend fun emit(groupId: MLSGroupId, epoch: ULong)
}

internal class EpochChangesObserverImpl : EpochChangesObserver {
    private val epochsFlow = MutableSharedFlow<GroupWithEpoch>()

    override fun observe(): Flow<GroupWithEpoch> = epochsFlow
    override suspend fun emit(groupId: MLSGroupId, epoch: ULong) {
        epochsFlow.emit(GroupWithEpoch(GroupID(groupId), epoch))
    }

    override suspend fun onEpochChange(groupId: MLSGroupId, epoch: ULong) {
        epochsFlow.emit(GroupWithEpoch(GroupID(groupId), epoch))
    }
}

data class GroupWithEpoch(val groupId: GroupID, val epoch: ULong)
