/*
 * Wire
 * Copyright (C) 2025 Wire Swiss GmbH
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see http://www.gnu.org/licenses/.
 */
package com.wire.android.feature.cells.ui.tags

import android.widget.Toast
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.ExperimentalLayoutApi
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.LazyRow
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.text.input.TextFieldState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.hilt.navigation.compose.hiltViewModel
import com.wire.android.feature.cells.R
import com.wire.android.model.ClickBlockParams
import com.wire.android.navigation.WireNavigator
import com.wire.android.navigation.annotation.features.cells.WireDestination
import com.wire.android.navigation.style.PopUpNavigationAnimation
import com.wire.android.ui.common.HandleActions
import com.wire.android.ui.common.button.WireButtonState
import com.wire.android.ui.common.button.WirePrimaryButton
import com.wire.android.ui.common.chip.ChipData
import com.wire.android.ui.common.chip.WireFilterChip
import com.wire.android.ui.common.colorsScheme
import com.wire.android.ui.common.dimensions
import com.wire.android.ui.common.preview.MultipleThemePreviews
import com.wire.android.ui.common.scaffold.WireScaffold
import com.wire.android.ui.common.topappbar.NavigationIconType
import com.wire.android.ui.common.topappbar.WireCenterAlignedTopAppBar
import com.wire.android.ui.theme.WireTheme
import com.wire.android.ui.theme.wireColorScheme
import com.wire.android.ui.theme.wireDimensions
import com.wire.android.ui.theme.wireTypography
import kotlinx.coroutines.launch

@WireDestination(
    navArgsDelegate = AddRemoveTagsNavArgs::class,
    style = PopUpNavigationAnimation::class,
)
@Composable
fun AddRemoveTagsScreen(
    navigator: WireNavigator,
    modifier: Modifier = Modifier,
    addRemoveTagsViewModel: AddRemoveTagsViewModel = hiltViewModel(),
) {
    val context = LocalContext.current

    WireScaffold(
        modifier = modifier,
        snackbarHost = {},
        topBar = {
            WireCenterAlignedTopAppBar(
                title = stringResource(R.string.add_or_remove_tags_title),
                navigationIconType = NavigationIconType.Close(),
                onNavigationPressed = {
                    navigator.navigateBack()
                },
            )
        },
        bottomBar = {
            val isLoading = addRemoveTagsViewModel.isLoading.collectAsState().value
            val tags = addRemoveTagsViewModel.suggestedTags.collectAsState()
            Column(
                modifier = Modifier.background(colorsScheme().background)
            ) {
                if (tags.value.isNotEmpty()) {
                    LazyRow(
                        modifier = Modifier
                            .fillMaxWidth()
                            .padding(start = dimensions().spacing16x, end = dimensions().spacing16x)
                    ) {
                        tags.value.forEach { tag ->
                            item {
                                WireFilterChip(
                                    modifier = Modifier.padding(
                                        end = dimensions().spacing8x,
                                        bottom = dimensions().spacing8x
                                    ),
                                    label = tag,
                                    isSelected = false,
                                    onSelectChip = { addRemoveTagsViewModel.addTag(tag) }
                                )
                            }
                        }
                    }
                }

                Surface(
                    color = MaterialTheme.wireColorScheme.background,
                    shadowElevation = MaterialTheme.wireDimensions.bottomNavigationShadowElevation
                ) {
                    Row(
                        horizontalArrangement = Arrangement.Center,
                        verticalAlignment = Alignment.CenterVertically,
                        modifier = Modifier
                            .padding(horizontal = dimensions().spacing16x)
                            .height(dimensions().groupButtonHeight)
                    ) {
                        val shouldDisabledSaveButton =
                            isLoading || addRemoveTagsViewModel.initialTags == addRemoveTagsViewModel.addedTags.collectAsState().value
                        WirePrimaryButton(
                            text = stringResource(R.string.save_label),
                            onClick = {
                                addRemoveTagsViewModel.updateTags()
                            },
                            state = if (shouldDisabledSaveButton) WireButtonState.Disabled else WireButtonState.Default,
                            loading = isLoading,
                            clickBlockParams = ClickBlockParams(blockWhenSyncing = true, blockWhenConnecting = true),
                        )
                    }
                }
            }
        }
    ) { internalPadding ->

        AddRemoveTagsScreenContent(
            internalPadding = internalPadding,
            textFieldState = addRemoveTagsViewModel.tagsTextState,
            addedTags = addRemoveTagsViewModel.addedTags.collectAsState().value,
            onAddTag = { tag ->
                addRemoveTagsViewModel.addTag(tag)
            },
            onRemoveTag = { tag ->
                addRemoveTagsViewModel.removeTag(tag)
            },
            onRemoveLastTag = {
                addRemoveTagsViewModel.removeLastTag()
            },
            isValidTag = { addRemoveTagsViewModel.isValidTag() },
            modifier = Modifier.fillMaxSize()
        )
    }

    HandleActions(addRemoveTagsViewModel.actions) { action ->
        when (action) {
            is AddRemoveTagsViewModelAction.Success -> {
                Toast.makeText(context, context.resources.getString(R.string.tags_edited), Toast.LENGTH_SHORT).show()
            }

            is AddRemoveTagsViewModelAction.Failure -> {
                Toast.makeText(context, context.resources.getString(R.string.failed_edit_tags), Toast.LENGTH_SHORT).show()
            }
        }
        navigator.navigateBack()
    }
}

@OptIn(ExperimentalLayoutApi::class)
@Composable
fun AddRemoveTagsScreenContent(
    internalPadding: PaddingValues,
    textFieldState: TextFieldState,
    addedTags: Set<String>,
    isValidTag: () -> Boolean,
    onAddTag: (String) -> Unit,
    onRemoveTag: (String) -> Unit,
    onRemoveLastTag: () -> Unit,
    modifier: Modifier = Modifier
) {

    val scrollState = rememberScrollState()
    val coroutineScope = rememberCoroutineScope()

    // Auto-scroll to bottom when item is added
    LaunchedEffect(addedTags.size) {
        coroutineScope.launch {
            scrollState.animateScrollTo(scrollState.maxValue)
        }
    }

    Column(
        modifier = modifier
            .verticalScroll(scrollState)
            .padding(internalPadding)
    ) {

        Text(
            modifier = Modifier.padding(dimensions().spacing16x),
            text = stringResource(R.string.tags_description),
            style = MaterialTheme.wireTypography.body01,
        )

        ChipAndTextFieldLayout(
            textFieldState = textFieldState,
            tags = addedTags.map { ChipData(label = it, isSelected = true, isEnabled = false) }.toSet(),
            isValidTag = isValidTag,
            onDone = onAddTag,
            onRemoveLastTag = onRemoveLastTag,
        ) { data: ChipData ->

            WireFilterChip(
                label = data.label,
                isSelected = data.isSelected,
                onSelectChip = onRemoveTag
            )
        }
    }
}

@MultipleThemePreviews
@Preview
@Composable
fun PreviewAddRemoveTagsScreen() {
    WireTheme {
        AddRemoveTagsScreenContent(
            internalPadding = PaddingValues(0.dp),
            textFieldState = TextFieldState(),
            addedTags = setOf("Android", "Web", "iOS"),
            onAddTag = {},
            onRemoveTag = {},
            isValidTag = { true },
            onRemoveLastTag = {},
            modifier = Modifier.fillMaxSize()
        )
    }
}
