package com.wbpxre150.shoppinglist

import android.app.Activity
import android.content.Intent
import android.os.Bundle
import android.view.View
import android.view.WindowManager
import android.widget.Button
import android.widget.EditText
import android.widget.Toast
import androidx.activity.result.contract.ActivityResultContracts
import androidx.appcompat.app.AlertDialog
import androidx.appcompat.app.AppCompatActivity
import androidx.lifecycle.ViewModelProvider
import androidx.recyclerview.widget.ItemTouchHelper
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.wbpxre150.shoppinglist.databinding.ActivityMainBinding

class MainActivity : AppCompatActivity() {

    private lateinit var binding: ActivityMainBinding
    private lateinit var shoppingViewModel: ShoppingViewModel
    private lateinit var adapter: ShoppingListAdapter
    private lateinit var itemTouchHelper: ItemTouchHelper

    private val newShoppingListLauncher = registerForActivityResult(
        ActivityResultContracts.StartActivityForResult()
    ) { result ->
        if (result.resultCode == Activity.RESULT_OK) {
            result.data?.getStringExtra(NewShoppingListActivity.EXTRA_REPLY)?.let { listName ->
                val currentListCount = shoppingViewModel.allShoppingLists.value?.size ?: 0
                val shoppingList = ShoppingList(name = listName, position = currentListCount)
                shoppingViewModel.insert(shoppingList)
            }
        }
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = ActivityMainBinding.inflate(layoutInflater)
        setContentView(binding.root)

        adapter = ShoppingListAdapter(
            onListClicked = { shoppingList ->
                val intent = Intent(this, ShoppingListDetailActivity::class.java).apply {
                    putExtra(ShoppingListDetailActivity.EXTRA_LIST_ID, shoppingList.id)
                    putExtra(ShoppingListDetailActivity.EXTRA_LIST_NAME, shoppingList.name)
                }
                startActivity(intent)
            },
            onStartDrag = { viewHolder ->
                itemTouchHelper.startDrag(viewHolder)
            },
            onListDoubleTapped = { shoppingList ->
                showEditListDialog(shoppingList)
            }
        )

        binding.recyclerView.adapter = adapter
        binding.recyclerView.layoutManager = LinearLayoutManager(this)

        val callback = object : ItemTouchHelper.SimpleCallback(
            ItemTouchHelper.UP or ItemTouchHelper.DOWN, 0
        ) {
            override fun onMove(
                recyclerView: RecyclerView,
                viewHolder: RecyclerView.ViewHolder,
                target: RecyclerView.ViewHolder
            ): Boolean {
                val fromPosition = viewHolder.adapterPosition
                val toPosition = target.adapterPosition
                adapter.moveItem(fromPosition, toPosition)
                return true
            }

            override fun onSwiped(viewHolder: RecyclerView.ViewHolder, direction: Int) {
                // Not used
            }

            override fun onSelectedChanged(viewHolder: RecyclerView.ViewHolder?, actionState: Int) {
                super.onSelectedChanged(viewHolder, actionState)

                when (actionState) {
                    ItemTouchHelper.ACTION_STATE_DRAG -> {
                        // Store original background and add highlight border when drag starts
                        viewHolder?.itemView?.let { itemView ->
                            itemView.tag = itemView.background
                            itemView.setBackgroundResource(R.drawable.item_drag_highlight)
                        }
                    }
                }
            }

            override fun clearView(recyclerView: RecyclerView, viewHolder: RecyclerView.ViewHolder) {
                super.clearView(recyclerView, viewHolder)

                // Restore original background when drag ends
                val originalBackground = viewHolder.itemView.tag as? android.graphics.drawable.Drawable
                if (originalBackground != null) {
                    viewHolder.itemView.background = originalBackground
                    viewHolder.itemView.tag = null
                } else {
                    // Fallback: remove any background to let CardView show through
                    viewHolder.itemView.background = null
                }

                updateListPositions()
            }
        }

        itemTouchHelper = ItemTouchHelper(callback)
        itemTouchHelper.attachToRecyclerView(binding.recyclerView)

        // Get a new or existing ViewModel from the ViewModelProvider
        val factory = ShoppingViewModel.ShoppingViewModelFactory((application as ShoppingApplication).repository)
        shoppingViewModel = ViewModelProvider(this, factory)[ShoppingViewModel::class.java]

        // Observe consolidated list data to prevent race conditions
        shoppingViewModel.allShoppingListsWithCounts.observe(this) { listsWithCounts ->
            if (listsWithCounts.isEmpty()) {
                binding.emptyView.visibility = View.VISIBLE
                binding.recyclerView.visibility = View.GONE
            } else {
                binding.emptyView.visibility = View.GONE
                binding.recyclerView.visibility = View.VISIBLE
            }

            adapter.submitList(listsWithCounts)
        }

        binding.fab.setOnClickListener {
            val intent = Intent(this, NewShoppingListActivity::class.java)
            newShoppingListLauncher.launch(intent)
        }
    }

    private fun updateListPositions() {
        val currentItems = adapter.currentList
        val updatedLists = mutableListOf<ShoppingList>()

        for (i in currentItems.indices) {
            val item = currentItems[i]
            val updatedList = item.shoppingList.copy(position = i)
            updatedLists.add(updatedList)
        }

        if (updatedLists.isNotEmpty()) {
            shoppingViewModel.updateShoppingLists(updatedLists)
        }
    }

    private fun showEditListDialog(shoppingList: ShoppingList) {
        val dialogView = layoutInflater.inflate(R.layout.dialog_edit_list, null)
        val editTextListName = dialogView.findViewById<EditText>(R.id.editTextListName)
        val buttonCancel = dialogView.findViewById<Button>(R.id.buttonCancel)
        val buttonSave = dialogView.findViewById<Button>(R.id.buttonSave)

        // Set current value
        editTextListName.setText(shoppingList.name)

        val dialog = AlertDialog.Builder(this)
            .setView(dialogView)
            .create()

        // Set transparent background for MaterialCardView styling
        dialog.window?.setBackgroundDrawableResource(android.R.color.transparent)

        // Set up button listeners
        buttonCancel.setOnClickListener {
            dialog.dismiss()
        }

        buttonSave.setOnClickListener {
            val newName = editTextListName.text.toString().trim()

            if (newName.isNotEmpty()) {
                // Create updated list
                val updatedList = shoppingList.copy(name = newName)

                // Update in database
                shoppingViewModel.update(updatedList)
                Toast.makeText(this, "List renamed successfully", Toast.LENGTH_SHORT).show()
                dialog.dismiss()
            } else {
                Toast.makeText(this, "List name cannot be empty", Toast.LENGTH_SHORT).show()
            }
        }

        dialog.show()

        // Show keyboard automatically for dialog
        dialog.window?.setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_STATE_VISIBLE)
        editTextListName.requestFocus()
        editTextListName.selectAll()
    }

}
