/**
 * Copyright (C) Azureus Software, Inc, All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package com.vuze.android.remote;

import java.lang.reflect.Method;
import java.net.*;
import java.util.*;

import android.annotation.TargetApi;
import android.content.*;
import android.content.pm.PackageManager;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.net.wifi.WifiInfo;
import android.net.wifi.WifiManager;
import android.os.Build;
import android.util.Log;

public class NetworkState
{
	public final static String ETHERNET_SERVICE = "ethernet";

	static final String TAG = "NetworkState";

	public interface NetworkStateListener
	{
		void onlineStateChanged(boolean isOnline, boolean isOnlineMobile);
	}

	private Object oEthernetManager;

	private BroadcastReceiver mConnectivityReceiver;

	private boolean isOnline;

	/* @Thunk */ String onlineStateReason;

	private final Context context;

	private final List<NetworkStateListener> listeners = new ArrayList<>();

	public NetworkState(Context context) {
		this.context = context;

		try {
			//noinspection ResourceType ETHERNET_SERVICE is real! :)
			oEthernetManager = context.getSystemService(ETHERNET_SERVICE);
		} catch (Throwable ignore) {
		}

		// register BroadcastReceiver on network state changes
		mConnectivityReceiver = new BroadcastReceiver() {
			public void onReceive(Context context, Intent intent) {
				String action = intent.getAction();
				if (!action.equals(ConnectivityManager.CONNECTIVITY_ACTION)) {
					return;
				}
				boolean noConnectivity = intent.getBooleanExtra(
						ConnectivityManager.EXTRA_NO_CONNECTIVITY, false);
				setOnline(!noConnectivity, noConnectivity ? false : isOnlineMobile());
				onlineStateReason = intent.getStringExtra(
						ConnectivityManager.EXTRA_REASON);
				if (AndroidUtils.DEBUG) {
					@SuppressWarnings("deprecation")
					NetworkInfo networkInfo = intent.getParcelableExtra(
							ConnectivityManager.EXTRA_NETWORK_INFO);
					NetworkInfo otherNetworkInfo = intent.getParcelableExtra(
							ConnectivityManager.EXTRA_OTHER_NETWORK_INFO);
					boolean isFailover = intent.getBooleanExtra(
							ConnectivityManager.EXTRA_IS_FAILOVER, false);
					Log.d(TAG, "networkInfo=" + networkInfo);
					Log.d(TAG, "otherNetworkInfo=" + otherNetworkInfo);
					Log.d(TAG, "reason=" + onlineStateReason);
					Log.d(TAG, "isFailOver=" + isFailover);
					Log.d(TAG, "Active IP=" + getActiveIpAddress());
				}
			}
		};
		boolean online = isOnline(context);
		setOnline(online, !online ? false : isOnlineMobile());
		final IntentFilter mIFNetwork = new IntentFilter();
		mIFNetwork.addAction(ConnectivityManager.CONNECTIVITY_ACTION);
		context.registerReceiver(mConnectivityReceiver, mIFNetwork);
	}

	protected void setOnline(boolean online, boolean onlineMobile) {
		if (AndroidUtils.DEBUG) {
			Log.d(TAG, "setOnline " + online);
		}
		isOnline = online;
		synchronized (listeners) {
			for (NetworkStateListener l : listeners) {
				l.onlineStateChanged(online, onlineMobile);
			}
		}
	}

	public boolean isOnline() {
		return isOnline;
	}

	public void dipose() {
		if (mConnectivityReceiver != null) {
			context.unregisterReceiver(mConnectivityReceiver);
			mConnectivityReceiver = null;
		}
	}

	public void addListener(NetworkStateListener l) {
		synchronized (listeners) {
			if (!listeners.contains(l)) {
				listeners.add(l);
			}
		}
		l.onlineStateChanged(isOnline, isOnlineMobile());
	}

	public void removeListener(NetworkStateListener l) {
		synchronized (listeners) {
			listeners.remove(l);
		}
	}

	private static boolean isOnline(Context context) {
		ConnectivityManager cm = (ConnectivityManager) context.getSystemService(
				Context.CONNECTIVITY_SERVICE);
		if (cm == null) {
			return false;
		}
		NetworkInfo netInfo = cm.getActiveNetworkInfo();
		if (netInfo != null && netInfo.isConnected()) {
			return true;
		}
		if (netInfo == null) {
			if (AndroidUtils.DEBUG) {
				Log.d(TAG, "no active network");
			}
			netInfo = cm.getNetworkInfo(ConnectivityManager.TYPE_MOBILE);
			if (netInfo != null && netInfo.isConnected()) {
				if (AndroidUtils.DEBUG) {
					Log.d(TAG, "mobile network connected");
				}
				return true;
			}
		}
		return false;
	}

	public boolean hasMobileDataCapability() {
		if (context.getPackageManager().hasSystemFeature(
				PackageManager.FEATURE_TELEPHONY)) {
			return true;
		}

		// could have no phone ability, but still a data plan (somehow..)

		ConnectivityManager cm = (ConnectivityManager) context.getSystemService(
				Context.CONNECTIVITY_SERVICE);
		NetworkInfo ni = cm.getNetworkInfo(ConnectivityManager.TYPE_MOBILE);

		return ni != null;
	}

	public boolean isOnlineMobile() {
		ConnectivityManager cm = (ConnectivityManager) context.getSystemService(
				Context.CONNECTIVITY_SERVICE);
		if (cm == null) {
			return false;
		}
		NetworkInfo netInfo = cm.getActiveNetworkInfo();
		if (netInfo != null && netInfo.isConnected()) {
			int type = netInfo.getType();
			return type == ConnectivityManager.TYPE_MOBILE || type == 4 //ConnectivityManager.TYPE_MOBILE_DUN
					|| type == 5 //ConnectivityManager.TYPE_MOBILE_HIPRI
					|| type == 2 //ConnectivityManager.TYPE_MOBILE_MMS
					|| type == 3; //ConnectivityManager.TYPE_MOBILE_SUPL;
		}
		return false;
	}

	public String getActiveIpAddress() {
		String ipAddress = "127.0.0.1";

		ConnectivityManager cm = (ConnectivityManager) context.getSystemService(
				Context.CONNECTIVITY_SERVICE);
		if (cm == null) {
			return ipAddress;
		}
		NetworkInfo netInfo = cm.getActiveNetworkInfo();
		if (netInfo != null && netInfo.isConnected()) {

			if (AndroidUtils.DEBUG) {
				Log.e("IP address", "activeNetwork=" + netInfo.getTypeName() + "/"
						+ netInfo.getType() + "/" + netInfo.getExtraInfo());
			}

			int netType = netInfo.getType();
			if (netType == ConnectivityManager.TYPE_WIFI) {
				WifiManager wifiManager = (WifiManager) context.getSystemService(
						Context.WIFI_SERVICE);
				if (wifiManager != null) {
					WifiInfo wifiInfo = wifiManager.getConnectionInfo();
					if (wifiInfo != null) {
						int ipAddressInt = wifiInfo.getIpAddress();

						ipAddress = String.format(Locale.getDefault(), "%d.%d.%d.%d",
								(ipAddressInt & 0xff), (ipAddressInt >> 8 & 0xff),
								(ipAddressInt >> 16 & 0xff), (ipAddressInt >> 24 & 0xff));

						if (AndroidUtils.DEBUG) {
							Log.e("IP address", "activeNetwork Wifi=" + ipAddress);
						}
						return ipAddress;
					}
				}

				return getIpAddress("wlan");
			} else if (netType == ConnectivityManager.TYPE_ETHERNET) {
				if (AndroidUtils.DEBUG) {
					Log.e("IP address", "activeNetwork Ethernet");
				}
				if (oEthernetManager != null) {

					// Try ethernetManager.readConfiguration.getiFaceAddress, if present
					try {
						Method methEthernetConfiguration = oEthernetManager.getClass().getDeclaredMethod(
								"readConfiguration");
						Object oEthernetConfiguration = methEthernetConfiguration.invoke(
								oEthernetManager);
						Method methGetIpAddress = oEthernetConfiguration.getClass().getDeclaredMethod(
								"getIfaceAddress");
						Object oIPAddress = methGetIpAddress.invoke(oEthernetConfiguration);
						if (AndroidUtils.DEBUG) {
							Log.e("IP address", "" + oIPAddress);
						}
						if (oIPAddress instanceof InetAddress) {
							return ((InetAddress) oIPAddress).getHostAddress();
						}
					} catch (NoSuchMethodException ex) {

					} catch (Throwable e) {
						Log.e("IP address", e.getMessage(), e);
					}

					// Try ethernetManager.getSaveEthConfig.getIpAddress, if present
					// (never have seen this)
					try {
						Method methGetSavedEthConfig = oEthernetManager.getClass().getDeclaredMethod(
								"getSavedEthConfig");
						Object oEthernetDevInfo = methGetSavedEthConfig.invoke(
								oEthernetManager);
						Method methGetIpAddress = oEthernetDevInfo.getClass().getDeclaredMethod(
								"getIpAddress");
						Object oIPAddress = methGetIpAddress.invoke(oEthernetDevInfo);

						Log.e("IP address", "" + oIPAddress);
						if (oIPAddress instanceof String) {
							return (String) oIPAddress;
						}
					} catch (NoSuchMethodException ex) {
					} catch (Throwable e) {
						Log.e("IP address", e.getMessage(), e);
					}

					return getIpAddress("eth");
				}
			}

		}
		// best guess
		ipAddress = getIpAddress(null);
		return ipAddress;
	}

	/* Don't need Wifi check -- !isOnlineMobile is better because it includes LAN and Wifi 
	public boolean isWifiConnected() {
		ConnectivityManager connManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
		if (connManager == null) {
			return false;
		}
		NetworkInfo mWifi = connManager.getNetworkInfo(ConnectivityManager.TYPE_WIFI);
		if (mWifi == null) {
			return false;
		}
	
		return mWifi.isConnected();
	}
	*/

	/**
	 * Returns the IP that is "UP", preferring the one that "startsWith"
	 * Returns IP even if none "startsWith"
	 */
	public static String getIpAddress(String startsWith) {
		if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.GINGERBREAD) {
			return getIpAddress_9(startsWith);
		}
		return getIpAddress_Old(startsWith);
	}

	@TargetApi(Build.VERSION_CODES.GINGERBREAD)
	private static String getIpAddress_9(String startsWith) {
		String ipAddress = "127.0.0.1";
		try {
			for (Enumeration<NetworkInterface> en = NetworkInterface.getNetworkInterfaces(); en.hasMoreElements();) {
				NetworkInterface intf = en.nextElement();
				if (intf.getName().startsWith("usb") || !intf.isUp()) {
					// ignore usb and !up
					if (AndroidUtils.DEBUG) {
						if (startsWith == null || intf.getName().startsWith(startsWith)) {
							Log.e("IP address",
									"IGNORE: " + intf.getDisplayName() + "/" + intf.getName()
											+ "/PtoP=" + intf.isPointToPoint() + "/lb="
											+ intf.isLoopback() + "/up=" + intf.isUp() + "/virtual="
											+ intf.isVirtual());
						}
					}
					continue;
				}
				for (Enumeration<InetAddress> enumIpAddr = intf.getInetAddresses(); enumIpAddr.hasMoreElements();) {
					InetAddress inetAddress = enumIpAddr.nextElement();
					if (!inetAddress.isLoopbackAddress()
							&& (inetAddress instanceof Inet4Address)) {
						ipAddress = inetAddress.getHostAddress();

						if (AndroidUtils.DEBUG) {
							Log.e("IP address", intf.getDisplayName() + "/" + intf.getName()
									+ "/PtoP=" + intf.isPointToPoint() + "/lb="
									+ intf.isLoopback() + "/up=" + intf.isUp() + "/virtual="
									+ intf.isVirtual() + "/"
									+ (inetAddress.isSiteLocalAddress() ? "Local" : "NotLocal")
									+ "/" + ipAddress);
						}
						if (startsWith != null && intf.getName().startsWith(startsWith)) {
							return ipAddress;
						}
					}
				}
			}
		} catch (SocketException ex) {
			Log.e("IPAddress", ex.toString());
		}
		return ipAddress;
	}

	private static String getIpAddress_Old(String startsWith) {
		String ipAddress = "127.0.0.1";
		try {
			for (Enumeration<NetworkInterface> en = NetworkInterface.getNetworkInterfaces(); en.hasMoreElements();) {
				NetworkInterface intf = en.nextElement();
				if (intf.getName().startsWith("usb")) {
					if (AndroidUtils.DEBUG) {
						if (startsWith == null || intf.getName().startsWith(startsWith)) {
							Log.e("IP address", "IGNORE: " + intf.getDisplayName() + "/"
									+ intf.getName() + ";" + intf);
						}
					}
					continue;
				}
				for (Enumeration<InetAddress> enumIpAddr = intf.getInetAddresses(); enumIpAddr.hasMoreElements();) {
					InetAddress inetAddress = enumIpAddr.nextElement();
					if (!inetAddress.isLoopbackAddress()
							&& (inetAddress instanceof Inet4Address)) {
						ipAddress = inetAddress.getHostAddress();

						if (AndroidUtils.DEBUG) {
							Log.e("IP address", intf.getDisplayName() + "/" + intf.getName()
									+ "/"
									+ (inetAddress.isSiteLocalAddress() ? "Local" : "NotLocal")
									+ "/" + ipAddress);
						}
						if (startsWith != null && intf.getName().startsWith(startsWith)) {
							return ipAddress;
						}
					}
				}
			}
		} catch (SocketException ex) {
			Log.e("IPAddress", ex.toString());
		}
		return ipAddress;
	}

	public String getOnlineStateReason() {
		return onlineStateReason;
	}
}
