part of "../js_web_bluetooth.dart";

///
/// A class for calling methods and values for a [BluetoothDevice].
///
/// You can get a [WebBluetoothDevice] by calling [Bluetooth.requestDevice].
///
/// See:
///
/// - [Bluetooth]
///
/// - https://developer.mozilla.org/en-US/docs/Web/API/BluetoothDevice
///
/// - https://webbluetoothcg.github.io/web-bluetooth/#bluetoothdevice-interface
///
@JS()
extension type WebBluetoothDevice._(JSObject _)
    implements EventTarget, JSObject {
  @JS("id")
  external JSString get _id;

  ///
  /// Get the id of the device.
  ///
  /// This id is randomly generated by the browser. A new id is generated for
  /// each session unless a flag is used to remember previously connected
  /// devices.
  ///
  /// The browser may decide how this is generated. On Chrome this is 128
  /// randomly generated bits that are encoded in base 64.
  ///
  /// See:
  ///
  /// - https://developer.mozilla.org/en-US/docs/Web/API/BluetoothDevice/id
  ///
  /// - https://webbluetoothcg.github.io/web-bluetooth/#dom-bluetoothdevice-id
  ///
  String get id => _id.toDart;

  @JS("name")
  external JSString? get _name;

  ///
  /// A human readable name of the device.
  ///
  /// This name comes straight from the device itself.
  ///
  /// See:
  ///
  /// - https://developer.mozilla.org/en-US/docs/Web/API/BluetoothDevice/name
  ///
  /// - https://webbluetoothcg.github.io/web-bluetooth/#dom-bluetoothdevice-name
  ///
  String? get name {
    if (_name == null || _name.isUndefinedOrNull) {
      return null;
    }
    return _name!.toDart;
  }

  ///
  /// Get a reference to the [NativeBluetoothRemoteGATTServer] to communicate
  /// with the [WebBluetoothRemoteGATTService]s.
  ///
  /// Some devices may allow you to find them, but they are on a blocklist
  /// restricting the ability to communicate with its GATT service. In this
  /// case the returned value will be `null`.
  ///
  /// See:
  ///
  /// - https://developer.mozilla.org/en-US/docs/Web/API/BluetoothDevice/gatt
  ///
  /// - https://webbluetoothcg.github.io/web-bluetooth/#dom-bluetoothdevice-gatt
  ///
  @JS("gatt")
  external NativeBluetoothRemoteGATTServer? get gatt;

  ///
  /// Start watching for advertisements. The advertisements will be received
  /// on the `advertisementreceived` event. See [BluetoothAdvertisementReceivedEvent]
  /// for the object that is emitted every time the event fires.
  ///
  /// Not very browser has this implemented yet without the
  /// `enable-experimental-web-platform-features` flag enabled.
  /// So use [hasWatchAdvertisements] or it will throw an
  /// [NativeAPINotImplementedError].
  ///
  /// If you want to stop watching for advertisements then you will need to call
  /// this method again and with a [WatchAdvertisementsOptions.signal] that has
  /// already been aborted. If you do this then the method will throw a
  /// DOMException that you will need to handle.
  ///
  /// Example of the exception is: `DOMException: Failed to execute
  /// 'watchAdvertisements' on 'BluetoothDevice': The Bluetooth operation was cancelled.`
  ///
  Future<void> watchAdvertisements(
      [final WatchAdvertisementsOptions? options]) async {
    if (!hasWatchAdvertisements()) {
      throw NativeAPINotImplementedError("watchAdvertisements");
    }
    if (options == null) {
      await _watchAdvertisements().toDart;
    } else {
      await _watchAdvertisements(options).toDart;
    }
  }

  @JS("watchAdvertisements")
  external JSPromise _watchAdvertisements(
      [final WatchAdvertisementsOptions? options]);

  ///
  /// Check to see if the current browser has the watch advertisements method
  /// implemented
  ///
  bool hasWatchAdvertisements() => has("watchAdvertisements");

  ///
  /// Forget the device. This means that the device will be forgotten by the
  /// browser and no longer shows up in [Bluetooth.getDevices].
  ///
  /// Once forget has been called you won't be able to communicate with the
  /// device anymore.
  ///
  /// May throw:
  ///
  /// - [NativeAPINotImplementedError] if the user agent doesn't have the
  ///   forget method. Use [hasForget] to mitigate this.
  ///
  /// **NOTE:** Calling forget on a device retrieved via
  /// [Bluetooth.requestLEScan] will cause chrome to crash
  /// (last tested on 2022-12-23).
  ///
  /// **NOTE:** Forgetting a device twice will cause chrome to crash
  /// (last tested on 2022-12-23).
  ///
  /// See:
  ///
  /// - https://webbluetoothcg.github.io/web-bluetooth/#dom-bluetoothdevice-forget
  ///
  Future<void> forget() async {
    if (!hasForget) {
      throw NativeAPINotImplementedError("forget");
    }
    await _forget().toDart;
  }

  @JS("forget")
  external JSPromise _forget();

  ///
  /// Check to see if the browser/ user agent has the forget method.
  ///
  bool get hasForget => has("forget");

  @JS("watchingAdvertisements")
  external JSBoolean? get _watchingAdvertisements;

  ///
  /// If the device is watching for advertisements.
  /// If advertisements are not unsupported then it will always return `false`.
  ///
  bool get watchingAdvertisements =>
      _watchingAdvertisements != null &&
      _watchingAdvertisements.isDefinedAndNotNull &&
      _watchingAdvertisements!.toDart;

  ///
  /// Add a new event listener to the device.
  ///
  /// Converting the method to a [JSFunction] will be done automatically for you.
  /// This function is then returned so that you can later call [removeEventListener].
  ///
  /// Events to be handled are:
  ///
  /// - onadvertisementreceived
  ///
  /// - ongattserverdisconnected
  ///
  /// - onserviceadded
  ///
  /// - onservicechanged
  ///
  /// - onserviceremoved
  ///
  /// - oncharacteristicvaluechanged
  ///
  /// See:
  ///
  /// - [removeEventListener]
  ///
  /// - https://webbluetoothcg.github.io/web-bluetooth/#bluetoothdeviceeventhandlers
  ///
  /// - https://webbluetoothcg.github.io/web-bluetooth/#serviceeventhandlers
  ///
  /// - https://webbluetoothcg.github.io/web-bluetooth/#characteristiceventhandlers
  ///
  JSFunction addEventListenerDart<T extends JSAny?>(
      final String type, final void Function(T) listener,
      [final AddEventListenerOptions? options]) {
    final callback = listener.toJS;
    if (options?.isDefinedAndNotNull ?? false) {
      addEventListener(type, callback, options!);
    } else {
      addEventListener(type, callback);
    }
    return callback;
  }
}

///
/// The options to configure the [WebBluetoothDevice.watchAdvertisements]
/// method.
///
/// The only option that this object has right now is [signal] which can be
/// used to abort watching for advertisements.
///
@JS()
@anonymous
extension type WatchAdvertisementsOptions._(JSObject _) implements JSObject {
  ///
  /// This signal can be used to abort watching later on in the program.
  ///
  /// See [AbortController] and [AbortSignal] on how to create these signals.
  ///
  external AbortSignal? get signal;

  ///
  /// The constructor for the options.
  ///
  external factory WatchAdvertisementsOptions({final AbortSignal? signal});
}
