package com.vayunmathur.calendar.ui.dialog

import androidx.compose.material3.Button
import androidx.compose.material3.DatePicker
import androidx.compose.material3.DatePickerDialog
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.SelectableDates
import androidx.compose.material3.Text
import androidx.compose.material3.rememberDatePickerState
import androidx.compose.runtime.Composable
import androidx.navigation3.runtime.NavBackStack
import com.vayunmathur.calendar.Route
import com.vayunmathur.calendar.vutil.LocalNavResultRegistry
import com.vayunmathur.calendar.vutil.pop
import kotlin.time.Instant
import kotlinx.coroutines.launch
import androidx.compose.runtime.rememberCoroutineScope
import kotlinx.datetime.TimeZone
import kotlinx.datetime.atStartOfDayIn
import kotlinx.datetime.toLocalDateTime
import kotlinx.datetime.LocalDate

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun DatePickerDialog(backStack: NavBackStack<Route>, resultKey: String, initialDate: LocalDate, minDate: LocalDate? = null) {
    val registry = LocalNavResultRegistry.current
    val state = rememberDatePickerState(initialDate.atStartOfDayIn(TimeZone.UTC).toEpochMilliseconds(),
        selectableDates = object: SelectableDates {
            override fun isSelectableDate(utcTimeMillis: Long): Boolean {
                if(minDate == null) return true
                return Instant.fromEpochMilliseconds(utcTimeMillis).toLocalDateTime(TimeZone.UTC).date >= minDate
            }

            override fun isSelectableYear(year: Int): Boolean {
                if(minDate == null) return true
                return year >= minDate.year
            }
        })
    val scope = rememberCoroutineScope()
    DatePickerDialog(
        onDismissRequest = { backStack.pop() },
        confirmButton = {
            Button(onClick = {
                val selectedMs = state.selectedDateMillis!!
                val result = Instant.fromEpochMilliseconds(selectedMs)
                    .toLocalDateTime(TimeZone.UTC).date
                scope.launch { registry.dispatchResult(resultKey, result) }
                backStack.pop()
            }, enabled = state.selectedDateMillis != null) {
                Text("OK")
            }
        },
        dismissButton = {
            Button(onClick = { backStack.pop() }) { Text("Cancel") }
        }
    ) {
        DatePicker(state)
    }
}
