#pragma once

/**
** \defgroup taskstub Task Stub
** Task Stub is a poorly-named representation of a remote Task that can be
** exposed to JS.
** @{
*/

#include "quickjs.h"
#include "uv.h"

/** A task identifier. */
typedef int taskid_t;

/** A packet stream. */
typedef struct _tf_packetstream_t tf_packetstream_t;

/** A task. */
typedef struct _tf_task_t tf_task_t;

/** A handle to another task. */
typedef struct _tf_taskstub_t tf_taskstub_t;

/** Initialize task stub.  Call before using the rest. */
void tf_taskstub_startup();

/**
** Register the task stub script interface.
** @param context The JS context.
** @return The task stub constructor.
*/
JSValue tf_taskstub_register(JSContext* context);

/**
** Get a unique identifier for the task stub.
** @param stub The task stub.
** @return An identifier for the stub.
*/
taskid_t tf_taskstub_get_id(const tf_taskstub_t* stub);

/**
** Get the JS object representing the task stub.
** @param stub The task stub.
** @return The JS object.
*/
JSValue tf_taskstub_get_task_object(const tf_taskstub_t* stub);

/**
** Get the packet stream that can be used to communicate with the task stub.
** @param stub The task stub.
** @return The packet stream.
*/
tf_packetstream_t* tf_taskstub_get_stream(const tf_taskstub_t* stub);

/**
** Get the task owning the task stub.
** @param stub The task stub.
** @return The task from which the task stub was created.
*/
tf_task_t* tf_taskstub_get_owner(const tf_taskstub_t* stub);

/**
** Create a task stub representing the parent task of the running process.
** @param task The running task.
** @param file A file descriptor of a pipe connected to a parent process task.
** @return The created task stub.
*/
tf_taskstub_t* tf_taskstub_create_parent(tf_task_t* task, uv_file file);

/**
** Report an error to a task stub.
** @param stub The stub to which to report th eerror.
** @param error The error to report.
*/
void tf_taskstub_on_error(tf_taskstub_t* stub, JSValue error);

/**
** Print to a task stub.
** @param stub The task stub to which to print.
** @param arguments The values to print.
*/
void tf_taskstub_on_print(tf_taskstub_t* stub, JSValue arguments);

/**
** Terminate a task stub.
** @param stub The task stub to kill.
** @return A promise if it is happening asynchronously.
*/
JSValue tf_taskstub_kill(tf_taskstub_t* stub);

/** @} */
