#include "ssb.js.h"

#include "log.h"
#include "mem.h"
#include "ssb.db.h"
#include "ssb.ebt.h"
#include "ssb.h"
#include "util.js.h"

#include "sodium/crypto_sign.h"
#include "sqlite3.h"
#include "string.h"
#include "uv.h"

#include <inttypes.h>

static const int k_sql_async_timeout_ms = 60 * 1000;

static JSClassID _tf_ssb_classId;

typedef struct _create_identity_t
{
	char id[k_id_base64_len];
	bool error_add;
	bool error_too_many;
	JSValue promise[2];
	char user[];
} create_identity_t;

static void _tf_ssb_create_identity_work(tf_ssb_t* ssb, void* user_data)
{
	create_identity_t* work = user_data;
	int count = tf_ssb_db_identity_get_count_for_user(ssb, work->user);
	if (count < 16)
	{
		char public[k_id_base64_len - 1];
		char private[512];
		tf_ssb_generate_keys_buffer(public, sizeof(public), private, sizeof(private));
		if (tf_ssb_db_identity_add(ssb, work->user, public, private))
		{
			snprintf(work->id, sizeof(work->id), "@%s", public);
		}
		else
		{
			work->error_add = true;
		}
	}
	else
	{
		work->error_too_many = true;
	}
}

static void _tf_ssb_create_identity_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	JSContext* context = tf_ssb_get_context(ssb);
	JSValue result = JS_UNDEFINED;
	create_identity_t* work = user_data;
	if (work->error_too_many)
	{
		result = JS_ThrowInternalError(context, "Too many identities for user.");
	}
	else if (work->error_add)
	{
		result = JS_ThrowInternalError(context, "Unable to add identity.");
	}
	else
	{
		result = JS_NewString(context, work->id);
	}

	JSValue error = JS_Call(context, work->promise[0], JS_UNDEFINED, 1, &result);
	JS_FreeValue(context, result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	tf_free(work);
}

static JSValue _tf_ssb_createIdentity(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	tf_ssb_t* ssb = JS_GetOpaque(this_val, _tf_ssb_classId);
	JSValue result = JS_UNDEFINED;
	if (ssb)
	{
		size_t length = 0;
		const char* user = JS_ToCStringLen(context, &length, argv[0]);
		create_identity_t* work = tf_malloc(sizeof(create_identity_t) + length + 1);
		*work = (create_identity_t) { 0 };
		memcpy(work->user, user, length + 1);
		JS_FreeCString(context, user);
		result = JS_NewPromiseCapability(context, work->promise);
		tf_ssb_run_work(ssb, _tf_ssb_create_identity_work, _tf_ssb_create_identity_after_work, work);
	}
	return result;
}

static JSValue _tf_ssb_getServerIdentity(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	JSValue result = JS_UNDEFINED;
	tf_ssb_t* ssb = JS_GetOpaque(this_val, _tf_ssb_classId);
	if (ssb)
	{
		char id[k_id_base64_len] = { 0 };
		if (tf_ssb_whoami(ssb, id, sizeof(id)))
		{
			result = JS_NewString(context, id);
		}
	}
	return result;
}

typedef struct _identity_info_work_t
{
	JSContext* context;
	const char* name;
	const char* package_owner;
	const char* package_name;
	tf_ssb_identity_info_t* info;
	JSValue promise[2];
} identity_info_work_t;

static void _tf_ssb_getIdentityInfo_work(tf_ssb_t* ssb, void* user_data)
{
	identity_info_work_t* request = user_data;
	request->info = tf_ssb_db_get_identity_info(ssb, request->name, request->package_owner, request->package_name);
}

static void _tf_ssb_getIdentityInfo_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	identity_info_work_t* request = user_data;
	JSContext* context = request->context;
	JSValue result = JS_NewObject(context);

	JSValue identities = JS_NewArray(context);
	for (int i = 0; i < request->info->count; i++)
	{
		JS_SetPropertyUint32(context, identities, i, JS_NewString(context, request->info->identity[i]));
	}
	JS_SetPropertyStr(context, result, "identities", identities);

	JSValue names = JS_NewObject(context);
	for (int i = 0; i < request->info->count; i++)
	{
		JS_SetPropertyStr(context, names, request->info->identity[i], JS_NewString(context, request->info->name[i] ? request->info->name[i] : request->info->identity[i]));
	}
	JS_SetPropertyStr(context, result, "names", names);

	JS_SetPropertyStr(context, result, "identity", JS_NewString(context, request->info->active_identity));

	JSValue error = JS_Call(context, request->promise[0], JS_UNDEFINED, 1, &result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, result);
	JS_FreeValue(context, request->promise[0]);
	JS_FreeValue(context, request->promise[1]);

	tf_free((void*)request->name);
	tf_free((void*)request->package_owner);
	tf_free((void*)request->package_name);
	tf_free(request->info);
	tf_free(request);
}

static JSValue _tf_ssb_getIdentityInfo(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	tf_ssb_t* ssb = JS_GetOpaque(this_val, _tf_ssb_classId);
	const char* name = JS_ToCString(context, argv[0]);
	const char* package_owner = JS_ToCString(context, argv[1]);
	const char* package_name = JS_ToCString(context, argv[2]);
	identity_info_work_t* work = tf_malloc(sizeof(identity_info_work_t));
	*work = (identity_info_work_t) {
		.context = context,
		.name = tf_strdup(name),
		.package_owner = tf_strdup(package_owner),
		.package_name = tf_strdup(package_name),
	};
	JSValue result = JS_NewPromiseCapability(context, work->promise);
	JS_FreeCString(context, name);
	JS_FreeCString(context, package_owner);
	JS_FreeCString(context, package_name);

	tf_ssb_run_work(ssb, _tf_ssb_getIdentityInfo_work, _tf_ssb_getIdentityInfo_after_work, work);
	return result;
}

typedef struct _blob_get_t
{
	JSContext* context;
	JSValue promise[2];
} blob_get_t;

static void _tf_ssb_blobGet_callback(bool found, const uint8_t* data, size_t size, void* user_data)
{
	blob_get_t* get = user_data;
	JSValue result = JS_UNDEFINED;
	if (found)
	{
		result = JS_NewArrayBufferCopy(get->context, data, size);
	}
	JSValue error = JS_Call(get->context, get->promise[0], JS_UNDEFINED, 1, &result);
	JS_FreeValue(get->context, result);
	JS_FreeValue(get->context, get->promise[0]);
	JS_FreeValue(get->context, get->promise[1]);
	tf_util_report_error(get->context, error);
	JS_FreeValue(get->context, error);
	tf_free(get);
}

static JSValue _tf_ssb_blobGet(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	JSValue result = JS_NULL;
	tf_ssb_t* ssb = JS_GetOpaque(this_val, _tf_ssb_classId);
	if (ssb)
	{
		const char* id = JS_ToCString(context, argv[0]);
		blob_get_t* get = tf_malloc(sizeof(blob_get_t));
		*get = (blob_get_t) { .context = context };
		result = JS_NewPromiseCapability(context, get->promise);
		tf_ssb_db_blob_get_async(ssb, id, _tf_ssb_blobGet_callback, get);
		JS_FreeCString(context, id);
	}
	return result;
}

typedef struct _blob_store_t
{
	JSContext* context;
	JSValue promise[2];
	uint8_t* buffer;
} blob_store_t;

static void _tf_ssb_blob_store_complete(blob_store_t* store, const char* id)
{
	JSValue result = JS_UNDEFINED;
	if (id)
	{
		JSValue id_value = JS_NewString(store->context, id);
		JSValue result = JS_Call(store->context, store->promise[0], JS_UNDEFINED, 1, &id_value);
		JS_FreeValue(store->context, id_value);
		tf_util_report_error(store->context, result);
		JS_FreeValue(store->context, result);
	}
	else
	{
		JSValue result = JS_Call(store->context, store->promise[1], JS_UNDEFINED, 0, NULL);
		tf_util_report_error(store->context, result);
		JS_FreeValue(store->context, result);
	}
	tf_util_report_error(store->context, result);
	JS_FreeValue(store->context, result);
	JS_FreeValue(store->context, store->promise[0]);
	JS_FreeValue(store->context, store->promise[1]);
	tf_free(store->buffer);
	tf_free(store);
}

static void _tf_ssb_blob_store_callback(const char* id, bool is_new, void* user_data)
{
	blob_store_t* store = user_data;
	_tf_ssb_blob_store_complete(store, id);
}

static JSValue _tf_ssb_blobStore(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	blob_store_t* store = tf_malloc(sizeof(blob_store_t));
	*store = (blob_store_t) { .context = context };
	JSValue result = JS_NewPromiseCapability(context, store->promise);

	tf_ssb_t* ssb = JS_GetOpaque(this_val, _tf_ssb_classId);
	if (ssb)
	{
		uint8_t* blob = NULL;
		size_t size = 0;
		if (JS_IsString(argv[0]))
		{
			const char* text = JS_ToCStringLen(context, &size, argv[0]);
			store->buffer = tf_malloc(size);
			memcpy(store->buffer, text, size);
			tf_ssb_db_blob_store_async(ssb, store->buffer, size, _tf_ssb_blob_store_callback, store);
			JS_FreeCString(context, text);
		}
		else if ((blob = tf_util_try_get_array_buffer(context, &size, argv[0])) != 0)
		{
			store->buffer = tf_malloc(size);
			memcpy(store->buffer, blob, size);
			tf_ssb_db_blob_store_async(ssb, store->buffer, size, _tf_ssb_blob_store_callback, store);
		}
		else
		{
			size_t offset;
			size_t element_size;
			JSValue buffer = tf_util_try_get_typed_array_buffer(context, argv[0], &offset, &size, &element_size);
			if (!JS_IsException(buffer))
			{
				blob = tf_util_try_get_array_buffer(context, &size, buffer);
				if (blob)
				{
					store->buffer = tf_malloc(size);
					memcpy(store->buffer, blob, size);
					tf_ssb_db_blob_store_async(ssb, store->buffer, size, _tf_ssb_blob_store_callback, store);
				}
				else
				{
					_tf_ssb_blob_store_complete(store, NULL);
				}
			}
			else
			{
				_tf_ssb_blob_store_complete(store, NULL);
			}
			JS_FreeValue(context, buffer);
		}
	}
	return result;
}

static JSValue _tf_ssb_connections(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	JSValue result = JS_NULL;
	tf_ssb_t* ssb = JS_GetOpaque(this_val, _tf_ssb_classId);
	if (ssb)
	{
		tf_ssb_connection_t* connections[32];
		int count = tf_ssb_get_connections(ssb, connections, tf_countof(connections));

		result = JS_NewArray(context);
		for (int i = 0; i < count; i++)
		{
			char id[k_id_base64_len] = { 0 };
			tf_ssb_connection_t* connection = connections[i];
			JSValue object = JS_NewObject(context);
			tf_ssb_connection_get_id(connection, id, sizeof(id));
			JS_SetPropertyStr(context, object, "id", JS_NewString(context, id));
			JS_SetPropertyStr(context, object, "host", JS_NewString(context, tf_ssb_connection_get_host(connection)));
			JS_SetPropertyStr(context, object, "port", JS_NewInt32(context, tf_ssb_connection_get_port(connection)));
			tf_ssb_connection_t* tunnel = tf_ssb_connection_get_tunnel(connection);
			if (tunnel)
			{
				int tunnel_index = -1;
				for (int j = 0; j < count; j++)
				{
					if (connections[j] == tunnel)
					{
						tunnel_index = j;
						break;
					}
				}
				JS_SetPropertyStr(context, object, "tunnel", JS_NewInt32(context, tunnel_index));
			}
			JS_SetPropertyStr(context, object, "requests", tf_ssb_connection_requests_to_object(connection));
			JSValue flags_object = JS_NewObject(context);
			int flags = tf_ssb_connection_get_flags(connection);
			JS_SetPropertyStr(context, flags_object, "one_shot", JS_NewBool(context, (flags & k_tf_ssb_connect_flag_one_shot) != 0));
			JS_SetPropertyStr(context, object, "flags", flags_object);
			JS_SetPropertyStr(context, object, "connected", JS_NewBool(context, tf_ssb_connection_is_connected(connection)));
			const char* destroy_reason = tf_ssb_connection_get_destroy_reason(connection);
			if (destroy_reason)
			{
				JS_SetPropertyStr(context, object, "destroy_reason", JS_NewString(context, destroy_reason));
			}
			int in = 0;
			int out = 0;
			int max_in = 0;
			int max_out = 0;
			tf_ssb_ebt_get_progress(tf_ssb_connection_get_ebt(connection), &in, &max_in, &out, &max_out);
			JSValue progress = JS_NewObject(context);
			JSValue in_progress = JS_NewObject(context);
			JS_SetPropertyStr(context, in_progress, "current", JS_NewInt32(context, in));
			JS_SetPropertyStr(context, in_progress, "total", JS_NewInt32(context, max_in));
			JS_SetPropertyStr(context, progress, "in", in_progress);
			JSValue out_progress = JS_NewObject(context);
			JS_SetPropertyStr(context, out_progress, "current", JS_NewInt32(context, out));
			JS_SetPropertyStr(context, out_progress, "total", JS_NewInt32(context, max_out));
			JS_SetPropertyStr(context, progress, "out", out_progress);
			JS_SetPropertyStr(context, object, "progress", progress);
			JS_SetPropertyUint32(context, result, i, object);
		}
	}
	return result;
}

typedef struct _stored_connections_t
{
	int count;
	tf_ssb_db_stored_connection_t* connections;
	JSValue promise[2];
} stored_connections_t;

static void _tf_ssb_stored_connections_work(tf_ssb_t* ssb, void* user_data)
{
	stored_connections_t* work = user_data;
	work->connections = tf_ssb_db_get_stored_connections(ssb, &work->count);
}

static void _tf_ssb_stored_connections_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	stored_connections_t* work = user_data;
	JSContext* context = tf_ssb_get_context(ssb);
	JSValue result = JS_NewArray(context);
	for (int i = 0; i < work->count; i++)
	{
		JSValue connection = JS_NewObject(context);
		JS_SetPropertyStr(context, connection, "address", JS_NewString(context, work->connections[i].address));
		JS_SetPropertyStr(context, connection, "port", JS_NewInt32(context, work->connections[i].port));
		JS_SetPropertyStr(context, connection, "pubkey", JS_NewString(context, work->connections[i].pubkey));
		JS_SetPropertyStr(context, connection, "last_attempt", JS_NewInt64(context, work->connections[i].last_attempt));
		JS_SetPropertyStr(context, connection, "last_success", JS_NewInt64(context, work->connections[i].last_success));
		JS_SetPropertyUint32(context, result, i, connection);
	}
	tf_free(work->connections);

	JSValue error = JS_Call(context, work->promise[0], JS_UNDEFINED, 1, &result);
	JS_FreeValue(context, result);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	tf_free(work);
}

static JSValue _tf_ssb_storedConnections(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	JSValue result = JS_NULL;
	tf_ssb_t* ssb = JS_GetOpaque(this_val, _tf_ssb_classId);
	if (ssb)
	{
		stored_connections_t* work = tf_malloc(sizeof(stored_connections_t));
		*work = (stored_connections_t) { 0 };
		result = JS_NewPromiseCapability(context, work->promise);
		tf_ssb_run_work(ssb, _tf_ssb_stored_connections_work, _tf_ssb_stored_connections_after_work, work);
	}
	return result;
}

static JSValue _tf_ssb_getConnection(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	tf_ssb_t* ssb = JS_GetOpaque(this_val, _tf_ssb_classId);
	const char* id = JS_ToCString(context, argv[0]);
	tf_ssb_connection_t* connection = tf_ssb_connection_get(ssb, id);
	JS_FreeCString(context, id);
	return JS_DupValue(context, tf_ssb_connection_get_object(connection));
}

static JSValue _tf_ssb_closeConnection(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	tf_ssb_t* ssb = JS_GetOpaque(this_val, _tf_ssb_classId);
	const char* id = JS_ToCString(context, argv[0]);
	tf_ssb_connection_t* connection = tf_ssb_connection_get(ssb, id);
	if (connection)
	{
		tf_ssb_connection_close(connection, "Closed by user");
	}
	JS_FreeCString(context, id);
	return connection ? JS_TRUE : JS_FALSE;
}

typedef struct _sql_work_t
{
	tf_ssb_t* ssb;
	sqlite3* db;
	char* error;
	const char* query;
	uint8_t* binds;
	uint8_t* rows;
	size_t binds_count;
	size_t rows_count;
	uv_async_t async;
	uv_timer_t timeout;
	uv_mutex_t lock;
	JSValue callback;
	JSValue promise[2];
	int result;
} sql_work_t;

static void _tf_ssb_sql_append(uint8_t** rows, size_t* rows_count, const void* data, size_t size)
{
	*rows = tf_resize_vec(*rows, *rows_count + size);
	memcpy(*rows + *rows_count, data, size);
	*rows_count += size;
}

static void _tf_ssb_sqlAsync_work(tf_ssb_t* ssb, void* user_data)
{
	sql_work_t* sql_work = user_data;
	sqlite3* db = tf_ssb_acquire_db_reader_restricted(ssb);
	uv_mutex_lock(&sql_work->lock);
	sql_work->db = db;
	uv_mutex_unlock(&sql_work->lock);
	uv_async_send(&sql_work->async);
	sqlite3_stmt* statement = NULL;
	sql_work->result = sqlite3_prepare_v2(db, sql_work->query, -1, &statement, NULL);
	if (sql_work->result == SQLITE_OK && statement)
	{
		const uint8_t* p = sql_work->binds;
		int column = 0;
		while (p < sql_work->binds + sql_work->binds_count)
		{
			switch (*p++)
			{
			case SQLITE_INTEGER:
				{
					int64_t value = 0;
					memcpy(&value, p, sizeof(value));
					sqlite3_bind_int64(statement, column + 1, value);
					p += sizeof(value);
				}
				break;
			case SQLITE_TEXT:
				{
					size_t length = 0;
					memcpy(&length, p, sizeof(length));
					p += sizeof(length);
					sqlite3_bind_text(statement, column + 1, (const char*)p, length, NULL);
					p += length;
				}
				break;
			case SQLITE_NULL:
				sqlite3_bind_null(statement, column + 1);
				break;
			default:
				abort();
			}
			column++;
		}
		int r = SQLITE_OK;
		while ((r = sqlite3_step(statement)) == SQLITE_ROW)
		{
			_tf_ssb_sql_append(&sql_work->rows, &sql_work->rows_count, &(uint8_t[]) { 'r' }, 1);
			for (int i = 0; i < sqlite3_column_count(statement); i++)
			{
				_tf_ssb_sql_append(&sql_work->rows, &sql_work->rows_count, &(uint8_t[]) { 'c' }, 1);
				const char* name = sqlite3_column_name(statement, i);
				_tf_ssb_sql_append(&sql_work->rows, &sql_work->rows_count, name, strlen(name) + 1);
				uint8_t type = sqlite3_column_type(statement, i);
				_tf_ssb_sql_append(&sql_work->rows, &sql_work->rows_count, &type, sizeof(type));
				switch (type)
				{
				case SQLITE_INTEGER:
					{
						int64_t value = sqlite3_column_int64(statement, i);
						_tf_ssb_sql_append(&sql_work->rows, &sql_work->rows_count, &value, sizeof(value));
					}
					break;
				case SQLITE_FLOAT:
					{
						double value = sqlite3_column_double(statement, i);
						_tf_ssb_sql_append(&sql_work->rows, &sql_work->rows_count, &value, sizeof(value));
					}
					break;
				case SQLITE_TEXT:
					{
						size_t bytes = sqlite3_column_bytes(statement, i);
						_tf_ssb_sql_append(&sql_work->rows, &sql_work->rows_count, &bytes, sizeof(bytes));
						_tf_ssb_sql_append(&sql_work->rows, &sql_work->rows_count, sqlite3_column_text(statement, i), bytes);
					}
					break;
				case SQLITE_BLOB:
					{
						size_t bytes = sqlite3_column_bytes(statement, i);
						_tf_ssb_sql_append(&sql_work->rows, &sql_work->rows_count, &bytes, sizeof(bytes));
						_tf_ssb_sql_append(&sql_work->rows, &sql_work->rows_count, sqlite3_column_blob(statement, i), bytes);
					}
					break;
				case SQLITE_NULL:
					break;
				default:
					abort();
				}
			}
		}
		sql_work->result = r;
		if (r == SQLITE_MISUSE)
		{
			sql_work->error = tf_strdup(sqlite3_errstr(sql_work->result));
		}
		else if (r != SQLITE_OK && r != SQLITE_DONE)
		{
			if (sqlite3_is_interrupted(db))
			{
				sql_work->error = tf_strdup("Timed out");
			}
			else
			{
				sql_work->error = tf_strdup(sqlite3_errmsg(db));
			}
		}
		_tf_ssb_sql_append(&sql_work->rows, &sql_work->rows_count, &(uint8_t[]) { 0 }, 1);
		sqlite3_finalize(statement);
	}
	else if (sql_work->result != SQLITE_OK)
	{
		sql_work->error = tf_strdup(sqlite3_errmsg(db));
	}
	else
	{
		sql_work->result = SQLITE_ERROR;
		sql_work->error = tf_strdup("Statement not prepared");
	}
	uv_mutex_lock(&sql_work->lock);
	sql_work->db = NULL;
	uv_mutex_unlock(&sql_work->lock);
	tf_ssb_release_db_reader(ssb, db);
}

static void _tf_ssb_sqlAsync_handle_close(uv_handle_t* handle)
{
	sql_work_t* work = handle->data;
	handle->data = NULL;
	if (!work->async.data && !work->timeout.data)
	{
		tf_free(work);
	}
}

static void _tf_ssb_sqlAsync_destroy(sql_work_t* work)
{
	tf_free(work->binds);
	tf_free(work->error);
	if (work->rows)
	{
		tf_free(work->rows);
	}
	uv_mutex_destroy(&work->lock);
	uv_close((uv_handle_t*)&work->timeout, _tf_ssb_sqlAsync_handle_close);
	uv_close((uv_handle_t*)&work->async, _tf_ssb_sqlAsync_handle_close);
}

static void _tf_ssb_sqlAsync_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	sql_work_t* sql_work = user_data;
	JSContext* context = tf_ssb_get_context(ssb);
	uint8_t* p = sql_work->rows;
	JSValue result = JS_UNDEFINED;
	while (p < sql_work->rows + sql_work->rows_count && JS_IsUndefined(result))
	{
		if (*p++ == 'r')
		{
			JSValue row = JS_NewObject(context);

			while (*p == 'c' && JS_IsUndefined(result))
			{
				p++;
				const char* column_name = (const char*)p;
				size_t length = strlen((char*)p);
				p += length + 1;

				switch (*p++)
				{
				case SQLITE_INTEGER:
					{
						int64_t value = 0;
						memcpy(&value, p, sizeof(value));
						JS_SetPropertyStr(context, row, column_name, JS_NewInt64(context, value));
						p += sizeof(value);
					}
					break;
				case SQLITE_FLOAT:
					{
						double value = 0.0;
						memcpy(&value, p, sizeof(value));
						JS_SetPropertyStr(context, row, column_name, JS_NewFloat64(context, value));
						p += sizeof(value);
					}
					break;
				case SQLITE_TEXT:
				case SQLITE_BLOB:
					{
						size_t length = 0;
						memcpy(&length, p, sizeof(length));
						p += sizeof(length);
						JS_SetPropertyStr(context, row, column_name, JS_NewStringLen(context, (const char*)p, length));
						p += length;
					}
					break;
				case SQLITE_NULL:
					JS_SetPropertyStr(context, row, column_name, JS_NULL);
					break;
				}
			}

			result = JS_Call(context, sql_work->callback, JS_UNDEFINED, 1, &row);
			if (!JS_IsException(result))
			{
				JS_FreeValue(context, result);
				result = JS_UNDEFINED;
			}
			JS_FreeValue(context, row);
		}
		else
		{
			break;
		}
	}

	if (!JS_IsUndefined(result))
	{
		bool is_exception = JS_IsException(result);
		if (is_exception)
		{
			JSValue exception = JS_GetException(context);
			JS_FreeValue(context, result);
			result = exception;
		}
		JSValue promise_result = JS_Call(context, sql_work->promise[is_exception ? 1 : 0], JS_UNDEFINED, 1, &result);
		tf_util_report_error(context, promise_result);
		JS_FreeValue(context, promise_result);
	}
	else if (sql_work->result == SQLITE_OK || sql_work->result == SQLITE_DONE)
	{
		result = JS_Call(context, sql_work->promise[0], JS_UNDEFINED, 0, NULL);
		tf_util_report_error(context, result);
	}
	else
	{
		JSValue error = JS_ThrowInternalError(context, "SQL Error %s: %s", sql_work->error, sql_work->query);
		JSValue exception = JS_GetException(context);
		result = JS_Call(context, sql_work->promise[1], JS_UNDEFINED, 1, &exception);
		tf_util_report_error(context, result);
		JS_FreeValue(context, exception);
		JS_FreeValue(context, error);
	}
	JS_FreeValue(context, result);
	JS_FreeValue(context, sql_work->promise[0]);
	JS_FreeValue(context, sql_work->promise[1]);
	JS_FreeValue(context, sql_work->callback);
	JS_FreeCString(context, sql_work->query);
	_tf_ssb_sqlAsync_destroy(sql_work);
}

static void _tf_ssb_sqlAsync_timeout(uv_timer_t* timer)
{
	sql_work_t* work = timer->data;
	uv_mutex_lock(&work->lock);
	if (work->db)
	{
		sqlite3_interrupt(work->db);
	}
	uv_mutex_unlock(&work->lock);
}

static void _tf_ssb_sqlAsync_start_timer(uv_async_t* async)
{
	sql_work_t* work = async->data;
	uv_timer_start(&work->timeout, _tf_ssb_sqlAsync_timeout, k_sql_async_timeout_ms, 0);
}

static JSValue _tf_ssb_sqlAsync(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	tf_ssb_t* ssb = JS_GetOpaque(this_val, _tf_ssb_classId);
	const char* query = JS_ToCString(context, argv[0]);
	sql_work_t* work = tf_malloc(sizeof(sql_work_t));
	*work = (sql_work_t)
	{
		.async =
		{
			.data = work,
		},
		.timeout =
		{
			.data = work,
		},
		.ssb = ssb,
		.callback = JS_DupValue(context, argv[2]),
		.query = query,
	};
	uv_mutex_init(&work->lock);
	uv_async_init(tf_ssb_get_loop(ssb), &work->async, _tf_ssb_sqlAsync_start_timer);
	uv_timer_init(tf_ssb_get_loop(ssb), &work->timeout);
	JSValue result = JS_UNDEFINED;
	if (ssb)
	{
		result = JS_NewPromiseCapability(context, work->promise);
		int32_t length = tf_util_get_length(context, argv[1]);
		for (int i = 0; i < length; i++)
		{
			JSValue value = JS_GetPropertyUint32(context, argv[1], i);
			if (JS_IsNumber(value))
			{
				uint8_t type = SQLITE_INTEGER;
				int64_t number = 0;
				JS_ToInt64(context, &number, value);
				_tf_ssb_sql_append(&work->binds, &work->binds_count, &type, sizeof(type));
				_tf_ssb_sql_append(&work->binds, &work->binds_count, &number, sizeof(number));
			}
			else if (JS_IsBool(value))
			{
				uint8_t type = SQLITE_INTEGER;
				int64_t number = JS_ToBool(context, value) ? 1 : 0;
				_tf_ssb_sql_append(&work->binds, &work->binds_count, &type, sizeof(type));
				_tf_ssb_sql_append(&work->binds, &work->binds_count, &number, sizeof(number));
			}
			else if (JS_IsNull(value))
			{
				uint8_t type = SQLITE_NULL;
				_tf_ssb_sql_append(&work->binds, &work->binds_count, &type, sizeof(type));
			}
			else
			{
				uint8_t type = SQLITE_TEXT;
				size_t length = 0;
				const char* string = JS_ToCStringLen(context, &length, value);
				if (!string)
				{
					string = "";
				}
				_tf_ssb_sql_append(&work->binds, &work->binds_count, &type, sizeof(type));
				_tf_ssb_sql_append(&work->binds, &work->binds_count, &length, sizeof(length));
				_tf_ssb_sql_append(&work->binds, &work->binds_count, string, length);
				JS_FreeCString(context, string);
			}
			JS_FreeValue(context, value);
		}
		tf_ssb_run_work(ssb, _tf_ssb_sqlAsync_work, _tf_ssb_sqlAsync_after_work, work);
	}
	return result;
}

typedef struct _message_store_t
{
	JSContext* context;
	JSValue promise[2];
} message_store_t;

static void _tf_ssb_message_store_callback(const char* id, bool verified, bool is_new, void* user_data)
{
	message_store_t* store = user_data;
	JSValue result = JS_Call(store->context, id ? store->promise[0] : store->promise[1], JS_UNDEFINED, 0, NULL);
	tf_util_report_error(store->context, result);
	JS_FreeValue(store->context, result);
	JS_FreeValue(store->context, store->promise[0]);
	JS_FreeValue(store->context, store->promise[1]);
	tf_free(store);
}

static JSValue _tf_ssb_storeMessage(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	tf_ssb_t* ssb = JS_GetOpaque(this_val, _tf_ssb_classId);
	message_store_t* store = tf_malloc(sizeof(message_store_t));
	*store = (message_store_t) { .context = context };
	JSValue result = JS_NewPromiseCapability(context, store->promise);
	tf_ssb_verify_strip_and_store_message(ssb, argv[0], _tf_ssb_message_store_callback, store);
	return result;
}

typedef struct _broadcasts_t
{
	JSContext* context;
	JSValue array;
	int length;
} broadcasts_t;

static void _tf_ssb_broadcasts_visit(
	const char* host, const struct sockaddr_in* addr, tf_ssb_broadcast_origin_t origin, tf_ssb_connection_t* tunnel, const uint8_t* pub, void* user_data)
{
	broadcasts_t* broadcasts = user_data;
	JSValue entry = JS_NewObject(broadcasts->context);
	char pubkey[k_id_base64_len];
	tf_ssb_id_bin_to_str(pubkey, sizeof(pubkey), pub);
	switch (origin)
	{
	case k_tf_ssb_broadcast_origin_discovery:
		JS_SetPropertyStr(broadcasts->context, entry, "origin", JS_NewString(broadcasts->context, "discovery"));
		break;
	case k_tf_ssb_broadcast_origin_room:
		JS_SetPropertyStr(broadcasts->context, entry, "origin", JS_NewString(broadcasts->context, "room"));
		break;
	case k_tf_ssb_broadcast_origin_peer_exchange:
		JS_SetPropertyStr(broadcasts->context, entry, "origin", JS_NewString(broadcasts->context, "peer_exchange"));
		break;
	}
	if (tunnel)
	{
		JS_SetPropertyStr(broadcasts->context, entry, "tunnel", JS_DupValue(broadcasts->context, tf_ssb_connection_get_object(tunnel)));
	}
	else
	{
		JS_SetPropertyStr(broadcasts->context, entry, "address", JS_NewString(broadcasts->context, host));
		JS_SetPropertyStr(broadcasts->context, entry, "port", JS_NewInt32(broadcasts->context, ntohs(addr->sin_port)));
	}
	JS_SetPropertyStr(broadcasts->context, entry, "pubkey", JS_NewString(broadcasts->context, pubkey));
	JS_SetPropertyUint32(broadcasts->context, broadcasts->array, broadcasts->length++, entry);
}

static JSValue _tf_ssb_getBroadcasts(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	JSValue result = JS_UNDEFINED;
	tf_ssb_t* ssb = JS_GetOpaque(this_val, _tf_ssb_classId);
	if (ssb)
	{
		result = JS_NewArray(context);
		broadcasts_t broadcasts = {
			.context = context,
			.array = result,
			.length = 0,
		};
		tf_ssb_visit_broadcasts(ssb, _tf_ssb_broadcasts_visit, &broadcasts);
	}
	return result;
}

typedef struct _connect_t
{
	JSContext* context;
	JSValue promise[2];
} connect_t;

static void _tf_ssb_connect_callback(tf_ssb_connection_t* connection, const char* reason, void* user_data)
{
	connect_t* connect = user_data;
	JSContext* context = connect->context;
	JSValue arg = connection ? JS_UNDEFINED : JS_NewString(context, reason);
	JSValue result = JS_Call(context, connection ? connect->promise[0] : connect->promise[1], JS_UNDEFINED, connection ? 0 : 1, &arg);
	tf_util_report_error(context, result);
	JS_FreeValue(context, result);
	JS_FreeValue(context, connect->promise[0]);
	JS_FreeValue(context, connect->promise[1]);
	JS_FreeValue(context, arg);
	tf_free(connect);
}

static JSValue _tf_ssb_connect(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	JSValue result = JS_UNDEFINED;
	JSValue args = argv[0];
	tf_ssb_t* ssb = JS_GetOpaque(this_val, _tf_ssb_classId);
	if (ssb)
	{
		connect_t* connect = tf_malloc(sizeof(connect_t));
		*connect = (connect_t) { .context = context };
		result = JS_NewPromiseCapability(context, connect->promise);
		if (JS_IsString(args))
		{
			const char* address_str = JS_ToCString(context, args);
			tf_printf("Connecting to %s\n", address_str);
			tf_ssb_connect_str(ssb, address_str, 0, _tf_ssb_connect_callback, connect);
			JS_FreeCString(context, address_str);
		}
		else
		{
			JSValue address = JS_GetPropertyStr(context, args, "address");
			JSValue port = JS_GetPropertyStr(context, args, "port");
			JSValue pubkey = JS_GetPropertyStr(context, args, "pubkey");
			const char* address_str = JS_ToCString(context, address);
			int32_t port_int = 0;
			JS_ToInt32(context, &port_int, port);
			const char* pubkey_str = JS_ToCString(context, pubkey);
			if (pubkey_str)
			{
				tf_printf("Connecting to %s:%d\n", address_str, port_int);
				uint8_t pubkey_bin[k_id_bin_len];
				tf_ssb_id_str_to_bin(pubkey_bin, pubkey_str);
				tf_ssb_connect(ssb, address_str, port_int, pubkey_bin, 0, _tf_ssb_connect_callback, connect);
			}
			else
			{
				_tf_ssb_connect_callback(NULL, "Not connecting to null.", connect);
			}
			JS_FreeCString(context, pubkey_str);
			JS_FreeCString(context, address_str);
			JS_FreeValue(context, address);
			JS_FreeValue(context, port);
			JS_FreeValue(context, pubkey);
		}
	}
	return result;
}

typedef struct _forget_stored_connection_t
{
	const char* address;
	int32_t port;
	const char* pubkey;
	JSValue promise[2];
} forget_stored_connection_t;

static void _tf_ssb_forget_stored_connection_work(tf_ssb_t* ssb, void* user_data)
{
	forget_stored_connection_t* work = user_data;
	if (work->pubkey)
	{
		tf_ssb_db_forget_stored_connection(ssb, work->address, work->port, work->pubkey);
	}
}

static void _tf_ssb_forget_stored_connection_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	forget_stored_connection_t* work = user_data;
	JSContext* context = tf_ssb_get_context(ssb);
	JS_FreeCString(context, work->pubkey);
	JS_FreeCString(context, work->address);
	JSValue result = JS_Call(context, work->promise[0], JS_UNDEFINED, 0, NULL);
	tf_util_report_error(context, result);
	JS_FreeValue(context, result);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	tf_free(work);
}

static JSValue _tf_ssb_forgetStoredConnection(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	JSValue result = JS_UNDEFINED;
	JSValue args = argv[0];
	tf_ssb_t* ssb = JS_GetOpaque(this_val, _tf_ssb_classId);
	if (ssb)
	{
		JSValue address = JS_GetPropertyStr(context, args, "address");
		JSValue port = JS_GetPropertyStr(context, args, "port");
		JSValue pubkey = JS_GetPropertyStr(context, args, "pubkey");
		const char* address_str = JS_ToCString(context, address);
		int32_t port_int = 0;
		JS_ToInt32(context, &port_int, port);
		const char* pubkey_str = JS_ToCString(context, pubkey);

		forget_stored_connection_t* work = tf_malloc(sizeof(forget_stored_connection_t));
		*work = (forget_stored_connection_t) {
			.address = address_str,
			.port = port_int,
			.pubkey = pubkey_str,
		};
		result = JS_NewPromiseCapability(context, work->promise);
		JS_FreeValue(context, address);
		JS_FreeValue(context, port);
		JS_FreeValue(context, pubkey);
		tf_ssb_run_work(ssb, _tf_ssb_forget_stored_connection_work, _tf_ssb_forget_stored_connection_after_work, work);
	}
	return result;
}

static void _tf_ssb_cleanup_value(tf_ssb_t* ssb, void* user_data)
{
	JSValue callback = JS_MKPTR(JS_TAG_OBJECT, user_data);
	JS_FreeValue(tf_ssb_get_context(ssb), callback);
}

static void _tf_ssb_on_message_added_callback(tf_ssb_t* ssb, const char* author, int32_t sequence, const char* id, void* user_data)
{
	JSContext* context = tf_ssb_get_context(ssb);
	JSValue callback = JS_MKPTR(JS_TAG_OBJECT, user_data);
	JSValue string = JS_NewString(context, id);
	JSValue response = JS_Call(context, callback, JS_UNDEFINED, 1, &string);
	if (tf_util_report_error(context, response))
	{
		tf_ssb_remove_message_added_callback(ssb, _tf_ssb_on_message_added_callback, user_data);
	}
	JS_FreeValue(context, response);
	JS_FreeValue(context, string);
}

static void _tf_ssb_on_blob_stored_callback(tf_ssb_t* ssb, const char* id, void* user_data)
{
	JSContext* context = tf_ssb_get_context(ssb);
	JSValue callback = JS_MKPTR(JS_TAG_OBJECT, user_data);
	JSValue string = JS_NewString(context, id);
	JSValue response = JS_Call(context, callback, JS_UNDEFINED, 1, &string);
	if (tf_util_report_error(context, response))
	{
		tf_ssb_remove_blob_stored_callback(ssb, _tf_ssb_on_blob_stored_callback, user_data);
	}
	JS_FreeValue(context, response);
	JS_FreeValue(context, string);
}

static void _tf_ssb_on_blob_want_added_callback(tf_ssb_t* ssb, const char* id, void* user_data)
{
	JSContext* context = tf_ssb_get_context(ssb);
	JSValue callback = JS_MKPTR(JS_TAG_OBJECT, user_data);
	JSValue string = JS_NewString(context, id);
	JSValue response = JS_Call(context, callback, JS_UNDEFINED, 1, &string);
	if (tf_util_report_error(context, response))
	{
		tf_ssb_remove_blob_want_added_callback(ssb, _tf_ssb_on_blob_want_added_callback, user_data);
	}
	JS_FreeValue(context, response);
	JS_FreeValue(context, string);
}

static void _tf_ssb_on_connections_changed_callback(tf_ssb_t* ssb, tf_ssb_change_t change, tf_ssb_connection_t* connection, void* user_data)
{
	JSContext* context = tf_ssb_get_context(ssb);
	JSValue callback = JS_MKPTR(JS_TAG_OBJECT, user_data);
	JSValue response = JS_UNDEFINED;
	switch (change)
	{
	case k_tf_ssb_change_create:
		break;
	case k_tf_ssb_change_update:
		{
			JSValue object = JS_DupValue(context, tf_ssb_connection_get_object(connection));
			JSValue args[] = {
				JS_NewString(context, "update"),
				object,
			};
			response = JS_Call(context, callback, JS_UNDEFINED, 2, args);
			if (tf_util_report_error(context, response))
			{
				tf_ssb_remove_connections_changed_callback(ssb, _tf_ssb_on_connections_changed_callback, user_data);
			}
			JS_FreeValue(context, args[0]);
			JS_FreeValue(context, object);
		}
		break;
	case k_tf_ssb_change_connect:
		{
			JSValue object = JS_DupValue(context, tf_ssb_connection_get_object(connection));
			JSValue args[] = {
				JS_NewString(context, "add"),
				object,
			};
			response = JS_Call(context, callback, JS_UNDEFINED, 2, args);
			if (tf_util_report_error(context, response))
			{
				tf_ssb_remove_connections_changed_callback(ssb, _tf_ssb_on_connections_changed_callback, user_data);
			}
			JS_FreeValue(context, args[0]);
			JS_FreeValue(context, object);
		}
		break;
	case k_tf_ssb_change_remove:
		{
			JSValue object = JS_DupValue(context, tf_ssb_connection_get_object(connection));
			JSValue args[] = {
				JS_NewString(context, "remove"),
				object,
			};
			response = JS_Call(context, callback, JS_UNDEFINED, 2, args);
			if (tf_util_report_error(context, response))
			{
				tf_ssb_remove_connections_changed_callback(ssb, _tf_ssb_on_connections_changed_callback, user_data);
			}
			JS_FreeValue(context, args[0]);
			JS_FreeValue(context, object);
		}
		break;
	}
	JS_FreeValue(context, response);
}

static void _tf_ssb_on_broadcasts_changed_callback(tf_ssb_t* ssb, void* user_data)
{
	JSContext* context = tf_ssb_get_context(ssb);
	JSValue callback = JS_MKPTR(JS_TAG_OBJECT, user_data);
	JSValue argv = JS_UNDEFINED;
	JSValue response = JS_Call(context, callback, JS_UNDEFINED, 1, &argv);
	if (tf_util_report_error(context, response))
	{
		tf_ssb_remove_broadcasts_changed_callback(ssb, _tf_ssb_on_broadcasts_changed_callback, user_data);
	}
	JS_FreeValue(context, response);
}

static JSValue _tf_ssb_add_event_listener(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	tf_ssb_t* ssb = JS_GetOpaque(this_val, _tf_ssb_classId);
	const char* event_name = JS_ToCString(context, argv[0]);
	JSValue callback = argv[1];
	JSValue result = JS_UNDEFINED;

	if (!event_name)
	{
		result = JS_ThrowTypeError(context, "Expected argument 1 to be a string event name.");
	}
	else if (!JS_IsFunction(context, callback))
	{
		result = JS_ThrowTypeError(context, "Expected argument 2 to be a function.");
	}
	else
	{
		if (strcmp(event_name, "connections") == 0)
		{
			void* ptr = JS_VALUE_GET_PTR(JS_DupValue(context, callback));
			tf_ssb_add_connections_changed_callback(ssb, _tf_ssb_on_connections_changed_callback, _tf_ssb_cleanup_value, ptr);
		}
		else if (strcmp(event_name, "broadcasts") == 0)
		{
			void* ptr = JS_VALUE_GET_PTR(JS_DupValue(context, callback));
			tf_ssb_add_broadcasts_changed_callback(ssb, _tf_ssb_on_broadcasts_changed_callback, _tf_ssb_cleanup_value, ptr);
		}
		else if (strcmp(event_name, "message") == 0)
		{
			void* ptr = JS_VALUE_GET_PTR(JS_DupValue(context, callback));
			tf_ssb_add_message_added_callback(ssb, _tf_ssb_on_message_added_callback, _tf_ssb_cleanup_value, ptr);
		}
		else if (strcmp(event_name, "blob") == 0)
		{
			void* ptr = JS_VALUE_GET_PTR(JS_DupValue(context, callback));
			tf_ssb_add_blob_stored_callback(ssb, _tf_ssb_on_blob_stored_callback, _tf_ssb_cleanup_value, ptr);
		}
		else if (strcmp(event_name, "blob_want_added") == 0)
		{
			void* ptr = JS_VALUE_GET_PTR(JS_DupValue(context, callback));
			tf_ssb_add_blob_want_added_callback(ssb, _tf_ssb_on_blob_want_added_callback, _tf_ssb_cleanup_value, ptr);
		}
	}

	JS_FreeCString(context, event_name);
	return result;
}

static JSValue _tf_ssb_remove_event_listener(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	tf_ssb_t* ssb = JS_GetOpaque(this_val, _tf_ssb_classId);
	const char* event_name = JS_ToCString(context, argv[0]);
	JSValue callback = argv[1];
	JSValue result = JS_UNDEFINED;

	if (!event_name)
	{
		result = JS_ThrowTypeError(context, "Expected argument 1 to be a string event name.");
	}
	else if (!JS_IsFunction(context, callback))
	{
		result = JS_ThrowTypeError(context, "Expected argument 2 to be a function.");
	}
	else
	{
		if (strcmp(event_name, "connections") == 0)
		{
			void* ptr = JS_VALUE_GET_PTR(JS_DupValue(context, callback));
			tf_ssb_remove_connections_changed_callback(ssb, _tf_ssb_on_connections_changed_callback, ptr);
		}
		else if (strcmp(event_name, "broadcasts") == 0)
		{
			void* ptr = JS_VALUE_GET_PTR(JS_DupValue(context, callback));
			tf_ssb_remove_broadcasts_changed_callback(ssb, _tf_ssb_on_broadcasts_changed_callback, ptr);
		}
		else if (strcmp(event_name, "message") == 0)
		{
			void* ptr = JS_VALUE_GET_PTR(JS_DupValue(context, callback));
			tf_ssb_remove_message_added_callback(ssb, _tf_ssb_on_message_added_callback, ptr);
		}
		else if (strcmp(event_name, "blob") == 0)
		{
			void* ptr = JS_VALUE_GET_PTR(JS_DupValue(context, callback));
			tf_ssb_remove_blob_stored_callback(ssb, _tf_ssb_on_blob_stored_callback, ptr);
		}
		else if (strcmp(event_name, "blob_want_added") == 0)
		{
			void* ptr = JS_VALUE_GET_PTR(JS_DupValue(context, callback));
			tf_ssb_remove_blob_want_added_callback(ssb, _tf_ssb_on_blob_want_added_callback, ptr);
		}
	}

	JS_FreeCString(context, event_name);
	return result;
}

static JSValue _tf_ssb_createTunnel(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	tf_ssb_t* ssb = JS_GetOpaque(this_val, _tf_ssb_classId);
	const char* portal_id = JS_ToCString(context, argv[0]);
	const char* target_id = JS_ToCString(context, argv[1]);

	bool result = portal_id && target_id && tf_ssb_tunnel_create(ssb, portal_id, target_id, 0);

	JS_FreeCString(context, target_id);
	JS_FreeCString(context, portal_id);
	return result ? JS_TRUE : JS_FALSE;
}

typedef struct _following_t
{
	JSContext* context;
	JSValue promise[2];

	tf_ssb_following_t* out_following;

	int depth;
	int ids_count;
	const char* ids[];
} following_t;

static void _tf_ssb_following_work(tf_ssb_t* ssb, void* user_data)
{
	following_t* following = user_data;
	following->out_following = tf_ssb_db_following_deep(ssb, following->ids, following->ids_count, following->depth, false);
}

static void _tf_ssb_following_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	following_t* following = user_data;
	JSContext* context = following->context;
	if (status == 0)
	{
		JSValue object = JS_NewObject(context);
		for (int i = 0; *following->out_following[i].id; i++)
		{
			JSValue entry = JS_NewObject(context);
			JS_SetPropertyStr(context, entry, "of", JS_NewInt32(context, following->out_following[i].following_count));
			JS_SetPropertyStr(context, entry, "ob", JS_NewInt32(context, following->out_following[i].blocking_count));
			JS_SetPropertyStr(context, entry, "if", JS_NewInt32(context, following->out_following[i].followed_by_count));
			JS_SetPropertyStr(context, entry, "ib", JS_NewInt32(context, following->out_following[i].blocked_by_count));
			JS_SetPropertyStr(context, entry, "d", JS_NewInt32(context, following->out_following[i].depth));
			JS_SetPropertyStr(context, object, following->out_following[i].id, entry);
		}
		JSValue result = JS_Call(context, following->promise[0], JS_UNDEFINED, 1, &object);
		tf_util_report_error(context, result);
		JS_FreeValue(context, result);
		JS_FreeValue(context, object);
	}
	else
	{
		char buffer[256];
		uv_strerror_r(status, buffer, sizeof(buffer));
		JSValue message = JS_NewString(context, buffer);
		JSValue result = JS_Call(context, following->promise[1], JS_UNDEFINED, 1, &message);
		tf_util_report_error(context, result);
		JS_FreeValue(context, result);
		JS_FreeValue(context, message);
	}

	JS_FreeValue(context, following->promise[0]);
	JS_FreeValue(context, following->promise[1]);

	for (int i = 0; i < following->ids_count; i++)
	{
		tf_free((void*)following->ids[i]);
	}
	tf_free(following->out_following);
	tf_free(following);
}

static JSValue _tf_ssb_following(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	tf_ssb_t* ssb = JS_GetOpaque(this_val, _tf_ssb_classId);
	int ids_count = tf_util_get_length(context, argv[0]);

	following_t* following = tf_malloc(sizeof(following_t) + sizeof(char*) * ids_count);
	*following = (following_t) {
		.context = context,
	};
	JS_ToInt32(context, &following->depth, argv[1]);

	JSValue result = JS_NewPromiseCapability(context, following->promise);

	for (int i = 0; i < ids_count; i++)
	{
		JSValue id_value = JS_GetPropertyUint32(context, argv[0], i);
		if (!JS_IsUndefined(id_value))
		{
			const char* id_string = JS_ToCString(context, id_value);
			following->ids[following->ids_count++] = tf_strdup(id_string);
			JS_FreeCString(context, id_string);
			JS_FreeValue(context, id_value);
		}
	}

	tf_ssb_run_work(ssb, _tf_ssb_following_work, _tf_ssb_following_after_work, following);
	return result;
}

static JSValue _tf_ssb_sync(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	tf_ssb_t* ssb = JS_GetOpaque(this_val, _tf_ssb_classId);
	tf_ssb_sync_start(ssb);
	return JS_UNDEFINED;
}

typedef struct _set_user_permission_t
{
	tf_ssb_t* ssb;
	JSContext* context;
	const char* user;
	const char* package_owner;
	const char* package_name;
	const char* permission;
	bool allow;
	bool reset;
	bool result;
	JSValue promise[2];
} set_user_permission_t;

static void _tf_ssb_set_user_permission_work(tf_ssb_t* ssb, void* user_data)
{
	set_user_permission_t* work = user_data;

	JSMallocFunctions funcs = { 0 };
	tf_get_js_malloc_functions(&funcs);
	JSRuntime* runtime = JS_NewRuntime2(&funcs, NULL);
	JSContext* context = JS_NewContext(runtime);

	/* XXX: Do this with one DB writer. */
	const char* settings = tf_ssb_db_get_property(ssb, "core", "settings");
	if (settings)
	{
		JSValue settings_value = JS_ParseJSON(context, settings, strlen(settings), NULL);
		JSValue user_permissions = JS_GetPropertyStr(context, settings_value, "userPermissions");
		if (JS_IsUndefined(user_permissions))
		{
			user_permissions = JS_NewObject(context);
			JS_SetPropertyStr(context, settings_value, "userPermissions", JS_DupValue(context, user_permissions));
		}
		JSValue user = JS_GetPropertyStr(context, user_permissions, work->user);
		if (JS_IsUndefined(user))
		{
			user = JS_NewObject(context);
			JS_SetPropertyStr(context, user_permissions, work->user, JS_DupValue(context, user));
		}
		JSValue package_owner = JS_GetPropertyStr(context, user, work->package_owner);
		if (JS_IsUndefined(package_owner))
		{
			package_owner = JS_NewObject(context);
			JS_SetPropertyStr(context, user, work->package_owner, JS_DupValue(context, package_owner));
		}
		JSValue package_name = JS_GetPropertyStr(context, package_owner, work->package_name);
		if (JS_IsUndefined(package_name))
		{
			package_name = JS_NewObject(context);
			JS_SetPropertyStr(context, package_owner, work->package_name, JS_DupValue(context, package_name));
		}
		JSValue permission = JS_GetPropertyStr(context, package_name, work->permission);
		if (work->reset)
		{
			JSAtom atom = JS_NewAtom(context, work->permission);
			JS_DeleteProperty(context, package_name, atom, 0);
			JS_FreeAtom(context, atom);
		}
		else
		{
			JS_SetPropertyStr(context, package_name, work->permission, JS_NewBool(context, work->allow));
		}
		JSValue settings_json = JS_JSONStringify(context, settings_value, JS_NULL, JS_NULL);
		const char* settings_string = JS_ToCString(context, settings_json);
		work->result = tf_ssb_db_set_property(ssb, "core", "settings", settings_string);
		JS_FreeCString(context, settings_string);
		JS_FreeValue(context, settings_json);
		JS_FreeValue(context, permission);
		JS_FreeValue(context, package_owner);
		JS_FreeValue(context, package_name);
		JS_FreeValue(context, user);
		JS_FreeValue(context, user_permissions);
		JS_FreeValue(context, settings_value);
		tf_free((void*)settings);
	}

	JS_FreeContext(context);
	JS_FreeRuntime(runtime);
}

static void _tf_ssb_set_user_permission_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	set_user_permission_t* work = user_data;
	JSContext* context = work->context;
	JS_FreeCString(context, work->user);
	JS_FreeCString(context, work->package_owner);
	JS_FreeCString(context, work->package_name);
	JS_FreeCString(context, work->permission);
	JSValue error = JS_Call(context, work->result ? work->promise[0] : work->promise[0], JS_UNDEFINED, 0, NULL);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	tf_free(work);
}

static JSValue _tf_ssb_set_user_permission(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	set_user_permission_t* set = tf_malloc(sizeof(set_user_permission_t));
	*set = (set_user_permission_t) {
		.ssb = JS_GetOpaque(this_val, _tf_ssb_classId),
		.context = context,
		.user = JS_ToCString(context, argv[0]),
		.package_owner = JS_ToCString(context, argv[1]),
		.package_name = JS_ToCString(context, argv[2]),
		.permission = JS_ToCString(context, argv[3]),
		.allow = JS_ToBool(context, argv[4]),
		.reset = JS_IsUndefined(argv[4]),
	};
	JSValue result = JS_NewPromiseCapability(context, set->promise);
	tf_ssb_run_work(set->ssb, _tf_ssb_set_user_permission_work, _tf_ssb_set_user_permission_after_work, set);
	return result;
}

static JSValue _tf_ssb_port(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	tf_ssb_t* ssb = JS_GetOpaque(this_val, _tf_ssb_classId);
	return JS_NewInt32(context, tf_ssb_server_get_port(ssb));
}

void tf_ssb_register(JSContext* context, tf_ssb_t* ssb)
{
	JS_NewClassID(&_tf_ssb_classId);
	JSClassDef def = {
		.class_name = "ssb",
	};
	if (JS_NewClass(JS_GetRuntime(context), _tf_ssb_classId, &def) != 0)
	{
		fprintf(stderr, "Failed to register ssb.\n");
	}

	JSValue global = JS_GetGlobalObject(context);
	JSValue object = JS_NewObjectClass(context, _tf_ssb_classId);
	JS_SetPropertyStr(context, global, "ssb", object);
	JS_SetOpaque(object, ssb);

	JSValue object_internal = JS_NewObjectClass(context, _tf_ssb_classId);
	JS_SetPropertyStr(context, global, "ssb_internal", object_internal);
	JS_SetOpaque(object_internal, ssb);

	/* Requires an identity. */
	JS_SetPropertyStr(context, object, "createIdentity", JS_NewCFunction(context, _tf_ssb_createIdentity, "createIdentity", 1));
	JS_SetPropertyStr(context, object, "setUserPermission", JS_NewCFunction(context, _tf_ssb_set_user_permission, "setUserPermission", 5));

	/* Does not require an identity. */
	JS_SetPropertyStr(context, object, "getServerIdentity", JS_NewCFunction(context, _tf_ssb_getServerIdentity, "getServerIdentity", 0));
	JS_SetPropertyStr(context, object, "blobGet", JS_NewCFunction(context, _tf_ssb_blobGet, "blobGet", 1));
	JS_SetPropertyStr(context, object, "connections", JS_NewCFunction(context, _tf_ssb_connections, "connections", 0));
	JS_SetPropertyStr(context, object, "storedConnections", JS_NewCFunction(context, _tf_ssb_storedConnections, "storedConnections", 0));
	JS_SetPropertyStr(context, object, "getConnection", JS_NewCFunction(context, _tf_ssb_getConnection, "getConnection", 1));
	JS_SetPropertyStr(context, object, "closeConnection", JS_NewCFunction(context, _tf_ssb_closeConnection, "closeConnection", 1));
	JS_SetPropertyStr(context, object, "forgetStoredConnection", JS_NewCFunction(context, _tf_ssb_forgetStoredConnection, "forgetStoredConnection", 1));
	JS_SetPropertyStr(context, object, "sqlAsync", JS_NewCFunction(context, _tf_ssb_sqlAsync, "sqlAsync", 3));
	JS_SetPropertyStr(context, object, "getBroadcasts", JS_NewCFunction(context, _tf_ssb_getBroadcasts, "getBroadcasts", 0));
	JS_SetPropertyStr(context, object, "connect", JS_NewCFunction(context, _tf_ssb_connect, "connect", 1));
	JS_SetPropertyStr(context, object, "createTunnel", JS_NewCFunction(context, _tf_ssb_createTunnel, "createTunnel", 3));
	JS_SetPropertyStr(context, object, "following", JS_NewCFunction(context, _tf_ssb_following, "following", 2));
	JS_SetPropertyStr(context, object, "sync", JS_NewCFunction(context, _tf_ssb_sync, "sync", 0));
	JS_SetPropertyStr(context, object, "port", JS_NewCFunction(context, _tf_ssb_port, "port", 0));
	/* Write. */
	JS_SetPropertyStr(context, object, "storeMessage", JS_NewCFunction(context, _tf_ssb_storeMessage, "storeMessage", 1));
	JS_SetPropertyStr(context, object, "blobStore", JS_NewCFunction(context, _tf_ssb_blobStore, "blobStore", 1));

	/* Trusted only. */
	JS_SetPropertyStr(context, object_internal, "getIdentityInfo", JS_NewCFunction(context, _tf_ssb_getIdentityInfo, "getIdentityInfo", 3));
	JS_SetPropertyStr(context, object_internal, "addEventListener", JS_NewCFunction(context, _tf_ssb_add_event_listener, "addEventListener", 2));
	JS_SetPropertyStr(context, object_internal, "removeEventListener", JS_NewCFunction(context, _tf_ssb_remove_event_listener, "removeEventListener", 2));

	JS_FreeValue(context, global);
}
