/*
 * SHA1 internal definitions
 * Copyright (c) 2003-2005, Jouni Malinen <j@w1.fi>
 *
 * This software may be distributed under the terms of the BSD license.
 * See README for more details.
 */

/**
** \defgroup sha1 SHA1
** SHA1 API.
** Adapted from
** https://web.mit.edu/freebsd/head/contrib/wpa/src/crypto/sha1_i.h by Cory
** McWilliams 2025-09-28.
** @{
*/

#ifndef SHA1_I_H
#define SHA1_I_H

#include <inttypes.h>

/**
** SHA1 context struct.
*/
struct SHA1Context
{
	/** SHA1 state. */
	uint32_t state[5];
	/** SHA1 count. */
	uint32_t count[2];
	/** SHA1 buffer. */
	unsigned char buffer[64];
};

/**
** SHA1 context.
*/
typedef struct SHA1Context SHA1_CTX;

/**
** Initialize a SHA1 context.
** @param context The context.
*/
void SHA1Init(struct SHA1Context* context);

/**
** Calculate an ongoing hash for a block of data.
** @param context The SHA1 context.
** @param data The data to hash.
** @param len The length of data.
*/
void SHA1Update(struct SHA1Context* context, const void* data, uint32_t len);

/**
** Calculate the final hash digest.
** @param digest Populated with the digest.
** @param context The SHA1 context.
*/
void SHA1Final(unsigned char digest[20], struct SHA1Context* context);

/**
** Perform a SHA1 transformation.
** @param state The SHA1 state.
** @param buffer The data.
*/
void SHA1Transform(uint32_t state[5], const unsigned char buffer[64]);

#endif /* SHA1_I_H */

/** @} */
