#include "httpd.js.h"

#include "file.js.h"
#include "http.h"
#include "mem.h"
#include "ssb.db.h"
#include "ssb.h"
#include "task.h"
#include "util.js.h"

#include "ow-crypt.h"
#include "sodium/utils.h"

#include <inttypes.h>
#include <stdlib.h>

#if !defined(__APPLE__) && !defined(__OpenBSD__) && !defined(_WIN32)
#include <alloca.h>
#endif

typedef struct _login_request_t
{
	tf_http_request_t* request;
	const char* name;
	const char* error;
	const char* settings;
	const char* code_of_conduct;
	bool have_administrator;
	bool session_is_new;

	char location_header[1024];
	const char* set_cookie_header;

	int pending;
} login_request_t;

const char* tf_httpd_make_set_session_cookie_header(tf_http_request_t* request, const char* session_cookie)
{
	const char* k_pattern = "session=%s; path=/; Max-Age=%" PRId64 "; %sSameSite=Strict; HttpOnly";
	int length = session_cookie ? snprintf(NULL, 0, k_pattern, session_cookie, k_httpd_auth_refresh_interval, "") : 0;
	char* cookie = length ? tf_malloc(length + 1) : NULL;
	if (cookie)
	{
		snprintf(cookie, length + 1, k_pattern, session_cookie, k_httpd_auth_refresh_interval, "");
	}
	return cookie;
}

static void _login_release(login_request_t* login)
{
	int ref_count = --login->pending;
	if (ref_count == 0)
	{
		tf_free((void*)login->name);
		tf_free((void*)login->code_of_conduct);
		tf_free((void*)login->set_cookie_header);
		tf_free(login);
	}
}

static void _httpd_endpoint_login_file_read_callback(tf_task_t* task, const char* path, int result, const void* data, void* user_data)
{
	login_request_t* login = user_data;
	tf_http_request_t* request = login->request;
	if (result >= 0)
	{
		const char* headers[] = {
			"Content-Type",
			"text/html; charset=utf-8",
			"Set-Cookie",
			login->set_cookie_header ? login->set_cookie_header : "",
		};
		const char* replace_me = "$AUTH_DATA";
		const char* auth = strstr(data, replace_me);
		if (auth)
		{
			JSContext* context = tf_task_get_context(task);
			JSValue object = JS_NewObject(context);
			JS_SetPropertyStr(context, object, "session_is_new", JS_NewBool(context, login->session_is_new));
			JS_SetPropertyStr(context, object, "name", login->name ? JS_NewString(context, login->name) : JS_UNDEFINED);
			JS_SetPropertyStr(context, object, "error", login->error ? JS_NewString(context, login->error) : JS_UNDEFINED);
			JS_SetPropertyStr(context, object, "code_of_conduct", login->code_of_conduct ? JS_NewString(context, login->code_of_conduct) : JS_UNDEFINED);
			JS_SetPropertyStr(context, object, "have_administrator", JS_NewBool(context, login->have_administrator));
			JSValue object_json = JS_JSONStringify(context, object, JS_NULL, JS_NULL);
			size_t json_length = 0;
			const char* json = JS_ToCStringLen(context, &json_length, object_json);

			char* copy = tf_malloc(result + json_length);
			int replace_start = (auth - (const char*)data);
			int replace_end = (auth - (const char*)data) + (int)strlen(replace_me);
			memcpy(copy, data, replace_start);
			memcpy(copy + replace_start, json, json_length);
			memcpy(copy + replace_start + json_length, ((const char*)data) + replace_end, result - replace_end);
			tf_http_respond(request, 200, headers, tf_countof(headers) / 2, copy, replace_start + json_length + (result - replace_end));
			tf_free(copy);

			JS_FreeCString(context, json);
			JS_FreeValue(context, object_json);
			JS_FreeValue(context, object);
		}
		else
		{
			tf_http_respond(request, 200, headers, tf_countof(headers) / 2, data, result);
		}
	}
	else
	{
		const char* k_payload = tf_http_status_text(404);
		tf_http_respond(request, 404, NULL, 0, k_payload, strlen(k_payload));
	}
	tf_http_request_unref(request);
	_login_release(login);
}

static bool _session_is_authenticated_as_user(JSContext* context, JSValue session)
{
	bool result = false;
	JSValue user = JS_GetPropertyStr(context, session, "name");
	const char* user_string = JS_ToCString(context, user);
	result = user_string && strcmp(user_string, "guest") != 0;
	JS_FreeCString(context, user_string);
	JS_FreeValue(context, user);
	return result;
}

static bool _make_administrator_if_first(tf_ssb_t* ssb, JSContext* context, const char* account_name_copy, bool may_become_first_admin)
{
	const char* settings = tf_ssb_db_get_property(ssb, "core", "settings");
	JSValue settings_value = settings && *settings ? JS_ParseJSON(context, settings, strlen(settings), NULL) : JS_UNDEFINED;
	if (JS_IsUndefined(settings_value))
	{
		settings_value = JS_NewObject(context);
	}

	bool have_administrator = false;
	JSValue permissions = JS_GetPropertyStr(context, settings_value, "permissions");

	JSPropertyEnum* ptab = NULL;
	uint32_t plen = 0;
	JS_GetOwnPropertyNames(context, &ptab, &plen, permissions, JS_GPN_STRING_MASK);
	for (int i = 0; i < (int)plen; i++)
	{
		JSPropertyDescriptor desc = { 0 };
		if (JS_GetOwnProperty(context, &desc, permissions, ptab[i].atom) == 1)
		{
			int permission_length = tf_util_get_length(context, desc.value);
			for (int i = 0; i < permission_length; i++)
			{
				JSValue entry = JS_GetPropertyUint32(context, desc.value, i);
				const char* permission = JS_ToCString(context, entry);
				if (permission && strcmp(permission, "administration") == 0)
				{
					have_administrator = true;
				}
				JS_FreeCString(context, permission);
				JS_FreeValue(context, entry);
			}
			JS_FreeValue(context, desc.setter);
			JS_FreeValue(context, desc.getter);
			JS_FreeValue(context, desc.value);
		}
	}
	for (uint32_t i = 0; i < plen; ++i)
	{
		JS_FreeAtom(context, ptab[i].atom);
	}
	js_free(context, ptab);

	if (!have_administrator && may_become_first_admin)
	{
		if (JS_IsUndefined(permissions))
		{
			permissions = JS_NewObject(context);
			JS_SetPropertyStr(context, settings_value, "permissions", JS_DupValue(context, permissions));
		}
		JSValue user = JS_GetPropertyStr(context, permissions, account_name_copy);
		if (JS_IsUndefined(user))
		{
			user = JS_NewArray(context);
			JS_SetPropertyStr(context, permissions, account_name_copy, JS_DupValue(context, user));
		}
		JS_SetPropertyUint32(context, user, tf_util_get_length(context, user), JS_NewString(context, "administration"));
		JS_FreeValue(context, user);

		JSValue settings_json = JS_JSONStringify(context, settings_value, JS_NULL, JS_NULL);
		const char* settings_string = JS_ToCString(context, settings_json);
		tf_ssb_db_set_property(ssb, "core", "settings", settings_string);
		JS_FreeCString(context, settings_string);
		JS_FreeValue(context, settings_json);
	}

	JS_FreeValue(context, permissions);
	JS_FreeValue(context, settings_value);
	tf_free((void*)settings);
	return have_administrator;
}

static bool _verify_password(const char* password, const char* hash)
{
	char buffer[7 + 22 + 31 + 1];
	const char* out_hash = crypt_rn(password, hash, buffer, sizeof(buffer));
	return out_hash && strcmp(hash, out_hash) == 0;
}

static void _httpd_endpoint_login_work(tf_ssb_t* ssb, void* user_data)
{
	login_request_t* login = user_data;
	tf_http_request_t* request = login->request;

	JSMallocFunctions funcs = { 0 };
	tf_get_js_malloc_functions(&funcs);
	JSRuntime* runtime = JS_NewRuntime2(&funcs, NULL);
	JSContext* context = JS_NewContext(runtime);

	const char* session = tf_http_get_cookie(tf_http_request_get_header(request, "cookie"), "session");
	const char** form_data = tf_httpd_form_data_decode(request->query, request->query ? strlen(request->query) : 0);
	const char* account_name_copy = NULL;
	JSValue jwt = tf_httpd_authenticate_jwt(ssb, context, session);

	if (_session_is_authenticated_as_user(context, jwt))
	{
		const char* return_url = tf_httpd_form_data_get(form_data, "return");
		if (return_url)
		{
			tf_string_set(login->location_header, sizeof(login->location_header), return_url);
		}
		else
		{
			snprintf(login->location_header, sizeof(login->location_header), "%s%s/", "http://", tf_http_request_get_header(request, "host"));
		}
		goto done;
	}

	const char* send_session = tf_strdup(session);
	bool session_is_new = false;
	const char* login_error = NULL;
	bool may_become_first_admin = false;
	if (strcmp(request->method, "POST") == 0)
	{
		session_is_new = true;
		const char** post_form_data = tf_httpd_form_data_decode(request->body, request->content_length);
		const char* submit = tf_httpd_form_data_get(post_form_data, "submit");
		if (submit && strcmp(submit, "Login") == 0)
		{
			const char* account_name = tf_httpd_form_data_get(post_form_data, "name");
			account_name_copy = tf_strdup(account_name);
			const char* password = tf_httpd_form_data_get(post_form_data, "password");
			const char* new_password = tf_httpd_form_data_get(post_form_data, "new_password");
			const char* confirm = tf_httpd_form_data_get(post_form_data, "confirm");
			const char* change = tf_httpd_form_data_get(post_form_data, "change");
			const char* form_register = tf_httpd_form_data_get(post_form_data, "register");
			char account_passwd[256] = { 0 };
			bool have_account = tf_ssb_db_get_account_password_hash(ssb, tf_httpd_form_data_get(post_form_data, "name"), account_passwd, sizeof(account_passwd));

			if (form_register && strcmp(form_register, "1") == 0)
			{
				bool registered = false;
				if (!tf_httpd_is_name_valid(account_name))
				{
					login_error = "Invalid username.  Usernames must contain only letters from the English alphabet and digits and must start with a letter.";
				}
				else
				{
					if (!have_account && tf_httpd_is_name_valid(account_name) && password && confirm && strcmp(password, confirm) == 0)
					{
						sqlite3* db = tf_ssb_acquire_db_writer(ssb);
						registered = tf_ssb_db_register_account(tf_ssb_get_loop(ssb), db, context, account_name, password);
						tf_ssb_release_db_writer(ssb, db);
						if (registered)
						{
							tf_free((void*)send_session);
							send_session = tf_httpd_make_session_jwt(context, ssb, account_name);
							may_become_first_admin = true;
						}
					}
				}
				if (!registered && !login_error)
				{
					login_error = "Error registering account.";
				}
			}
			else if (change && strcmp(change, "1") == 0)
			{
				bool set = false;
				if (have_account && tf_httpd_is_name_valid(account_name) && new_password && confirm && strcmp(new_password, confirm) == 0 &&
					_verify_password(password, account_passwd))
				{
					sqlite3* db = tf_ssb_acquire_db_writer(ssb);
					set = tf_ssb_db_set_account_password(tf_ssb_get_loop(ssb), db, context, account_name, new_password);
					tf_ssb_release_db_writer(ssb, db);
					if (set)
					{
						tf_free((void*)send_session);
						send_session = tf_httpd_make_session_jwt(context, ssb, account_name);
					}
				}
				if (!set)
				{
					login_error = "Error changing password.";
				}
			}
			else
			{
				if (have_account && *account_passwd && _verify_password(password, account_passwd))
				{
					tf_free((void*)send_session);
					send_session = tf_httpd_make_session_jwt(context, ssb, account_name);
					may_become_first_admin = true;
				}
				else
				{
					login_error = "Invalid username or password.";
				}
			}
		}
		else
		{
			tf_free((void*)send_session);
			send_session = tf_httpd_make_session_jwt(context, ssb, "guest");
		}
		tf_free(post_form_data);
	}

	bool have_administrator = _make_administrator_if_first(ssb, context, account_name_copy, may_become_first_admin);

	if (session_is_new && tf_httpd_form_data_get(form_data, "return") && !login_error)
	{
		const char* return_url = tf_httpd_form_data_get(form_data, "return");
		if (return_url)
		{
			tf_string_set(login->location_header, sizeof(login->location_header), return_url);
		}
		else
		{
			snprintf(login->location_header, sizeof(login->location_header), "%s%s/", "http://", tf_http_request_get_header(request, "host"));
		}
		login->set_cookie_header = tf_httpd_make_set_session_cookie_header(request, send_session);
		tf_free((void*)send_session);
	}
	else
	{

		login->name = account_name_copy;
		login->error = login_error;
		login->set_cookie_header = tf_httpd_make_set_session_cookie_header(request, send_session);
		tf_free((void*)send_session);
		login->session_is_new = session_is_new;
		login->have_administrator = have_administrator;
		login->settings = tf_ssb_db_get_property(ssb, "core", "settings");

		if (login->settings)
		{
			JSValue settings_value = JS_ParseJSON(context, login->settings, strlen(login->settings), NULL);
			JSValue code_of_conduct_value = JS_GetPropertyStr(context, settings_value, "code_of_conduct");
			const char* code_of_conduct = JS_IsString(code_of_conduct_value) ? JS_ToCString(context, code_of_conduct_value) : NULL;
			const char* result = tf_strdup(code_of_conduct);
			JS_FreeCString(context, code_of_conduct);
			JS_FreeValue(context, code_of_conduct_value);
			JS_FreeValue(context, settings_value);
			tf_free((void*)login->settings);
			login->settings = NULL;
			login->code_of_conduct = result;
		}

		login->pending++;
		tf_http_request_ref(request);
		tf_file_read(login->request->user_data, "core/auth.html", _httpd_endpoint_login_file_read_callback, login);

		account_name_copy = NULL;
	}

done:
	tf_free((void*)session);
	tf_free(form_data);
	tf_free((void*)account_name_copy);
	JS_FreeValue(context, jwt);

	JS_FreeContext(context);
	JS_FreeRuntime(runtime);
}

static void _httpd_endpoint_login_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	login_request_t* login = user_data;
	tf_http_request_t* request = login->request;
	if (login->pending == 1)
	{
		if (*login->location_header)
		{
			const char* headers[] = {
				"Location",
				login->location_header,
				"Set-Cookie",
				login->set_cookie_header ? login->set_cookie_header : "",
			};
			tf_http_respond(request, 303, headers, tf_countof(headers) / 2, NULL, 0);
		}
	}
	tf_http_request_unref(request);
	_login_release(login);
}

void tf_httpd_endpoint_login(tf_http_request_t* request)
{
	tf_task_t* task = request->user_data;
	tf_http_request_ref(request);

	tf_ssb_t* ssb = tf_task_get_ssb(task);
	login_request_t* login = tf_malloc(sizeof(login_request_t));
	*login = (login_request_t) {
		.request = request,
	};
	login->pending++;
	tf_ssb_run_work(ssb, _httpd_endpoint_login_work, _httpd_endpoint_login_after_work, login);
}

void tf_httpd_endpoint_logout(tf_http_request_t* request)
{
	const char* k_set_cookie = "session=; path=/; SameSite=Strict; expires=Thu, 01 Jan 1970 00:00:00 GMT; HttpOnly";
	const char* k_location_format = "/login%s%s";
	int length = snprintf(NULL, 0, k_location_format, request->query ? "?" : "", request->query);
	char* location = alloca(length + 1);
	snprintf(location, length + 1, k_location_format, request->query ? "?" : "", request->query ? request->query : "");
	const char* headers[] = {
		"Set-Cookie",
		k_set_cookie,
		"Location",
		location,
	};
	tf_http_respond(request, 303, headers, tf_countof(headers) / 2, NULL, 0);
}

typedef struct _auto_login_t
{
	tf_http_request_t* request;
	bool autologin;
	const char* users;
} auto_login_t;

static void _httpd_auto_login_work(tf_ssb_t* ssb, void* user_data)
{
	auto_login_t* request = user_data;
	sqlite3* db = tf_ssb_acquire_db_reader(ssb);
	tf_ssb_db_get_global_setting_bool(db, "autologin", &request->autologin);
	tf_ssb_release_db_reader(ssb, db);

	if (request->autologin)
	{
		request->users = tf_ssb_db_get_property(ssb, "auth", "users");
		if (request->users && strcmp(request->users, "[]") == 0)
		{
			tf_free((void*)request->users);
			request->users = NULL;
		}

		if (!request->users)
		{
			JSMallocFunctions funcs = { 0 };
			tf_get_js_malloc_functions(&funcs);
			JSRuntime* runtime = JS_NewRuntime2(&funcs, NULL);
			JSContext* context = JS_NewContext(runtime);
			static const char* k_account_name = "mobile";
			sqlite3* db = tf_ssb_acquire_db_writer(ssb);
			bool registered = tf_ssb_db_register_account(tf_ssb_get_loop(ssb), db, context, k_account_name, k_account_name);
			tf_ssb_release_db_writer(ssb, db);
			if (registered)
			{
				_make_administrator_if_first(ssb, context, k_account_name, true);
			}
			JS_FreeContext(context);
			JS_FreeRuntime(runtime);

			request->users = tf_ssb_db_get_property(ssb, "auth", "users");
		}
	}
}

static void _httpd_auto_login_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	auto_login_t* work = user_data;
	JSContext* context = tf_ssb_get_context(ssb);
	const char* session_token = NULL;
	if (!work->autologin)
	{
		const char* k_payload = tf_http_status_text(404);
		tf_http_respond(work->request, 404, NULL, 0, k_payload, strlen(k_payload));
	}
	else
	{
		if (work->users)
		{
			JSValue json = JS_ParseJSON(context, work->users, strlen(work->users), NULL);
			JSValue user = JS_GetPropertyUint32(context, json, 0);
			const char* user_string = JS_ToCString(context, user);
			session_token = tf_httpd_make_session_jwt(context, ssb, user_string);
			JS_FreeCString(context, user_string);
			JS_FreeValue(context, user);
			JS_FreeValue(context, json);
		}
		if (session_token)
		{
			const char* cookie = tf_httpd_make_set_session_cookie_header(work->request, session_token);
			tf_free((void*)session_token);
			const char* headers[] = {
				"Set-Cookie",
				cookie,
				"Location",
				"/",
			};
			tf_http_respond(work->request, 303, headers, tf_countof(headers) / 2, NULL, 0);
			tf_free((void*)cookie);
		}
		else
		{
			const char* headers[] = {
				"Location",
				"/",
			};
			tf_http_respond(work->request, 303, headers, tf_countof(headers) / 2, NULL, 0);
		}
	}
	tf_http_request_unref(work->request);
	tf_free((void*)work->users);
	tf_free(work);
}

void tf_httpd_endpoint_login_auto(tf_http_request_t* request)
{
	tf_task_t* task = request->user_data;
	tf_http_request_ref(request);
	tf_ssb_t* ssb = tf_task_get_ssb(task);

	auto_login_t* work = tf_malloc(sizeof(auto_login_t));
	*work = (auto_login_t) { .request = request };
	tf_ssb_run_work(ssb, _httpd_auto_login_work, _httpd_auto_login_after_work, work);
}
