#include "api.js.h"

#include "log.h"
#include "mem.h"
#include "ssb.db.h"
#include "ssb.h"
#include "task.h"
#include "trace.h"
#include "util.js.h"

#include "quickjs.h"
#include "sodium/crypto_box.h"
#include "sodium/crypto_scalarmult.h"
#include "sodium/crypto_scalarmult_curve25519.h"
#include "sodium/crypto_scalarmult_ed25519.h"
#include "sodium/crypto_secretbox.h"
#include "sodium/crypto_sign.h"
#include "sodium/randombytes.h"
#include "sqlite3.h"

#include <assert.h>
#include <stdlib.h>

#if !defined(__APPLE__) && !defined(__OpenBSD__) && !defined(_WIN32)
#include <alloca.h>
#endif

static JSClassID s_permission_test_class_id;

typedef struct _app_path_pair_t
{
	const char* app;
	const char* path;
} app_path_pair_t;

typedef struct _get_apps_t
{
	app_path_pair_t* apps;
	int count;
	JSContext* context;
	JSValue promise[2];
	char user[];
} get_apps_t;

static void _tf_api_core_apps_work(tf_ssb_t* ssb, void* user_data)
{
	get_apps_t* work = user_data;

	JSMallocFunctions funcs = { 0 };
	tf_get_js_malloc_functions(&funcs);
	JSRuntime* runtime = JS_NewRuntime2(&funcs, NULL);
	JSContext* context = JS_NewContext(runtime);

	const char* apps = tf_ssb_db_get_property(ssb, work->user, "apps");
	if (apps)
	{
		JSValue apps_array = JS_ParseJSON(context, apps, strlen(apps), NULL);
		if (JS_IsArray(context, apps_array))
		{
			int length = tf_util_get_length(context, apps_array);
			for (int i = 0; i < length; i++)
			{
				JSValue name = JS_GetPropertyUint32(context, apps_array, i);
				const char* name_string = JS_ToCString(context, name);
				if (name_string)
				{
					work->apps = tf_resize_vec(work->apps, sizeof(app_path_pair_t) * (work->count + 1));
					work->apps[work->count].app = tf_strdup(name_string);
					size_t size = strlen("path:") + strlen(name_string) + 1;
					char* path_key = tf_malloc(size);
					snprintf(path_key, size, "path:%s", name_string);
					work->apps[work->count].path = tf_ssb_db_get_property(ssb, work->user, path_key);
					tf_free(path_key);
					work->count++;
				}
				JS_FreeCString(context, name_string);
				JS_FreeValue(context, name);
			}
		}
		JS_FreeValue(context, apps_array);
	}
	tf_free((void*)apps);

	JS_FreeContext(context);
	JS_FreeRuntime(runtime);
}

static void _tf_api_core_apps_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	get_apps_t* work = user_data;
	JSContext* context = work->context;
	JSValue result = JS_NewObject(context);
	for (int i = 0; i < work->count; i++)
	{
		JS_SetPropertyStr(context, result, work->apps[i].app, JS_NewString(context, work->apps[i].path ? work->apps[i].path : ""));
	}
	JSValue error = JS_Call(context, work->promise[0], JS_UNDEFINED, 1, &result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, result);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	for (int i = 0; i < work->count; i++)
	{
		tf_free((void*)work->apps[i].app);
		tf_free((void*)work->apps[i].path);
	}
	tf_free(work->apps);
	tf_free(work);
}

static const char* _tf_ssb_get_process_credentials_session_name(JSContext* context, JSValue process)
{
	JSValue credentials = JS_IsObject(process) ? JS_GetPropertyStr(context, process, "credentials") : JS_UNDEFINED;
	JSValue session = JS_IsObject(credentials) ? JS_GetPropertyStr(context, credentials, "session") : JS_UNDEFINED;
	JSValue name_value = JS_IsObject(session) ? JS_GetPropertyStr(context, session, "name") : JS_UNDEFINED;
	const char* result = JS_IsString(name_value) ? JS_ToCString(context, name_value) : NULL;
	JS_FreeValue(context, name_value);
	JS_FreeValue(context, session);
	JS_FreeValue(context, credentials);
	return result;
}

static JSValue _tf_api_core_apps(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	JSValue result = JS_UNDEFINED;
	JSValue user = argv[0];
	JSValue process = data[0];
	const char* user_string = JS_IsString(user) ? JS_ToCString(context, user) : NULL;
	const char* session_name_string = _tf_ssb_get_process_credentials_session_name(context, process);

	if (user_string && session_name_string && strcmp(user_string, session_name_string) && strcmp(user_string, "core"))
	{
		JS_FreeCString(context, user_string);
		user_string = NULL;
	}
	else if (!user_string)
	{
		user_string = session_name_string;
		session_name_string = NULL;
	}
	JS_FreeCString(context, session_name_string);

	if (user_string)
	{
		get_apps_t* work = tf_malloc(sizeof(get_apps_t) + strlen(user_string) + 1);
		*work = (get_apps_t) {
			.context = context,
		};
		memcpy(work->user, user_string, strlen(user_string) + 1);
		result = JS_NewPromiseCapability(context, work->promise);

		tf_task_t* task = tf_task_get(context);
		tf_ssb_t* ssb = tf_task_get_ssb(task);
		tf_ssb_run_work(ssb, _tf_api_core_apps_work, _tf_api_core_apps_after_work, work);
	}
	else
	{
		result = JS_NewObject(context);
	}
	JS_FreeCString(context, user_string);
	return result;
}

typedef struct _users_t
{
	const char* users;
	JSContext* context;
	JSValue promise[2];
} users_t;

static void _tf_api_core_users_work(tf_ssb_t* ssb, void* user_data)
{
	users_t* work = user_data;
	work->users = tf_ssb_db_get_property(ssb, "auth", "users");
}

static void _tf_api_core_users_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	users_t* work = user_data;
	JSContext* context = work->context;
	JSValue result = JS_UNDEFINED;
	if (work->users)
	{
		result = JS_ParseJSON(context, work->users, strlen(work->users), NULL);
		tf_free((void*)work->users);
	}
	if (JS_IsUndefined(result))
	{
		result = JS_NewArray(context);
	}
	JSValue error = JS_Call(context, JS_IsArray(context, result) ? work->promise[0] : work->promise[1], JS_UNDEFINED, 1, &result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, result);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	tf_free(work);
}

static JSValue _tf_api_core_users(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	users_t* work = tf_malloc(sizeof(users_t));
	*work = (users_t) {
		.context = context,
	};
	JSValue result = JS_NewPromiseCapability(context, work->promise);
	tf_ssb_run_work(ssb, _tf_api_core_users_work, _tf_api_core_users_after_work, work);
	return result;
}

static JSValue _tf_api_core_register(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	JSValue event_name = argv[0];
	JSValue handler = argv[1];
	JSValue process = data[0];
	JSValue event_handlers = JS_GetPropertyStr(context, process, "eventHandlers");
	JSAtom atom = JS_ValueToAtom(context, event_name);
	JSValue array = JS_GetProperty(context, event_handlers, atom);
	if (!JS_IsArray(context, array))
	{
		JS_FreeValue(context, array);
		array = JS_NewArray(context);
		JS_SetProperty(context, event_handlers, atom, JS_DupValue(context, array));
	}
	JS_SetPropertyUint32(context, array, tf_util_get_length(context, array), JS_DupValue(context, handler));
	JS_FreeValue(context, array);
	JS_FreeAtom(context, atom);
	JS_FreeValue(context, event_handlers);
	return JS_UNDEFINED;
}

static JSValue _tf_api_core_unregister(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	JSValue event_name = argv[0];
	JSValue handler = argv[1];
	JSValue process = data[0];
	JSValue event_handlers = JS_GetPropertyStr(context, process, "eventHandlers");
	JSAtom atom = JS_ValueToAtom(context, event_name);
	JSValue array = JS_GetProperty(context, event_handlers, atom);

	if (JS_IsArray(context, array))
	{
		JSValue index_of = JS_GetPropertyStr(context, array, "indexOf");

		JSValue index = JS_Call(context, index_of, array, 1, &handler);
		int int_index = -1;
		JS_ToInt32(context, &int_index, index);
		if (int_index != -1)
		{
			JSValue splice = JS_GetPropertyStr(context, array, "splice");
			JSValue splice_args[] = {
				index,
				JS_NewInt32(context, 1),
			};
			JSValue result = JS_Call(context, splice, array, 2, splice_args);
			JS_FreeValue(context, result);
			JS_FreeValue(context, splice);
		}
		JS_FreeValue(context, index);
		JS_FreeValue(context, index_of);

		if (tf_util_get_length(context, array) == 0)
		{
			JS_DeleteProperty(context, event_handlers, atom, 0);
		}
	}

	JS_FreeValue(context, array);
	JS_FreeAtom(context, atom);
	JS_FreeValue(context, event_handlers);
	return JS_UNDEFINED;
}

typedef struct _permissions_for_user_t
{
	const char* user;
	const char* settings;
	JSContext* context;
	JSValue promise[2];
} permissions_for_user_t;

static void _tf_api_core_permissions_for_user_work(tf_ssb_t* ssb, void* user_data)
{
	permissions_for_user_t* work = user_data;
	work->settings = tf_ssb_db_get_property(ssb, "core", "settings");
}

static void _tf_api_core_permissions_for_user_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	permissions_for_user_t* work = user_data;
	JSContext* context = work->context;
	JSValue result = JS_UNDEFINED;
	if (work->settings)
	{
		JSValue json = JS_ParseJSON(context, work->settings, strlen(work->settings), NULL);
		if (JS_IsObject(json))
		{
			JSValue permissions = JS_GetPropertyStr(context, json, "permissions");
			if (JS_IsObject(permissions))
			{
				result = JS_GetPropertyStr(context, permissions, work->user);
			}
			JS_FreeValue(context, permissions);
		}
		JS_FreeValue(context, json);
		tf_free((void*)work->settings);
	}
	if (JS_IsUndefined(result))
	{
		result = JS_NewArray(context);
	}
	JSValue error = JS_Call(context, JS_IsArray(context, result) ? work->promise[0] : work->promise[1], JS_UNDEFINED, 1, &result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, result);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	JS_FreeCString(context, work->user);
	tf_free(work);
}

static JSValue _tf_api_core_permissionsForUser(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	permissions_for_user_t* work = tf_malloc(sizeof(permissions_for_user_t));
	*work = (permissions_for_user_t) {
		.context = context,
		.user = JS_ToCString(context, argv[0]),
	};
	JSValue result = JS_NewPromiseCapability(context, work->promise);
	tf_ssb_run_work(ssb, _tf_api_core_permissions_for_user_work, _tf_api_core_permissions_for_user_after_work, work);
	return result;
}

typedef struct _permissions_granted_t
{
	JSContext* context;
	const char* user;
	const char* package_owner;
	const char* package_name;
	const char* settings;
	JSValue promise[2];
} permissions_granted_t;

static void _tf_api_core_permissions_granted_work(tf_ssb_t* ssb, void* user_data)
{
	permissions_granted_t* work = user_data;
	work->settings = tf_ssb_db_get_property(ssb, "core", "settings");
}

static void _tf_api_core_permissions_granted_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	permissions_granted_t* work = user_data;
	JSContext* context = work->context;
	JSValue result = JS_UNDEFINED;
	if (work->settings)
	{
		JSValue json = JS_ParseJSON(context, work->settings, strlen(work->settings), NULL);
		if (JS_IsObject(json) && work->user && work->package_owner && work->package_name)
		{
			JSValue user_permissions = JS_GetPropertyStr(context, json, "userPermissions");
			if (JS_IsObject(user_permissions))
			{
				JSValue user = JS_GetPropertyStr(context, user_permissions, work->user);
				if (JS_IsObject(user))
				{
					JSValue package_owner = JS_GetPropertyStr(context, user, work->package_owner);
					if (JS_IsObject(package_owner))
					{
						result = JS_GetPropertyStr(context, package_owner, work->package_name);
					}
					JS_FreeValue(context, package_owner);
				}
				JS_FreeValue(context, user);
			}
			JS_FreeValue(context, user_permissions);
		}
		JS_FreeValue(context, json);
		tf_free((void*)work->settings);
	}

	JSValue error = JS_Call(context, work->promise[0], JS_UNDEFINED, 1, &result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, result);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	JS_FreeCString(context, work->user);
	tf_free((void*)work->package_owner);
	tf_free((void*)work->package_name);
	tf_free(work);
}

static JSValue _tf_api_core_permissionsGranted(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	JSValue process = data[0];
	JSValue package_owner_value = JS_GetPropertyStr(context, process, "packageOwner");
	JSValue package_name_value = JS_GetPropertyStr(context, process, "packageName");
	const char* package_owner = JS_ToCString(context, package_owner_value);
	const char* package_name = JS_ToCString(context, package_name_value);
	permissions_granted_t* work = tf_malloc(sizeof(permissions_granted_t));
	*work = (permissions_granted_t) {
		.context = context,
		.user = _tf_ssb_get_process_credentials_session_name(context, process),
		.package_owner = tf_strdup(package_owner),
		.package_name = tf_strdup(package_name),
	};
	JS_FreeCString(context, package_owner);
	JS_FreeCString(context, package_name);
	JS_FreeValue(context, package_owner_value);
	JS_FreeValue(context, package_name_value);
	JSValue result = JS_NewPromiseCapability(context, work->promise);
	tf_ssb_run_work(ssb, _tf_api_core_permissions_granted_work, _tf_api_core_permissions_granted_after_work, work);
	return result;
}

static void _tf_api_core_all_permissions_granted_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	permissions_granted_t* work = user_data;
	JSContext* context = work->context;
	JSValue result = JS_UNDEFINED;
	if (work->settings)
	{
		JSValue json = JS_ParseJSON(context, work->settings, strlen(work->settings), NULL);
		if (JS_IsObject(json) && work->user)
		{
			JSValue user_permissions = JS_GetPropertyStr(context, json, "userPermissions");
			if (JS_IsObject(user_permissions))
			{
				result = JS_GetPropertyStr(context, user_permissions, work->user);
			}
			JS_FreeValue(context, user_permissions);
		}
		JS_FreeValue(context, json);
		tf_free((void*)work->settings);
	}

	JSValue error = JS_Call(context, work->promise[0], JS_UNDEFINED, 1, &result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, result);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	JS_FreeCString(context, work->user);
	tf_free(work);
}

static JSValue _tf_api_core_allPermissionsGranted(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	JSValue process = data[0];
	permissions_granted_t* work = tf_malloc(sizeof(permissions_granted_t));
	*work = (permissions_granted_t) {
		.context = context,
		.user = _tf_ssb_get_process_credentials_session_name(context, process),
	};
	JSValue result = JS_NewPromiseCapability(context, work->promise);
	tf_ssb_run_work(ssb, _tf_api_core_permissions_granted_work, _tf_api_core_all_permissions_granted_after_work, work);
	return result;
}

typedef struct _active_identity_work_t
{
	JSContext* context;
	const char* name;
	const char* package_owner;
	const char* package_name;
	char identity[k_id_base64_len];
	int result;
	JSValue promise[2];
} active_identity_work_t;

static void _tf_ssb_getActiveIdentity_visit(const char* identity, void* user_data)
{
	active_identity_work_t* request = user_data;
	if (!*request->identity)
	{
		snprintf(request->identity, sizeof(request->identity), "@%s", identity);
	}
}

static void _tf_ssb_getActiveIdentity_work(tf_ssb_t* ssb, void* user_data)
{
	active_identity_work_t* request = user_data;
	sqlite3* db = tf_ssb_acquire_db_reader(ssb);
	tf_ssb_db_identity_get_active(db, request->name, request->package_owner, request->package_name, request->identity, sizeof(request->identity));
	tf_ssb_release_db_reader(ssb, db);

	if (!*request->identity && tf_ssb_db_user_has_permission(ssb, NULL, request->name, "administration"))
	{
		tf_ssb_whoami(ssb, request->identity, sizeof(request->identity));
	}

	if (!*request->identity)
	{
		tf_ssb_db_identity_visit(ssb, request->name, _tf_ssb_getActiveIdentity_visit, request);
	}
}

static void _tf_ssb_getActiveIdentity_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	active_identity_work_t* request = user_data;
	JSContext* context = request->context;
	if (request->result == 0)
	{
		JSValue identity = JS_NewString(context, request->identity);
		JSValue error = JS_Call(context, request->promise[0], JS_UNDEFINED, 1, &identity);
		JS_FreeValue(context, identity);
		tf_util_report_error(context, error);
		JS_FreeValue(context, error);
	}
	else
	{
		JSValue error = JS_Call(context, request->promise[1], JS_UNDEFINED, 0, NULL);
		tf_util_report_error(context, error);
		JS_FreeValue(context, error);
	}
	JS_FreeValue(context, request->promise[0]);
	JS_FreeValue(context, request->promise[1]);
	tf_free((void*)request->name);
	tf_free((void*)request->package_owner);
	tf_free((void*)request->package_name);
	tf_free(request);
}

static JSValue _tf_ssb_getActiveIdentity(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	JSValue process = data[0];
	JSValue package_owner_value = JS_GetPropertyStr(context, process, "packageOwner");
	JSValue package_name_value = JS_GetPropertyStr(context, process, "packageName");

	const char* name = _tf_ssb_get_process_credentials_session_name(context, process);
	const char* package_owner = JS_ToCString(context, package_owner_value);
	const char* package_name = JS_ToCString(context, package_name_value);
	active_identity_work_t* work = tf_malloc(sizeof(active_identity_work_t));
	*work = (active_identity_work_t) {
		.context = context,
		.name = tf_strdup(name),
		.package_owner = tf_strdup(package_owner),
		.package_name = tf_strdup(package_name),
	};
	JSValue result = JS_NewPromiseCapability(context, work->promise);
	JS_FreeCString(context, name);
	JS_FreeCString(context, package_owner);
	JS_FreeCString(context, package_name);

	JS_FreeValue(context, package_owner_value);
	JS_FreeValue(context, package_name_value);
	tf_ssb_run_work(ssb, _tf_ssb_getActiveIdentity_work, _tf_ssb_getActiveIdentity_after_work, work);
	return result;
}

typedef struct _identities_visit_t
{
	JSContext* context;
	JSValue promise[2];
	const char** identities;
	int count;
	char user[];
} identities_visit_t;

static void _tf_ssb_getIdentities_visit(const char* identity, void* user_data)
{
	identities_visit_t* work = user_data;
	work->identities = tf_resize_vec(work->identities, (work->count + 1) * sizeof(const char*));
	char id[k_id_base64_len];
	snprintf(id, sizeof(id), "@%s", identity);
	work->identities[work->count++] = tf_strdup(id);
}

static void _tf_ssb_get_all_identities_work(tf_ssb_t* ssb, void* user_data)
{
	tf_ssb_db_identity_visit_all(ssb, _tf_ssb_getIdentities_visit, user_data);
}

static void _tf_ssb_get_identities_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	identities_visit_t* work = user_data;
	JSContext* context = tf_ssb_get_context(ssb);
	JSValue result = JS_NewArray(context);
	for (int i = 0; i < work->count; i++)
	{
		JS_SetPropertyUint32(context, result, i, JS_NewString(context, work->identities[i]));
		tf_free((void*)work->identities[i]);
	}
	tf_free(work->identities);

	JSValue error = JS_Call(context, work->promise[0], JS_UNDEFINED, 1, &result);
	JS_FreeValue(context, result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	tf_free(work);
}

static JSValue _tf_ssb_getAllIdentities(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	JSValue result = JS_UNDEFINED;
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	if (ssb)
	{
		identities_visit_t* work = tf_malloc(sizeof(identities_visit_t));
		*work = (identities_visit_t) {
			.context = context,
		};

		result = JS_NewPromiseCapability(context, work->promise);
		tf_ssb_run_work(ssb, _tf_ssb_get_all_identities_work, _tf_ssb_get_identities_after_work, work);
	}
	return result;
}

static void _tf_ssb_get_identities_work(tf_ssb_t* ssb, void* user_data)
{
	identities_visit_t* work = user_data;
	if (tf_ssb_db_user_has_permission(ssb, NULL, work->user, "administration"))
	{
		char id[k_id_base64_len] = "";
		if (tf_ssb_whoami(ssb, id, sizeof(id)))
		{
			_tf_ssb_getIdentities_visit(*id == '@' ? id + 1 : id, work);
		}
	}
	tf_ssb_db_identity_visit(ssb, work->user, _tf_ssb_getIdentities_visit, user_data);
}

static JSValue _tf_ssb_getIdentities(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	JSValue result = JS_UNDEFINED;
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	JSValue process = data[0];
	if (ssb)
	{
		const char* user = _tf_ssb_get_process_credentials_session_name(context, process);
		if (user)
		{
			size_t user_length = user ? strlen(user) : 0;
			identities_visit_t* work = tf_malloc(sizeof(identities_visit_t) + user_length + 1);
			*work = (identities_visit_t) {
				.context = context,
			};
			memcpy(work->user, user, user_length + 1);
			JS_FreeCString(context, user);

			result = JS_NewPromiseCapability(context, work->promise);
			tf_ssb_run_work(ssb, _tf_ssb_get_identities_work, _tf_ssb_get_identities_after_work, work);
		}
	}
	return result;
}

static JSValue _tf_ssb_getOwnerIdentities(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	JSValue result = JS_UNDEFINED;
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	JSValue process = data[0];
	if (ssb)
	{
		JSValue value = JS_GetPropertyStr(context, process, "packageOwner");
		const char* user = JS_ToCString(context, value);
		size_t user_length = user ? strlen(user) : 0;
		identities_visit_t* work = tf_malloc(sizeof(identities_visit_t) + user_length + 1);
		*work = (identities_visit_t) {
			.context = context,
		};
		memcpy(work->user, user, user_length + 1);
		JS_FreeCString(context, user);
		JS_FreeValue(context, value);

		result = JS_NewPromiseCapability(context, work->promise);
		tf_ssb_run_work(ssb, _tf_ssb_get_identities_work, _tf_ssb_get_identities_after_work, work);
	}
	return result;
}

typedef struct _settings_descriptions_get_t
{
	const char* settings;
	JSContext* context;
	JSValue promise[2];
} settings_descriptions_get_t;

static void _tf_ssb_get_settings_descriptions_work(tf_ssb_t* ssb, void* user_data)
{
	settings_descriptions_get_t* work = user_data;
	work->settings = tf_ssb_db_get_property(ssb, "core", "settings");
}

static void _tf_ssb_get_settings_descriptions_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	settings_descriptions_get_t* work = user_data;
	JSContext* context = work->context;
	JSValue result = JS_NewObject(context);
	JSValue settings = JS_ParseJSON(context, work->settings ? work->settings : "", work->settings ? strlen(work->settings) : 0, NULL);
	const char* name;
	const char* type;
	tf_setting_kind_t kind;
	const char* description;
	for (int i = 0; tf_util_get_global_setting_by_index(i, &name, &type, &kind, &description); i++)
	{
		JSValue entry = JS_NewObject(context);
		JS_SetPropertyStr(context, entry, "type", JS_NewString(context, type));
		JS_SetPropertyStr(context, entry, "description", JS_NewString(context, description));
		switch (kind)
		{
		case k_kind_unknown:
			break;
		case k_kind_bool:
			JS_SetPropertyStr(context, entry, "default_value", JS_NewBool(context, tf_util_get_default_global_setting_bool(name)));
			break;
		case k_kind_int:
			JS_SetPropertyStr(context, entry, "default_value", JS_NewInt32(context, tf_util_get_default_global_setting_int(name)));
			break;
		case k_kind_string:
			JS_SetPropertyStr(context, entry, "default_value", JS_NewString(context, tf_util_get_default_global_setting_string(name)));
			break;
		}
		if (JS_IsObject(settings))
		{
			JS_SetPropertyStr(context, entry, "value", JS_GetPropertyStr(context, settings, name));
		}
		JS_SetPropertyStr(context, result, name, entry);
	}
	JS_FreeValue(context, settings);
	JSValue error = JS_Call(context, work->promise[0], JS_UNDEFINED, 1, &result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, result);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	tf_free((void*)work->settings);
	tf_free(work);
}

static JSValue _tf_ssb_globalSettingsDescriptions(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	settings_descriptions_get_t* work = tf_malloc(sizeof(settings_descriptions_get_t));
	*work = (settings_descriptions_get_t) { .context = context };
	JSValue result = JS_NewPromiseCapability(context, work->promise);
	tf_ssb_run_work(ssb, _tf_ssb_get_settings_descriptions_work, _tf_ssb_get_settings_descriptions_after_work, work);
	return result;
}

typedef struct _settings_get_t
{
	const char* key;
	tf_setting_kind_t kind;
	void* value;
	JSContext* context;
	JSValue promise[2];
} settings_get_t;

static void _tf_ssb_settings_get_work(tf_ssb_t* ssb, void* user_data)
{
	settings_get_t* work = user_data;
	work->kind = tf_util_get_global_setting_kind(work->key);
	switch (work->kind)
	{
	case k_kind_unknown:
		break;
	case k_kind_bool:
		{
			sqlite3* db = tf_ssb_acquire_db_reader(ssb);
			bool value = false;
			tf_ssb_db_get_global_setting_bool(db, work->key, &value);
			work->value = (void*)(intptr_t)value;
			tf_ssb_release_db_reader(ssb, db);
		}
		break;
	case k_kind_int:
		{
			sqlite3* db = tf_ssb_acquire_db_reader(ssb);
			int64_t value = 0;
			tf_ssb_db_get_global_setting_int64(db, work->key, &value);
			work->value = (void*)(intptr_t)value;
			tf_ssb_release_db_reader(ssb, db);
		}
		break;
	case k_kind_string:
		{
			sqlite3* db = tf_ssb_acquire_db_reader(ssb);
			work->value = (void*)tf_ssb_db_get_global_setting_string_alloc(db, work->key);
			tf_ssb_release_db_reader(ssb, db);
		}
		break;
	}
}

static void _tf_ssb_settings_get_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	settings_get_t* work = user_data;
	JSContext* context = tf_ssb_get_context(ssb);
	JSValue result = JS_UNDEFINED;
	switch (work->kind)
	{
	case k_kind_unknown:
		break;
	case k_kind_bool:
		result = work->value ? JS_TRUE : JS_FALSE;
		break;
	case k_kind_int:
		result = JS_NewInt64(context, (int64_t)(intptr_t)work->value);
		break;
	case k_kind_string:
		result = JS_NewString(context, (const char*)work->value);
		tf_free(work->value);
		break;
	}

	JSValue error = JS_Call(context, work->promise[0], JS_UNDEFINED, 1, &result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, result);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	JS_FreeCString(context, work->key);
	tf_free(work);
}

static JSValue _tf_ssb_globalSettingsGet(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	settings_get_t* work = tf_malloc(sizeof(settings_get_t));
	*work = (settings_get_t) { .context = context, .key = JS_ToCString(context, argv[0]) };
	JSValue result = JS_NewPromiseCapability(context, work->promise);
	tf_ssb_run_work(ssb, _tf_ssb_settings_get_work, _tf_ssb_settings_get_after_work, work);
	return result;
}

typedef struct _modify_block_t
{
	const char* user;
	char id[k_id_base64_len];
	bool add;
	bool completed;
	JSValue result;
	JSValue promise[2];
} modify_block_t;

static void _tf_ssb_modify_block_work(tf_ssb_t* ssb, void* user_data)
{
	modify_block_t* work = user_data;
	sqlite3* db = tf_ssb_acquire_db_writer(ssb);
	if (work->add)
	{
		tf_ssb_db_add_block(db, work->id);
	}
	else
	{
		tf_ssb_db_remove_block(db, work->id);
	}
	tf_ssb_release_db_writer(ssb, db);
	work->completed = true;
}

static void _tf_ssb_modify_block_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	modify_block_t* request = user_data;
	JSContext* context = tf_ssb_get_context(ssb);
	JSValue error = JS_Call(context, request->completed ? request->promise[0] : request->promise[1], JS_UNDEFINED, 1, &request->result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, request->promise[0]);
	JS_FreeValue(context, request->promise[1]);
	JS_FreeValue(context, request->result);
	JS_FreeCString(context, request->user);
	tf_free(request);
}

typedef void(permission_test_callback_t)(JSContext* context, bool granted, JSValue value, void* user_data);

typedef struct _permission_test_t
{
	JSContext* context;
	bool completed;
	permission_test_callback_t* callback;
	void* user_data;
	char name[];
} permission_test_t;

static JSValue _tf_ssb_permission_test_resolve(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	permission_test_t* work = JS_GetOpaque(data[0], s_permission_test_class_id);
	work->completed = true;
	tf_task_t* task = tf_task_get(work->context);
	tf_trace_t* trace = tf_task_get_trace(task);
	tf_trace_begin(trace, work->name);
	work->callback(context, true, argv[0], work->user_data);
	tf_trace_end(trace);
	return JS_UNDEFINED;
}

static JSValue _tf_ssb_permission_test_reject(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	permission_test_t* work = JS_GetOpaque(data[0], s_permission_test_class_id);
	work->completed = true;
	tf_task_t* task = tf_task_get(work->context);
	tf_trace_t* trace = tf_task_get_trace(task);
	tf_trace_begin(trace, work->name);
	work->callback(context, false, argv[0], work->user_data);
	tf_trace_end(trace);
	return JS_UNDEFINED;
}

static void _tf_ssb_permission_test_finalizer(JSRuntime* runtime, JSValue value)
{
	permission_test_t* work = JS_GetOpaque(value, s_permission_test_class_id);
	if (!work->completed)
	{
		JSValue arg = JS_ThrowInternalError(work->context, "Permission test incomplete.");
		tf_task_t* task = tf_task_get(work->context);
		tf_trace_t* trace = tf_task_get_trace(task);
		tf_trace_begin(trace, work->name);
		work->callback(work->context, false, arg, work->user_data);
		tf_trace_end(trace);
		JS_FreeValue(work->context, arg);
	}
	tf_free(work);
}

static void _tf_ssb_permission_test(JSContext* context, JSValue process, const char* permission, const char* description, permission_test_callback_t* callback, void* user_data)
{
	if (!s_permission_test_class_id)
	{
		JSClassDef def = {
			.class_name = "permission_test",
			.finalizer = _tf_ssb_permission_test_finalizer,
		};
		JS_NewClassID(&s_permission_test_class_id);
		JS_NewClass(JS_GetRuntime(context), s_permission_test_class_id, &def);
	}

	const char* name = tf_util_function_to_string(callback);
	size_t name_length = name ? strlen(name) : 0;
	permission_test_t* payload = tf_malloc(sizeof(permission_test_t) + name_length + 1);
	*payload = (permission_test_t) {
		.context = context,
		.callback = callback,
		.user_data = user_data,
	};
	tf_string_set(payload->name, name_length + 1, name);
	tf_free((void*)name);
	JSValue opaque = JS_NewObjectClass(context, s_permission_test_class_id);
	JS_SetOpaque(opaque, payload);
	JSValue imports = JS_GetPropertyStr(context, process, "imports");
	JSValue core = JS_GetPropertyStr(context, imports, "core");
	JSValue permission_test = JS_GetPropertyStr(context, core, "permissionTest");
	JSValue args[] = {
		JS_NewString(context, permission),
		JS_NewString(context, description),
	};
	JSValue promise = JS_Call(context, permission_test, imports, tf_countof(args), args);
	JSValue then = JS_GetPropertyStr(context, promise, "then");
	JSValue catch = JS_GetPropertyStr(context, promise, "catch");
	JSValue resolve = JS_NewCFunctionData(context, _tf_ssb_permission_test_resolve, 1, 0, 1, &opaque);
	JSValue reject = JS_NewCFunctionData(context, _tf_ssb_permission_test_reject, 1, 0, 1, &opaque);
	JSValue result = JS_Call(context, then, promise, 1, &resolve);
	tf_util_report_error(context, result);
	JS_FreeValue(context, result);
	result = JS_Call(context, catch, promise, 1, &reject);
	tf_util_report_error(context, result);
	JS_FreeValue(context, result);
	JS_FreeValue(context, opaque);
	JS_FreeValue(context, promise);
	JS_FreeValue(context, resolve);
	JS_FreeValue(context, reject);
	JS_FreeValue(context, then);
	JS_FreeValue(context, catch);
	for (int i = 0; i < tf_countof(args); i++)
	{
		JS_FreeValue(context, args[i]);
	}
	JS_FreeValue(context, permission_test);
	JS_FreeValue(context, core);
	JS_FreeValue(context, imports);
}

static void _tf_ssb_modify_block_start_work(JSContext* context, bool granted, JSValue value, void* user_data)
{
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	modify_block_t* work = user_data;
	work->result = JS_DupValue(context, value);
	if (granted)
	{
		tf_ssb_run_work(ssb, _tf_ssb_modify_block_work, _tf_ssb_modify_block_after_work, work);
	}
	else
	{
		_tf_ssb_modify_block_after_work(ssb, 0, work);
	}
}

static JSValue _tf_ssb_add_block(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	const char* id = JS_ToCString(context, argv[0]);
	modify_block_t* work = tf_malloc(sizeof(modify_block_t));
	*work = (modify_block_t) {
		.user = _tf_ssb_get_process_credentials_session_name(context, data[0]),
		.add = true,
	};
	tf_string_set(work->id, sizeof(work->id), id);
	JSValue result = JS_NewPromiseCapability(context, work->promise);
	JS_FreeCString(context, id);

	char description[256] = "";
	snprintf(description, sizeof(description), "Block %s.", work->id);
	_tf_ssb_permission_test(context, data[0], "modify_blocks", description, _tf_ssb_modify_block_start_work, work);
	return result;
}

static JSValue _tf_ssb_remove_block(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	const char* id = JS_ToCString(context, argv[0]);
	modify_block_t* work = tf_malloc(sizeof(modify_block_t));
	*work = (modify_block_t) {
		.user = _tf_ssb_get_process_credentials_session_name(context, data[0]),
		.add = false,
	};
	tf_string_set(work->id, sizeof(work->id), id);
	JSValue result = JS_NewPromiseCapability(context, work->promise);
	JS_FreeCString(context, id);

	char description[256] = "";
	snprintf(description, sizeof(description), "Unblock %s.", work->id);
	_tf_ssb_permission_test(context, data[0], "modify_blocks", description, _tf_ssb_modify_block_start_work, work);
	return result;
}

typedef struct _block_t
{
	char id[k_id_base64_len];
	double timestamp;
} block_t;

typedef struct _get_blocks_t
{
	block_t* blocks;
	int count;
	JSValue promise[2];
} get_blocks_t;

static void _get_blocks_callback(const char* id, double timestamp, void* user_data)
{
	get_blocks_t* work = user_data;
	work->blocks = tf_resize_vec(work->blocks, sizeof(block_t) * (work->count + 1));
	work->blocks[work->count] = (block_t) { .timestamp = timestamp };
	tf_string_set(work->blocks[work->count].id, sizeof(work->blocks[work->count].id), id);
	work->count++;
}

static void _tf_ssb_get_blocks_work(tf_ssb_t* ssb, void* user_data)
{
	sqlite3* db = tf_ssb_acquire_db_reader(ssb);
	tf_ssb_db_get_blocks(db, _get_blocks_callback, user_data);
	tf_ssb_release_db_reader(ssb, db);
}

static void _tf_ssb_get_blocks_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	get_blocks_t* work = user_data;
	JSContext* context = tf_ssb_get_context(ssb);

	JSValue result = JS_NewArray(context);
	for (int i = 0; i < work->count; i++)
	{
		JSValue entry = JS_NewObject(context);
		JS_SetPropertyStr(context, entry, "id", JS_NewString(context, work->blocks[i].id));
		JS_SetPropertyStr(context, entry, "timestamp", JS_NewFloat64(context, work->blocks[i].timestamp));
		JS_SetPropertyUint32(context, result, i, entry);
	}

	JSValue error = JS_Call(context, work->promise[0], JS_UNDEFINED, 1, &result);
	JS_FreeValue(context, result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	tf_free(work->blocks);
	tf_free(work);
}

static JSValue _tf_ssb_get_blocks(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	get_blocks_t* work = tf_malloc(sizeof(get_blocks_t));
	*work = (get_blocks_t) { 0 };
	JSValue result = JS_NewPromiseCapability(context, work->promise);
	tf_ssb_run_work(ssb, _tf_ssb_get_blocks_work, _tf_ssb_get_blocks_after_work, work);
	return result;
}

typedef struct _global_setting_set_t
{
	const char* key;
	const char* value;
	JSValue promise[2];
	bool done;
	JSValue result;
} global_setting_set_t;

static void _tf_ssb_globalSettingsSet_work(tf_ssb_t* ssb, void* user_data)
{
	global_setting_set_t* work = user_data;
	sqlite3* db = tf_ssb_acquire_db_writer(ssb);
	work->done = tf_ssb_db_set_global_setting_from_string(db, work->key, work->value);
	tf_ssb_release_db_writer(ssb, db);
}

static void _tf_ssb_globalSettingsSet_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	global_setting_set_t* work = user_data;
	JSContext* context = tf_ssb_get_context(ssb);
	JSValue error = JS_Call(context, work->done ? work->promise[0] : work->promise[1], JS_UNDEFINED, 1, &work->result);
	JS_FreeValue(context, work->result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	JS_FreeCString(context, work->key);
	JS_FreeCString(context, work->value);
	tf_free(work);
}

static void _tf_ssb_globalSettingsSet_permission_callback(JSContext* context, bool granted, JSValue value, void* user_data)
{
	global_setting_set_t* work = user_data;
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	work->result = value;
	if (granted)
	{
		tf_ssb_run_work(ssb, _tf_ssb_globalSettingsSet_work, _tf_ssb_globalSettingsSet_after_work, work);
	}
	else
	{
		_tf_ssb_globalSettingsSet_after_work(ssb, -1, work);
	}
}

static JSValue _tf_ssb_globalSettingsSet(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	const char* key = JS_ToCString(context, argv[0]);
	const char* value = JS_ToCString(context, argv[1]);

	global_setting_set_t* work = tf_malloc(sizeof(global_setting_set_t));
	*work = (global_setting_set_t) {
		.key = key,
		.value = value,
	};
	JSValue result = JS_NewPromiseCapability(context, work->promise);

	char description[256] = "";
	snprintf(description, sizeof(description), "Set %s to %s.", key, value);
	_tf_ssb_permission_test(context, data[0], "set_global_setting", description, _tf_ssb_globalSettingsSet_permission_callback, work);
	return result;
}

typedef struct _delete_user_t
{
	const char* user;
	bool completed;
	JSValue result;
	JSValue promise[2];
} delete_user_t;

static void _tf_ssb_delete_user_work(tf_ssb_t* ssb, void* user_data)
{
	delete_user_t* work = user_data;
	size_t length = strlen("user:") + strlen(work->user) + 1;
	char* buffer = alloca(length);
	snprintf(buffer, length, "user:%s", work->user);
	work->completed = tf_ssb_db_remove_property(ssb, "auth", buffer) || work->completed;
	work->completed = tf_ssb_db_remove_value_from_array_property(ssb, "auth", "users", work->user) || work->completed;
}

static void _tf_ssb_delete_user_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	delete_user_t* work = user_data;
	JSContext* context = tf_ssb_get_context(ssb);
	if (!work->completed && JS_IsUndefined(work->result))
	{
		work->result = JS_NewString(context, "User not found.");
	}
	JSValue error = JS_Call(context, work->completed ? work->promise[0] : work->promise[1], JS_UNDEFINED, 1, &work->result);
	JS_FreeValue(context, work->result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	JS_FreeCString(context, work->user);
	tf_free(work);
}

static void _tf_ssb_delete_user_permission_callback(JSContext* context, bool granted, JSValue value, void* user_data)
{
	delete_user_t* work = user_data;
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	if (granted)
	{
		tf_ssb_run_work(ssb, _tf_ssb_delete_user_work, _tf_ssb_delete_user_after_work, work);
	}
	else
	{
		_tf_ssb_delete_user_after_work(ssb, -1, work);
	}
}

static JSValue _tf_ssb_delete_user(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	delete_user_t* work = tf_malloc(sizeof(delete_user_t));
	*work = (delete_user_t) {
		.user = JS_ToCString(context, argv[0]),
	};
	JSValue result = JS_NewPromiseCapability(context, work->promise);

	char description[256] = "";
	snprintf(description, sizeof(description), "Delete user '%s'.", work->user);
	_tf_ssb_permission_test(context, data[0], "delete_user", description, _tf_ssb_delete_user_permission_callback, work);
	return result;
}

typedef struct _swap_with_server_identity_t
{
	char server_id[k_id_base64_len];
	char user_id[k_id_base64_len];
	JSValue promise[2];
	char* error;
	char user[];
} swap_with_server_identity_t;

static void _tf_ssb_swap_with_server_identity_work(tf_ssb_t* ssb, void* user_data)
{
	swap_with_server_identity_t* work = user_data;
	sqlite3* db = tf_ssb_acquire_db_writer(ssb);
	work->error = tf_ssb_db_swap_with_server_identity(db, work->user, work->user_id, work->server_id);
	tf_ssb_release_db_writer(ssb, db);
}

static void _tf_ssb_swap_with_server_identity_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	swap_with_server_identity_t* work = user_data;
	JSContext* context = tf_ssb_get_context(ssb);
	JSValue error = JS_UNDEFINED;
	if (work->error)
	{
		JSValue arg = JS_ThrowInternalError(context, "%s", work->error);
		JSValue exception = JS_GetException(context);
		error = JS_Call(context, work->promise[1], JS_UNDEFINED, 1, &exception);
		tf_free(work->error);
		JS_FreeValue(context, exception);
		JS_FreeValue(context, arg);
	}
	else
	{
		error = JS_Call(context, work->promise[0], JS_UNDEFINED, 0, NULL);
	}
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	tf_free(work);
}

static void _tf_ssb_swap_with_server_identity_permission_callback(JSContext* context, bool granted, JSValue value, void* user_data)
{
	swap_with_server_identity_t* work = user_data;
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	tf_ssb_run_work(ssb, _tf_ssb_swap_with_server_identity_work, _tf_ssb_swap_with_server_identity_after_work, work);
}

static JSValue _tf_ssb_swap_with_server_identity(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	const char* user = _tf_ssb_get_process_credentials_session_name(context, data[0]);
	size_t user_length = user ? strlen(user) : 0;
	const char* id = JS_ToCString(context, argv[0]);
	swap_with_server_identity_t* work = tf_malloc(sizeof(swap_with_server_identity_t) + user_length + 1);
	*work = (swap_with_server_identity_t) { 0 };
	tf_ssb_whoami(ssb, work->server_id, sizeof(work->server_id));
	tf_string_set(work->user_id, sizeof(work->user_id), id);
	if (user)
	{
		memcpy(work->user, user, user_length + 1);
	}
	else
	{
		*work->user = '\0';
	}
	JSValue result = JS_NewPromiseCapability(context, work->promise);
	char description[1024];
	snprintf(description, sizeof(description), "Swap identity %s with %s.", work->user_id, work->server_id);
	_tf_ssb_permission_test(context, data[0], "delete_user", description, _tf_ssb_swap_with_server_identity_permission_callback, work);
	JS_FreeCString(context, id);
	JS_FreeCString(context, user);
	return result;
}

static bool _tf_ssb_get_private_key_curve25519_internal(sqlite3* db, const char* user, const char* identity, uint8_t out_private_key[static crypto_sign_SECRETKEYBYTES])
{
	if (!user || !identity)
	{
		tf_printf("user=%p identity=%p out_private_key=%p\n", user, identity, out_private_key);
		return false;
	}

	bool success = false;
	sqlite3_stmt* statement = NULL;
	if (sqlite3_prepare_v2(db, "SELECT private_key FROM identities WHERE user = ? AND public_key = ?", -1, &statement, NULL) == SQLITE_OK)
	{
		if (sqlite3_bind_text(statement, 1, user, -1, NULL) == SQLITE_OK && sqlite3_bind_text(statement, 2, *identity == '@' ? identity + 1 : identity, -1, NULL) == SQLITE_OK)
		{
			while (sqlite3_step(statement) == SQLITE_ROW)
			{
				uint8_t key[crypto_sign_SECRETKEYBYTES] = { 0 };
				int length = tf_base64_decode((const char*)sqlite3_column_text(statement, 0), sqlite3_column_bytes(statement, 0) - strlen(".ed25519"), key, sizeof(key));
				if (length == crypto_sign_SECRETKEYBYTES)
				{
					success = crypto_sign_ed25519_sk_to_curve25519(out_private_key, key) == 0;
				}
			}
		}
		sqlite3_finalize(statement);
	}
	return success;
}

static bool _tf_ssb_get_private_key_curve25519(tf_ssb_t* ssb, sqlite3* db, const char* user, const char* identity, uint8_t out_private_key[static crypto_sign_SECRETKEYBYTES])
{
	if (_tf_ssb_get_private_key_curve25519_internal(db, user, identity, out_private_key))
	{
		return true;
	}

	if (tf_ssb_db_user_has_permission(ssb, db, user, "administration"))
	{
		return _tf_ssb_get_private_key_curve25519_internal(db, ":admin", identity, out_private_key);
	}

	return false;
}

typedef struct _private_message_encrypt_t
{
	const char* signer_user;
	const char* signer_identity;
	const char* recipients[k_max_private_message_recipients];
	int recipient_count;
	const char* message;
	size_t message_size;
	JSValue promise[2];
	bool error_id_not_found;
	char* encrypted;
	size_t encrypted_length;
} private_message_encrypt_t;

static void _tf_ssb_private_message_encrypt_work(tf_ssb_t* ssb, void* user_data)
{
	private_message_encrypt_t* work = user_data;

	uint8_t private_key[crypto_sign_SECRETKEYBYTES] = { 0 };
	sqlite3* db = tf_ssb_acquire_db_reader(ssb);
	bool found = _tf_ssb_get_private_key_curve25519(ssb, db, work->signer_user, work->signer_identity, private_key);
	tf_ssb_release_db_reader(ssb, db);

	if (found)
	{
		work->encrypted = tf_ssb_private_message_encrypt(private_key, work->recipients, work->recipient_count, work->message, work->message_size);
		work->encrypted_length = work->encrypted ? strlen(work->encrypted) : 0;
	}
	else
	{
		work->error_id_not_found = true;
	}
}

static void _tf_ssb_private_message_encrypt_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	private_message_encrypt_t* work = user_data;
	JSContext* context = tf_ssb_get_context(ssb);
	JSValue result = JS_UNDEFINED;
	bool success = false;
	if (work->error_id_not_found)
	{
		result = JS_ThrowInternalError(context, "Unable to get key for ID %s of user %s.", work->signer_identity, work->signer_user);
	}
	else if (!work->encrypted)
	{
		result = JS_ThrowInternalError(context, "Encrypt failed.");
	}
	else
	{
		result = JS_NewStringLen(context, work->encrypted, work->encrypted_length);
		tf_free((void*)work->encrypted);
		success = true;
	}

	for (int i = 0; i < work->recipient_count; i++)
	{
		tf_free((void*)work->recipients[i]);
	}
	JSValue error = JS_Call(context, work->promise[success ? 0 : 1], JS_UNDEFINED, 1, &result);
	JS_FreeValue(context, result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	JS_FreeCString(context, work->signer_user);
	JS_FreeCString(context, work->signer_identity);
	JS_FreeCString(context, work->message);
	tf_free(work);
}

static JSValue _tf_ssb_private_message_encrypt(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	JSValue result = JS_UNDEFINED;
	JSValue process = data[0];
	int recipient_count = tf_util_get_length(context, argv[1]);
	if (recipient_count < 1 || recipient_count > k_max_private_message_recipients)
	{
		return JS_ThrowRangeError(context, "Number of recipients must be between 1 and %d.", k_max_private_message_recipients);
	}

	const char* session_name_string = _tf_ssb_get_process_credentials_session_name(context, process);
	char* recipients[k_max_private_message_recipients] = { 0 };
	for (int i = 0; i < recipient_count && JS_IsUndefined(result); i++)
	{
		JSValue recipient = JS_GetPropertyUint32(context, argv[1], i);
		const char* id = JS_ToCString(context, recipient);
		if (id)
		{
			recipients[i] = tf_strdup(id);
			JS_FreeCString(context, id);
		}
		JS_FreeValue(context, recipient);
	}

	if (JS_IsUndefined(result))
	{
		const char* signer_identity = JS_ToCString(context, argv[0]);
		size_t message_size = 0;
		const char* message = JS_ToCStringLen(context, &message_size, argv[2]);

		tf_task_t* task = tf_task_get(context);
		tf_ssb_t* ssb = tf_task_get_ssb(task);
		private_message_encrypt_t* work = tf_malloc(sizeof(private_message_encrypt_t));
		*work = (private_message_encrypt_t) {
			.signer_user = session_name_string,
			.signer_identity = signer_identity,
			.recipient_count = recipient_count,
			.message = message,
			.message_size = message_size,
		};
		static_assert(sizeof(work->recipients) == sizeof(recipients), "size mismatch");
		memcpy(work->recipients, recipients, sizeof(recipients));
		result = JS_NewPromiseCapability(context, work->promise);
		tf_ssb_run_work(ssb, _tf_ssb_private_message_encrypt_work, _tf_ssb_private_message_encrypt_after_work, work);
	}

	return result;
}

typedef struct _private_message_decrypt_t
{
	const char* user;
	const char* identity;
	size_t message_size;
	const char* message;
	const char* decrypted;
	size_t decrypted_size;
	const char* error;
	JSValue promise[2];
} private_message_decrypt_t;

static void _tf_ssb_private_message_decrypt_work(tf_ssb_t* ssb, void* user_data)
{
	private_message_decrypt_t* work = user_data;

	uint8_t private_key[crypto_sign_SECRETKEYBYTES] = { 0 };
	sqlite3* db = tf_ssb_acquire_db_reader(ssb);
	bool found = _tf_ssb_get_private_key_curve25519(ssb, db, work->user, work->identity, private_key);
	tf_ssb_release_db_reader(ssb, db);

	if (found)
	{
		if (work->message_size >= strlen(".box") && memcmp(work->message + work->message_size - strlen(".box"), ".box", strlen(".box")) == 0)
		{
			uint8_t* decoded = tf_malloc(work->message_size);
			int decoded_length = tf_base64_decode(work->message, work->message_size - strlen(".box"), decoded, work->message_size);
			uint8_t* nonce = decoded;
			uint8_t* public_key = decoded + crypto_box_NONCEBYTES;
			if (public_key + crypto_secretbox_KEYBYTES < decoded + decoded_length)
			{
				uint8_t shared_secret[crypto_secretbox_KEYBYTES] = { 0 };
				if (crypto_scalarmult(shared_secret, private_key, public_key) == 0)
				{
					enum
					{
						k_recipient_header_bytes = crypto_secretbox_MACBYTES + sizeof(uint8_t) + crypto_secretbox_KEYBYTES
					};
					for (uint8_t* p = decoded + crypto_box_NONCEBYTES + crypto_secretbox_KEYBYTES; p <= decoded + decoded_length - k_recipient_header_bytes;
						 p += k_recipient_header_bytes)
					{
						uint8_t out[k_recipient_header_bytes] = { 0 };
						int opened = crypto_secretbox_open_easy(out, p, k_recipient_header_bytes, nonce, shared_secret);
						if (opened != -1)
						{
							int recipients = (int)out[0];
							uint8_t* body = decoded + crypto_box_NONCEBYTES + crypto_secretbox_KEYBYTES + k_recipient_header_bytes * recipients;
							size_t body_size = decoded + decoded_length - body;
							uint8_t* decrypted = tf_malloc(body_size);
							uint8_t* key = out + 1;
							if (crypto_secretbox_open_easy(decrypted, body, body_size, nonce, key) != -1)
							{
								work->decrypted = (const char*)decrypted;
								work->decrypted_size = body_size - crypto_secretbox_MACBYTES;
							}
							else
							{
								work->error = "Received key to open secret box containing message body, but it did not work.";
							}
						}
					}
				}
				else
				{
					work->error = "crypto_scalarmult failed.";
				}
			}
			else
			{
				work->error = "Encrypted message was not long enough to contain its one-time public key.";
			}
			tf_free(decoded);
		}
		else
		{
			work->error = "Message does not end in \".box\".";
		}
	}
	else
	{
		work->error = "Private key not found for user.";
	}
}

static void _tf_ssb_private_message_decrypt_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	private_message_decrypt_t* work = user_data;
	JSContext* context = tf_ssb_get_context(ssb);
	JSValue error = JS_UNDEFINED;
	if (work->error)
	{
		JSValue result = JS_ThrowInternalError(context, "%s", work->error);
		error = JS_Call(context, work->promise[1], JS_UNDEFINED, 1, &result);
		JS_FreeValue(context, result);
	}
	else if (work->decrypted)
	{
		JSValue result = JS_NewStringLen(context, work->decrypted, work->decrypted_size);
		error = JS_Call(context, work->promise[0], JS_UNDEFINED, 1, &result);
		JS_FreeValue(context, result);
	}
	else
	{
		JSValue result = JS_UNDEFINED;
		error = JS_Call(context, work->promise[0], JS_UNDEFINED, 1, &result);
	}
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	JS_FreeCString(context, work->user);
	JS_FreeCString(context, work->identity);
	JS_FreeCString(context, work->message);
	tf_free((void*)work->decrypted);
	tf_free(work);
}

static JSValue _tf_ssb_private_message_decrypt(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	JSValue process = data[0];
	const char* identity = JS_ToCString(context, argv[0]);
	size_t message_size = 0;
	const char* message = JS_ToCStringLen(context, &message_size, argv[1]);
	const char* session_name_string = _tf_ssb_get_process_credentials_session_name(context, process);

	private_message_decrypt_t* work = tf_malloc(sizeof(private_message_decrypt_t));
	*work = (private_message_decrypt_t) {
		.user = session_name_string,
		.identity = identity,
		.message_size = message_size,
		.message = message,
	};
	JSValue result = JS_NewPromiseCapability(context, work->promise);
	tf_ssb_run_work(ssb, _tf_ssb_private_message_decrypt_work, _tf_ssb_private_message_decrypt_after_work, work);
	return result;
}

typedef struct _append_message_t
{
	char id[k_id_base64_len];
	uint8_t private_key[crypto_sign_SECRETKEYBYTES];
	bool got_private_key;
	char previous_id[512];
	int32_t previous_sequence;
	JSContext* context;
	JSValue promise[2];
	JSValue message;
	char user[];
} append_message_t;

static void _tf_ssb_appendMessage_finish(append_message_t* async, bool success, JSValue result)
{
	JSValue error = JS_Call(async->context, success ? async->promise[0] : async->promise[1], JS_UNDEFINED, 1, &result);
	tf_util_report_error(async->context, error);
	JS_FreeValue(async->context, error);
	JS_FreeValue(async->context, async->message);
	JS_FreeValue(async->context, async->promise[0]);
	JS_FreeValue(async->context, async->promise[1]);
	tf_free(async);
}

static void _tf_ssb_appendMessageWithIdentity_callback(const char* id, bool verified, bool is_new, void* user_data)
{
	append_message_t* async = user_data;
	JSValue result = JS_UNDEFINED;
	if (verified)
	{
		result = is_new ? JS_TRUE : JS_FALSE;
	}
	_tf_ssb_appendMessage_finish(async, verified, result);
}

static void _tf_ssb_append_message_with_identity_get_key_work(tf_ssb_t* ssb, void* user_data)
{
	append_message_t* work = user_data;
	work->got_private_key = tf_ssb_db_identity_get_private_key(ssb, work->user, work->id, work->private_key, sizeof(work->private_key));
	if (!work->got_private_key && tf_ssb_db_user_has_permission(ssb, NULL, work->user, "administration"))
	{
		work->got_private_key = tf_ssb_db_identity_get_private_key(ssb, ":admin", work->id, work->private_key, sizeof(work->private_key));
	}
	tf_ssb_db_get_latest_message_by_author(ssb, work->id, &work->previous_sequence, work->previous_id, sizeof(work->previous_id));
}

static void _tf_ssb_append_message_with_identity_get_key_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	append_message_t* work = user_data;
	if (work->got_private_key)
	{
		JSValue signed_message = tf_ssb_sign_message(ssb, work->id, work->private_key, work->message, work->previous_id, work->previous_sequence);
		tf_ssb_verify_strip_and_store_message(ssb, signed_message, _tf_ssb_appendMessageWithIdentity_callback, work);
		JS_FreeValue(work->context, signed_message);
	}
	else
	{
		_tf_ssb_appendMessage_finish(work, false, JS_ThrowInternalError(work->context, "Unable to get private key for user %s with identity %s.", work->user, work->id));
	}
}

static void _tf_ssb_append_message_permission_callback(JSContext* context, bool granted, JSValue value, void* user_data)
{
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	append_message_t* work = user_data;
	if (granted)
	{
		tf_ssb_run_work(ssb, _tf_ssb_append_message_with_identity_get_key_work, _tf_ssb_append_message_with_identity_get_key_after_work, work);
	}
	else
	{
		_tf_ssb_appendMessage_finish(work, false, value);
	}
}

static JSValue _tf_ssb_appendMessageWithIdentity(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	JSValue process = data[0];

	char description[1024] = "Publish a new message.";
	const char* type = tf_util_get_property_as_string(context, argv[1], "type");
	if (type)
	{
		if (strcmp(type, "vote") == 0)
		{
			JSValue vote = JS_GetPropertyStr(context, argv[1], "vote");
			const char* expression = tf_util_get_property_as_string(context, vote, "expression");
			snprintf(description, sizeof(description), "React with %s.", expression);
			JS_FreeCString(context, expression);
			JS_FreeValue(context, vote);
		}
		else
		{
			snprintf(description, sizeof(description), "Publish a new %s message.", type);
		}
	}
	else if (JS_IsString(argv[1]))
	{
		tf_string_set(description, sizeof(description), "Publish a new private message.");
	}
	JS_FreeCString(context, type);

	const char* user = _tf_ssb_get_process_credentials_session_name(context, process);
	if (!user)
	{
		return JS_ThrowInternalError(context, "Invalid user.");
	}
	else
	{
		size_t user_length = strlen(user);
		const char* id = JS_ToCString(context, argv[0]);

		append_message_t* work = tf_malloc(sizeof(append_message_t) + user_length + 1);
		*work = (append_message_t) { .context = context, .message = JS_DupValue(context, argv[1]) };
		memcpy(work->user, user, user_length + 1);
		tf_string_set(work->id, sizeof(work->id), id);

		JS_FreeCString(context, user);
		JS_FreeCString(context, id);

		JSValue result = JS_NewPromiseCapability(context, work->promise);
		_tf_ssb_permission_test(context, process, "ssb_append", description, _tf_ssb_append_message_permission_callback, work);
		return result;
	}
}

typedef struct _add_identity_t
{
	uint8_t key[crypto_sign_SECRETKEYBYTES / 2];
	bool added;
	JSValue result;
	JSValue promise[2];
	char user[];
} add_identity_t;

static void _tf_ssb_add_identity_work(tf_ssb_t* ssb, void* user_data)
{
	add_identity_t* work = user_data;
	uint8_t public_key[crypto_sign_PUBLICKEYBYTES];
	unsigned char seed[crypto_sign_SEEDBYTES];
	uint8_t secret_key[crypto_sign_SECRETKEYBYTES] = { 0 };
	memcpy(secret_key, work->key, sizeof(secret_key) / 2);
	if (crypto_sign_ed25519_sk_to_seed(seed, secret_key) == 0 && crypto_sign_seed_keypair(public_key, secret_key, seed) == 0)
	{
		char public_key_b64[512];
		tf_base64_encode(public_key, sizeof(public_key), public_key_b64, sizeof(public_key_b64));
		snprintf(public_key_b64 + strlen(public_key_b64), sizeof(public_key_b64) - strlen(public_key_b64), ".ed25519");

		uint8_t combined[crypto_sign_SECRETKEYBYTES];
		memcpy(combined, work->key, sizeof(work->key));
		memcpy(combined + sizeof(work->key), public_key, sizeof(public_key));
		char combined_b64[512];
		tf_base64_encode(combined, sizeof(combined), combined_b64, sizeof(combined_b64));
		snprintf(combined_b64 + strlen(combined_b64), sizeof(combined_b64) - strlen(combined_b64), ".ed25519");

		work->added = tf_ssb_db_identity_add(ssb, work->user, public_key_b64, combined_b64);
	}
}

static void _tf_ssb_add_identity_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	add_identity_t* work = user_data;
	JSContext* context = tf_ssb_get_context(ssb);
	JSValue result = JS_IsUndefined(work->result) ? (work->added ? JS_TRUE : JS_UNDEFINED) : work->result;
	JSValue error = JS_Call(context, work->promise[work->added ? 0 : 1], JS_UNDEFINED, 1, &result);
	JS_FreeValue(context, result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, work->result);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	tf_free(work);
}

static void _tf_ssb_add_identity_permission_callback(JSContext* context, bool granted, JSValue value, void* user_data)
{
	add_identity_t* work = user_data;
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	if (granted)
	{
		tf_ssb_run_work(ssb, _tf_ssb_add_identity_work, _tf_ssb_add_identity_after_work, work);
	}
	else
	{
		work->result = JS_DupValue(context, value);
		_tf_ssb_add_identity_after_work(ssb, -1, work);
	}
}

static JSValue _tf_ssb_addIdentity(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	JSValue process = data[0];
	JSValue result = JS_UNDEFINED;
	const char* user = _tf_ssb_get_process_credentials_session_name(context, process);
	size_t user_length = user ? strlen(user) : 0;

	JSValue buffer = JS_UNDEFINED;
	size_t length = 0;
	uint8_t* array = tf_util_try_get_array_buffer(context, &length, argv[0]);
	if (!array)
	{
		size_t offset;
		size_t element_size;
		buffer = tf_util_try_get_typed_array_buffer(context, argv[0], &offset, &length, &element_size);
		if (!JS_IsException(buffer))
		{
			array = tf_util_try_get_array_buffer(context, &length, buffer);
		}
	}

	if (array)
	{
		if (length == crypto_sign_SECRETKEYBYTES / 2)
		{
			add_identity_t* work = tf_malloc(sizeof(add_identity_t) + user_length + 1);
			*work = (add_identity_t) { .result = JS_UNDEFINED };
			memcpy(work->key, array, sizeof(work->key));
			if (user)
			{
				memcpy(work->user, user, user_length + 1);
			}
			result = JS_NewPromiseCapability(context, work->promise);
			_tf_ssb_permission_test(context, process, "ssb_id_add", "Add an identity.", _tf_ssb_add_identity_permission_callback, work);
		}
		else
		{
			result = JS_ThrowInternalError(context, "Unexpected private key size: %d vs. %d\n", (int)length, crypto_sign_SECRETKEYBYTES);
		}
	}
	else
	{
		result = JS_ThrowInternalError(context, "Expected array argument.");
	}
	JS_FreeValue(context, buffer);
	JS_FreeCString(context, user);
	return result;
}

typedef struct _delete_identity_t
{
	char id[k_id_base64_len];
	bool deleted;
	JSValue result;
	JSValue promise[2];
	char user[];
} delete_identity_t;

static void _tf_ssb_delete_identity_work(tf_ssb_t* ssb, void* user_data)
{
	delete_identity_t* work = user_data;
	work->deleted = tf_ssb_db_identity_delete(ssb, work->user, work->id);
}

static void _tf_ssb_delete_identity_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	delete_identity_t* work = user_data;
	JSContext* context = tf_ssb_get_context(ssb);
	JSValue result = work->deleted ? JS_TRUE : JS_FALSE;
	JSValue error = JS_Call(context, work->promise[0], JS_UNDEFINED, 1, &result);
	JS_FreeValue(context, result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	tf_free(work);
}

static void _tf_ssb_delete_identity_permission_callback(JSContext* context, bool granted, JSValue value, void* user_data)
{
	delete_identity_t* work = user_data;
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	if (granted)
	{
		tf_ssb_run_work(ssb, _tf_ssb_delete_identity_work, _tf_ssb_delete_identity_after_work, work);
	}
	else
	{
		work->result = JS_DupValue(context, value);
		_tf_ssb_delete_identity_after_work(ssb, -1, work);
	}
}

static JSValue _tf_ssb_deleteIdentity(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	JSValue result = JS_UNDEFINED;
	JSValue process = data[0];
	const char* user = _tf_ssb_get_process_credentials_session_name(context, process);
	const char* id = JS_ToCString(context, argv[0]);
	if (id && user)
	{
		size_t user_length = strlen(user);
		delete_identity_t* work = tf_malloc(sizeof(delete_identity_t) + user_length + 1);
		*work = (delete_identity_t) { .result = JS_UNDEFINED };
		tf_string_set(work->id, sizeof(work->id), *id == '@' ? id + 1 : id);
		memcpy(work->user, user, user_length + 1);
		result = JS_NewPromiseCapability(context, work->promise);
		_tf_ssb_permission_test(context, process, "ssb_id_delete", "Delete an identity.", _tf_ssb_delete_identity_permission_callback, work);
	}
	JS_FreeCString(context, id);
	JS_FreeCString(context, user);
	return result;
}

typedef struct _get_private_key_t
{
	JSContext* context;
	JSValue result;
	JSValue promise[2];
	char id[k_id_base64_len];
	uint8_t private_key[crypto_sign_SECRETKEYBYTES];
	bool got_private_key;
	char user[];
} get_private_key_t;

static void _tf_ssb_get_private_key_work(tf_ssb_t* ssb, void* user_data)
{
	get_private_key_t* work = user_data;
	work->got_private_key = tf_ssb_db_identity_get_private_key(ssb, work->user, work->id, work->private_key, sizeof(work->private_key));
	if (!work->got_private_key && tf_ssb_db_user_has_permission(ssb, NULL, work->user, "administration"))
	{
		work->got_private_key = tf_ssb_db_identity_get_private_key(ssb, ":admin", work->id, work->private_key, sizeof(work->private_key));
	}
}

static void _tf_ssb_get_private_key_after_work(tf_ssb_t* ssb, int status, void* user_data)
{
	get_private_key_t* work = user_data;
	JSValue result = JS_UNDEFINED;
	JSContext* context = work->context;
	if (work->got_private_key && JS_IsUndefined(work->result))
	{
		result = tf_util_new_uint8_array(context, work->private_key, sizeof(work->private_key) / 2);
	}
	JSValue error = JS_Call(context, work->promise[work->got_private_key ? 0 : 1], JS_UNDEFINED, 1, &result);
	JS_FreeValue(context, result);
	tf_util_report_error(context, error);
	JS_FreeValue(context, error);
	JS_FreeValue(context, work->promise[0]);
	JS_FreeValue(context, work->promise[1]);
	JS_FreeValue(context, work->result);
	tf_free(work);
}
static void _tf_ssb_get_private_key_permission_callback(JSContext* context, bool granted, JSValue value, void* user_data)
{
	get_private_key_t* work = user_data;
	tf_task_t* task = tf_task_get(context);
	tf_ssb_t* ssb = tf_task_get_ssb(task);
	if (granted)
	{
		tf_ssb_run_work(ssb, _tf_ssb_get_private_key_work, _tf_ssb_get_private_key_after_work, work);
	}
	else
	{
		work->result = JS_DupValue(context, value);
		_tf_ssb_get_private_key_after_work(ssb, -1, work);
	}
}

static JSValue _tf_ssb_getPrivateKey(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv, int magic, JSValue* data)
{
	JSValue process = data[0];
	const char* user = _tf_ssb_get_process_credentials_session_name(context, process);
	size_t user_length = user ? strlen(user) : 0;
	const char* id = JS_ToCString(context, argv[0]);
	get_private_key_t* work = tf_malloc(sizeof(get_private_key_t) + user_length + 1);
	*work = (get_private_key_t) { .context = context, .result = JS_UNDEFINED };
	if (user)
	{
		memcpy(work->user, user, user_length + 1);
	}
	tf_string_set(work->id, sizeof(work->id), id);
	JSValue result = JS_NewPromiseCapability(context, work->promise);
	_tf_ssb_permission_test(context, process, "ssb_id_export", "Export a private key.", _tf_ssb_get_private_key_permission_callback, work);

	JS_FreeCString(context, user);
	JS_FreeCString(context, id);
	return result;
}

static JSValue _tf_api_register_imports(JSContext* context, JSValueConst this_val, int argc, JSValueConst* argv)
{
	JSValue imports = argv[0];
	JSValue process = argv[1];
	JSValue core = JS_GetPropertyStr(context, imports, "core");
	JS_SetPropertyStr(context, core, "apps", JS_NewCFunctionData(context, _tf_api_core_apps, 1, 0, 1, &process));
	JS_SetPropertyStr(context, core, "register", JS_NewCFunctionData(context, _tf_api_core_register, 2, 0, 1, &process));
	JS_SetPropertyStr(context, core, "unregister", JS_NewCFunctionData(context, _tf_api_core_unregister, 2, 0, 1, &process));

	JS_SetPropertyStr(context, core, "users", JS_NewCFunctionData(context, _tf_api_core_users, 0, 0, 1, &process));

	JS_SetPropertyStr(context, core, "permissionsForUser", JS_NewCFunctionData(context, _tf_api_core_permissionsForUser, 1, 0, 1, &process));
	JS_SetPropertyStr(context, core, "permissionsGranted", JS_NewCFunctionData(context, _tf_api_core_permissionsGranted, 0, 0, 1, &process));
	JS_SetPropertyStr(context, core, "allPermissionsGranted", JS_NewCFunctionData(context, _tf_api_core_allPermissionsGranted, 0, 0, 1, &process));

	JSValue app = JS_NewObject(context);
	JS_SetPropertyStr(context, app, "owner", JS_GetPropertyStr(context, process, "packageOwner"));
	JS_SetPropertyStr(context, app, "name", JS_GetPropertyStr(context, process, "packageName"));
	JS_SetPropertyStr(context, core, "app", app);

	JS_SetPropertyStr(context, core, "url", JS_GetPropertyStr(context, process, "url"));

	JSValue ssb = JS_GetPropertyStr(context, imports, "ssb");
	JS_SetPropertyStr(context, ssb, "getAllIdentities", JS_NewCFunction(context, _tf_ssb_getAllIdentities, "getAllIdentities", 0));
	JS_SetPropertyStr(context, ssb, "getActiveIdentity", JS_NewCFunctionData(context, _tf_ssb_getActiveIdentity, 0, 0, 1, &process));
	JS_SetPropertyStr(context, ssb, "getIdentities", JS_NewCFunctionData(context, _tf_ssb_getIdentities, 0, 0, 1, &process));
	JS_SetPropertyStr(context, ssb, "getOwnerIdentities", JS_NewCFunctionData(context, _tf_ssb_getOwnerIdentities, 0, 0, 1, &process));
	JS_SetPropertyStr(context, ssb, "privateMessageEncrypt", JS_NewCFunctionData(context, _tf_ssb_private_message_encrypt, 3, 0, 1, &process));
	JS_SetPropertyStr(context, ssb, "privateMessageDecrypt", JS_NewCFunctionData(context, _tf_ssb_private_message_decrypt, 2, 0, 1, &process));
	JS_SetPropertyStr(context, ssb, "appendMessageWithIdentity", JS_NewCFunctionData(context, _tf_ssb_appendMessageWithIdentity, 2, 0, 1, &process));
	JS_SetPropertyStr(context, ssb, "addIdentity", JS_NewCFunctionData(context, _tf_ssb_addIdentity, 1, 0, 1, &process));
	JS_SetPropertyStr(context, ssb, "deleteIdentity", JS_NewCFunctionData(context, _tf_ssb_deleteIdentity, 1, 0, 1, &process));
	JS_SetPropertyStr(context, ssb, "getPrivateKey", JS_NewCFunctionData(context, _tf_ssb_getPrivateKey, 1, 0, 1, &process));
	JS_FreeValue(context, ssb);

	JSValue credentials = JS_GetPropertyStr(context, process, "credentials");
	JSValue permissions = JS_IsObject(credentials) ? JS_GetPropertyStr(context, credentials, "permissions") : JS_UNDEFINED;
	JSValue administration = JS_IsObject(permissions) ? JS_GetPropertyStr(context, permissions, "administration") : JS_UNDEFINED;
	if (JS_ToBool(context, administration) > 0)
	{
		JS_SetPropertyStr(context, core, "globalSettingsDescriptions", JS_NewCFunction(context, _tf_ssb_globalSettingsDescriptions, "globalSettingsDescriptions", 0));
		JS_SetPropertyStr(context, core, "globalSettingsGet", JS_NewCFunction(context, _tf_ssb_globalSettingsGet, "globalSettingsGet", 1));
		JS_SetPropertyStr(context, core, "globalSettingsSet", JS_NewCFunctionData(context, _tf_ssb_globalSettingsSet, 2, 0, 1, &process));

		JS_SetPropertyStr(context, core, "deleteUser", JS_NewCFunctionData(context, _tf_ssb_delete_user, 0, 0, 1, &process));

		JS_SetPropertyStr(context, ssb, "addBlock", JS_NewCFunctionData(context, _tf_ssb_add_block, 1, 0, 1, &process));
		JS_SetPropertyStr(context, ssb, "removeBlock", JS_NewCFunctionData(context, _tf_ssb_remove_block, 1, 0, 1, &process));
		JS_SetPropertyStr(context, ssb, "getBlocks", JS_NewCFunctionData(context, _tf_ssb_get_blocks, 0, 0, 1, &process));

		JS_SetPropertyStr(context, ssb, "swapWithServerIdentity", JS_NewCFunctionData(context, _tf_ssb_swap_with_server_identity, 1, 0, 1, &process));
	}
	JS_FreeValue(context, administration);
	JS_FreeValue(context, permissions);
	JS_FreeValue(context, credentials);

	JS_FreeValue(context, core);
	return JS_UNDEFINED;
}

void tf_api_register(JSContext* context)
{
	JSValue global = JS_GetGlobalObject(context);
	JSValue ssb = JS_GetPropertyStr(context, global, "ssb");
	JS_SetPropertyStr(context, ssb, "registerImports", JS_NewCFunction(context, _tf_api_register_imports, "registerImports", 2));
	JS_FreeValue(context, ssb);
	JS_FreeValue(context, global);
}
