@page app_development_guide App Development Guide

A Tilde Friends application starts with code that runs on a Tilde Friends server, possibly far away from where you wrote it, in a little JavaScript environment, in its own restricted process, with the only access to the outside world being the ability to send messages to the server. This document gives some recipes showing how that can be used to build a functional user-facing application in light of the unique constraints present.

## Example 1: Hello, world!

Of course we must start with a classic.

### app.js

```js
app.setDocument('<h1 style="color: #fff">Hello, world!</h1>');
```

### Output

<h1 style="color: #fff">Hello, world!</h1>

### Explanation

At a glance, this might seem mundane, but for it to work:

- the server starts a real process for your app and loads your code into it
- your code runs
- `app.setDocument()` sends a message back to the server
- the server interprets the message and redirects it to the browser
- `core/client.js` in the browser receives the message and puts your HTML into an iframe
- your HTML is presented by the browser in an iframe sandbox

But you don't have to think about all that. Call a function, and you see the result.

## Example 2: Hit Counter

Let's take advantage of code running on the server and create a little hit counter using a key value store shared between all visitors.

### app.js

```js
async function main() {
	let db = await shared_database('visitors');
	let count = parseInt((await db.get('visitors')) ?? '0') + 1;
	await db.set('visitors', count.toString());
	await app.setDocument(`
		<h1 style="color: #fff">Welcome, visitor #${count}!</h1>
	`);
}

main();
```

### Output

<h1 style="color: #fff">Welcome, visitor #1!</h1>

### Explanation

Just as pure browser apps have access to `localStorage`, Tilde Friends apps have access to key-value storage on the server.

The interface is a bit clunky and will likely change someday, but this example gets a database object, from which you can get and set string values by key. There are various on `shared_database` that let you store data that is private to the user or shared by different criteria.

Also, even though any browser-side code is sandboxed, it is allowed to access browser local storage by going through Tilde Friends API, because sometimes that is useful.

## Example 3: Files

Suppose you don't want to create your entire app in a single server-side file as we've done with the previous examples. There are some tools to allow you to begin to organize.

### app.js

```js
async function main() {
	let html = utf8Decode(await getFile('index.html'));
	app.setDocument(html);
}

main();
```

### index.html

```html
<html>
	<head>
		<script type="module" src="script.js"></script>
	</head>
	<body style="color: #fff">
		<h1>File Test</h1>
	</body>
</html>
```

### script.js

```js
window.addEventListener('load', function() {
	document.body.appendChild(document.createTextNode('Hello, world');
});
```

### Output

<h1>File Test</h1><p>Hello, world!</p>

### Explanation

On the server, `utf8Decode(await getFile(fileName))` lets you load a file from your app. In the browser, your app files are made available by HTTP, so you can `<script src="my_script.js"></script>` and such to access them.

## Example 4: Remote Procedure Call

While making calls between the client and the server, it is possible to pass functions across that boundary. `tfrpc.js` is a tiny script which builds on that feature to try to hide some of the complexities.

### app.js

```js
import * as tf from '/tfrpc.js';

function sum() {
	let s = 0;
	for (let x of arguments) {
		s += x;
	}
	return s;
}
tf.register(sum);

async function main() {
	app.setDocument(utf8Decode(await getFile('index.html')));
}
main();
```

### index.html

```html
<html>
	<body>
		<h1 id="result">Calculating...</h1>
	</body>
	<script type="module" src="script.js"></script>
</html>
```

### script.js

```js
import * as tf from '/static/tfrpc.js';

window.addEventListener('load', async function () {
	document.getElementById('result').innerText = await tf.rpc.sum(1, 2, 3);
});
```

### Output

<h1>6</h1>

### Explanation

Here the browser makes an asynchronous call to the server to do some basic math and update its DOM with the result.

With your favorite Vue/Lit/React/... library on the client-side and your favorite Tilde Friends API calls registered with tfrpc, it becomes pretty easy to start extracting interesting information from, say, SQL queries over Secure Scuttlebutt data, and generating complicated, dynamic user interface. These are the building blocks I used to make the current Tilde Friends SSB client interface.

## Conclusion

Tilde Friends is currently a pile of all the parts that I thought I needed to build interesting web applications, tied together by code that tries to walk the fine line between being secure enough to let us safely run code on the same device and being usable enough that you can open a tab in your browser and start building just by typing code.

I don't claim it thoroughly accomplishes either yet, but I believe it is at a stage where it is showing how promising this approach can be, and I am excited for you to take it for a spin and share.
