import {LitElement, html} from './lit-all.min.js';
import * as tfrpc from '/static/tfrpc.js';

class TfCollectionElement extends LitElement {
	static get properties() {
		return {
			whoami: {type: String},
			category: {type: String},
			collection: {type: Object},
			selected_id: {type: String},
			is_creating: {type: Boolean},
			is_renaming: {type: Boolean},
		};
	}

	on_create(event) {
		let name = this.shadowRoot.getElementById('create_name').value;
		this.dispatchEvent(
			new CustomEvent('create', {
				bubbles: true,
				detail: {
					name: name,
				},
			})
		);
		this.is_creating = false;
	}

	on_rename(event) {
		let id = this.shadowRoot.getElementById('select').value;
		let name = this.shadowRoot.getElementById('rename_name').value;
		this.dispatchEvent(
			new CustomEvent('rename', {
				bubbles: true,
				detail: {
					id: id,
					value: this.collection[id],
					name: name,
				},
			})
		);
		this.is_renaming = false;
	}

	on_tombstone(event) {
		let id = this.shadowRoot.getElementById('select').value;
		if (
			confirm(`Are you sure you want to delete '${this.collection[id].name}'?`)
		) {
			this.dispatchEvent(
				new CustomEvent('tombstone', {
					bubbles: true,
					detail: {
						id: id,
						value: this.collection[id],
					},
				})
			);
		}
	}

	on_selected(event) {
		let id = event.srcElement.value;
		this.selected_id = id != '' ? id : undefined;
		this.dispatchEvent(
			new CustomEvent('change', {
				bubbles: true,
				detail: {
					id: id,
					value: this.collection[id],
				},
			})
		);
	}

	render() {
		let self = this;
		return html`
			<link rel="stylesheet" href="tildefriends.css"/>
			<span class="inline-flex-row">
				<select @change=${this.on_selected} id="select" value=${this.selected_id}>
					<option value="" ?selected=${this.selected_id === ''} disabled hidden>(select ${this.category})</option>
					${Object.values(this.collection ?? {})
						.sort((x, y) => x.name.localeCompare(y.name))
						.map(
							(x) =>
								html`<option
									value=${x.id}
									?selected=${this.selected_id === x.id}
								>
									${x.name}
								</option>`
						)}
				</select>
				<span ?hidden=${!this.is_renaming || !this.whoami}>
					<span class="inline-flex-row" style="margin-left: 8px; margin-right: 8px">
						<label for="rename_name">🏷Rename to:</label>
						<input type="text" id="rename_name"></input>
						<button @click=${this.on_rename}>Rename ${this.type}</button>
						<button @click=${() => (self.is_renaming = false)}>x</button>
					</span>
				</span>
				<button class="yellow" @click=${() => (self.is_renaming = true)} ?disabled=${this.is_renaming || !this.selected_id} ?hidden=${!this.whoami}>🏷</button>
				<button class="red" @click=${self.on_tombstone} ?disabled=${!this.selected_id} ?hidden=${!this.whoami}>🪦</button>
				<span ?hidden=${!this.is_creating || !this.whoami}>
					<label for="create_name">New ${this.type} name:</label>
					<input type="text" id="create_name"></input>
					<button @click=${this.on_create}>Create ${this.type}</button>
					<button @click=${() => (self.is_creating = false)}>x</button>
				</span>
				<button class="green" @click=${() => (self.is_creating = true)} ?hidden=${this.is_creating || !this.whoami}>+</button>
			</span>
		`;
	}
}

customElements.define('tf-collection', TfCollectionElement);
