import {LitElement, html} from './lit-all.min.js';
import * as tfrpc from '/static/tfrpc.js';
import {styles, generate_theme} from './tf-styles.js';

class TfTabConnectionsElement extends LitElement {
	static get properties() {
		return {
			broadcasts: {type: Array},
			identities: {type: Array},
			my_identities: {type: Array},
			connections: {type: Array},
			stored_connections: {type: Array},
			users: {type: Object},
			server_identity: {type: String},
			connect_attempt: {type: Object},
			connect_message: {type: String},
			connect_success: {type: Boolean},
			peer_exchange: {type: Boolean},
		};
	}

	static styles = styles;

	static k_broadcast_emojis = {
		discovery: '🏓',
		room: '🚪',
		peer_exchange: '🕸',
	};

	constructor() {
		super();
		let self = this;
		this.broadcasts = [];
		this.identities = [];
		this.my_identities = [];
		this.connections = [];
		this.stored_connections = [];
		this.users = {};
		tfrpc.rpc.getIdentities().then(function (identities) {
			self.my_identities = identities || [];
		});
		tfrpc.rpc.getAllIdentities().then(function (identities) {
			self.identities = identities || [];
		});
		tfrpc.rpc.getStoredConnections().then(function (connections) {
			self.stored_connections = connections || [];
		});
		tfrpc.rpc.getServerIdentity().then(function (identity) {
			self.server_identity = identity;
		});
		this.check_peer_exchange();
	}

	async check_peer_exchange() {
		if (await tfrpc.rpc.isAdministrator()) {
			this.peer_exchange = await tfrpc.rpc.globalSettingsGet('peer_exchange');
		} else {
			this.peer_exchange = undefined;
		}
	}

	async enable_peer_exchange() {
		await tfrpc.rpc.globalSettingsSet('peer_exchange', true);
		await this.check_peer_exchange();
	}

	render_connection_summary(connection) {
		if (connection.address && connection.port) {
			return html`<div>
				<small>${connection.address}:${connection.port}</small>
			</div>`;
		} else if (connection.tunnel) {
			return html`<div>room peer</div>`;
		} else {
			return JSON.stringify(connection);
		}
	}

	render_room_peers(connection) {
		let self = this;
		let peers = this.broadcasts.filter((x) => x.tunnel?.id == connection);
		if (peers.length) {
			let connections = this.connections.map((x) => x.id);
			return html`${peers
				.filter((x) => connections.indexOf(x.pubkey) == -1)
				.map((x) => html`${self.render_room_peer(x)}`)}`;
		}
	}

	async _tunnel(portal, target) {
		return tfrpc.rpc.createTunnel(portal, target);
	}

	render_room_peer(connection) {
		let self = this;
		return html`
			<li>
				<button
					class="w3-button w3-theme-d1"
					@click=${() => self._tunnel(connection.tunnel.id, connection.pubkey)}
				>
					Connect
				</button>
				<tf-user id=${connection.pubkey} .users=${this.users}></tf-user> 📡
			</li>
		`;
	}

	render_message(connection) {
		return html`<div
			?hidden=${this.connect_message === undefined ||
			this.connect_attempt != connection}
			style="cursor: pointer"
			class=${'w3-panel ' + (this.connect_success ? 'w3-green' : 'w3-red')}
			@click=${() => (this.connect_attempt = undefined)}
		>
			<p>${this.connect_message}</p>
		</div>`;
	}

	render_progress(name, value, max) {
		if (max && value != max) {
			return html`
				<div class="w3-theme-d1 w3-small">
					<div
						class="w3-container w3-theme-l1"
						style="width: ${Math.floor(
							(100.0 * value) / max
						)}%; text-wrap: nowrap"
					>
						${name} ${value} / ${max} (${Math.round((100.0 * value) / max)}%)
					</div>
				</div>
			`;
		}
	}

	render_broadcast(connection) {
		let self = this;
		return html`
			<li>
				<div class="w3-bar" style="overflow: hidden; overflow-wrap: nowrap">
					<button
						class="w3-bar-item w3-button w3-theme-d1"
						@click=${() => self.connect(connection)}
					>
						Connect
					</button>
					<div class="w3-bar-item">
						${TfTabConnectionsElement.k_broadcast_emojis[connection.origin]}
						<tf-user id=${connection.pubkey} .users=${this.users}></tf-user>
						${this.render_connection_summary(connection)}
					</div>
				</div>
				${this.render_message(connection)}
			</li>
		`;
	}

	async forget_stored_connection(connection) {
		await tfrpc.rpc.forgetStoredConnection(connection);
		this.stored_connections = (await tfrpc.rpc.getStoredConnections()) || [];
	}

	render_connection(connection) {
		let requests = Object.values(
			connection.requests.reduce(function (accumulator, value) {
				let key = `${value.name}:${Math.sign(value.request_number)}`;
				if (!accumulator[key]) {
					accumulator[key] = Object.assign({count: 0}, value);
				}
				accumulator[key].count++;
				return accumulator;
			}, {})
		);
		return html`
			${connection.connected
				? html`
						<button
							class="w3-button w3-theme-d1"
							@click=${() => tfrpc.rpc.closeConnection(connection.id)}
						>
							Close
						</button>
					`
				: undefined}
			${connection.flags.one_shot ? '🔃' : undefined}
			<tf-user id=${connection.id} .users=${this.users}></tf-user>
			${this.render_progress(
				'recv',
				connection.progress.in.total - connection.progress.in.current,
				connection.progress.in.total
			)}
			${this.render_progress(
				'send',
				connection.progress.out.total - connection.progress.out.current,
				connection.progress.out.total
			)}
			${connection.tunnel !== undefined
				? '🚇'
				: html`(${connection.host}:${connection.port})`}
			<div>
				${requests.map(
					(x) => html`
						<span
							class=${'w3-tag w3-small ' +
							(x.active ? 'w3-theme-l3' : 'w3-theme-d3')}
							>${x.request_number > 0 ? '🟩' : '🟥'} ${x.name}
							<span
								class="w3-badge w3-white"
								style=${x.count > 1 ? undefined : 'display: none'}
								>${x.count}</span
							></span
						>
					`
				)}
			</div>
			<ul>
				${this.connections
					.filter((x) => x.tunnel === this.connections.indexOf(connection))
					.map((x) => html`<li>${this.render_connection(x)}</li>`)}
				${this.render_room_peers(connection.id)}
			</ul>
			<div ?hidden=${!connection.destroy_reason} class="w3-panel w3-red">
				<p>${connection.destroy_reason}</p>
			</div>
		`;
	}

	connect(address) {
		let self = this;
		self.connect_attempt = address;
		self.connect_message = undefined;
		self.connect_success = false;
		tfrpc.rpc
			.connect(address)
			.then(function () {
				if (self.connect_attempt == address) {
					self.connect_message = 'Connected.';
					self.connect_success = true;
				}
			})
			.catch(function (error) {
				if (self.connect_attempt == address) {
					self.connect_message = 'Error: ' + error;
					self.connect_success = false;
				}
			});
	}

	toggle_accordian(id) {
		let element = this.renderRoot.getElementById(id);
		element.classList.toggle('w3-hide');
	}

	valid_connections() {
		return this.connections.filter((x) => x.tunnel === undefined);
	}

	valid_broadcasts() {
		return this.broadcasts
			.filter((x) => x.address)
			.filter((x) => this.connections.map((c) => c.id).indexOf(x.pubkey) == -1);
	}

	render() {
		let self = this;
		return html`
			<style>
				${generate_theme()}
			</style>
			<div class="w3-container" style="box-sizing: border-box">
				<div
					class=${'w3-panel w3-padding w3-theme-l3' +
					(this.peer_exchange !== false ? ' w3-hide' : '')}
				>
					<p>
						Looking for connections? Enabling this option will include publicly
						advertised rooms and pubs among the list of discovered connections
						to help you replicate.
					</p>
					<button
						class="w3-button w3-theme-d1"
						@click=${this.enable_peer_exchange}
					>
						🔍🌐 Use publicly advertised peers
					</button>
				</div>
				<h2>New Connection</h2>
				<textarea class="w3-input w3-theme-d1" id="code"></textarea>
				${this.render_message(this.renderRoot.getElementById('code')?.value)}
				<button
					class="w3-button w3-theme-d1"
					@click=${() =>
						self.connect(self.renderRoot.getElementById('code')?.value)}
				>
					Connect
				</button>
				<h2
					class="w3-button w3-block w3-theme-d1"
					@click=${() => self.toggle_accordian('connections')}
				>
					Connections (${this.valid_connections().length})
				</h2>
				<ul class="w3-ul w3-border" id="connections">
					${this.valid_connections().map(
						(x) => html` <li class="w3-bar">${this.render_connection(x)}</li> `
					)}
				</ul>
				<h2
					class="w3-button w3-block w3-theme-d1"
					@click=${() => self.toggle_accordian('broadcasts')}
				>
					Discovery (${this.valid_broadcasts().length})
				</h2>
				<ul class="w3-ul w3-border w3-hide" id="broadcasts">
					${this.valid_broadcasts().map((x) => self.render_broadcast(x))}
				</ul>
				<h2
					class="w3-button w3-block w3-theme-d1"
					@click=${() => self.toggle_accordian('stored_connections')}
				>
					Stored Connections (${this.stored_connections.length})
				</h2>
				<ul class="w3-ul w3-border w3-hide" id="stored_connections">
					${this.stored_connections.map(
						(x) => html`
							<li>
								<div class="w3-bar">
									<button
										class="w3-bar-item w3-button w3-theme-d1"
										@click=${() => self.forget_stored_connection(x)}
									>
										Forget
									</button>
									<button
										class="w3-bar-item w3-button w3-theme-d1"
										@click=${() => this.connect(x)}
									>
										Connect
									</button>
									<div class="w3-bar-item">
										<tf-user id=${x.pubkey} .users=${self.users}></tf-user>
										<div><small>${x.address}:${x.port}</small></div>
										<div>
											<small
												>Last connection:
												${new Date(x.last_success * 1000)}</small
											>
										</div>
									</div>
								</div>
								${this.render_message(x)}
							</li>
						`
					)}
				</ul>
				<h2
					class="w3-button w3-block w3-theme-d1"
					@click=${() => self.toggle_accordian('local_accounts')}
				>
					Local Accounts (${this.identities.length})
				</h2>
				<div class="w3-container w3-hide" id="local_accounts">
					${this.identities.map(
						(x) =>
							html`<div
								class="w3-tag w3-round w3-theme-l3"
								style="padding: 4px; margin: 2px; max-width: 100%; text-wrap: nowrap; overflow: hidden"
							>
								${x == this.server_identity
									? html`<div class="w3-tag w3-medium w3-round w3-theme-l1">
											🖥 local server
										</div>`
									: undefined}
								${this.my_identities.indexOf(x) != -1
									? html`<div class="w3-tag w3-medium w3-round w3-theme-d1">
											😎 you
										</div>`
									: undefined}
								<tf-user id=${x} .users=${this.users}></tf-user>
							</div>`
					)}
				</div>
			</div>
		`;
	}
}

customElements.define('tf-tab-connections', TfTabConnectionsElement);
